#!/usr/bin/perl
use strict;
use warnings;

die "Usage: boilerplater SOURCE_DIR DEST_DIR" unless @ARGV == 2;
my ( $base_dir, $dest_dir ) = @ARGV;
die "'$base_dir' isn't a directory" unless -d $base_dir;
if ( !-d $dest_dir ) {
    mkdir $dest_dir or die "Can't mkdir '$dest_dir': $!";
}

my $hierarchy = Hierarchy->new(
    base_dir => $base_dir,
    dest_dir => $dest_dir,
);
$hierarchy->build;
$hierarchy->write_all_modified;
exit;

#----------------------------------------------------------------------

package Util;
use Carp;

# Return a string auto-gen warning.
sub autogen_header {
    return <<END_STUFF;
/***********************************************

 !!!! DO NOT EDIT THIS FILE !!!!

 All content has been auto-generated by 
 the boilerplater.pl utility.

 See boilerplater's documentation for details.

 ***********************************************/

END_STUFF
}

sub copyright_footer {
    return <<END_STUFF;
/* Copyright 2007 Marvin Humphrey
 *
 * This program is free software; you can redistribute it and/or modify
 * under the same terms as Perl itself.
 */
END_STUFF
}

# Open a file, read it in, return its contents.
sub slurp_file {
    my $path = shift;
    open( my $fh, '<', $path ) or confess("Can't open '$path': $!");
    local $/;
    return <$fh>;
}

# Given two elements, which may be either scalars or arrays, verify that
# everything in the second group exists and was created later than anything in
# the first group.
sub current {
    my ( $orig, $dest ) = @_;
    my $bubble_time = time;
    $orig = [$orig] unless ref($orig) eq 'ARRAY';
    $dest = [$dest] unless ref($dest) eq 'ARRAY';

    # if a destination file doesn't exist, we're not current
    for (@$dest) {
        return 0 unless -e $_;
    }

    # find the oldest file from the destination group
    for (@$dest) {
        my $candidate = ( stat($_) )[9];
        $bubble_time = $candidate if $candidate < $bubble_time;
    }

    # if any source file is newer than the oldest dest, we're not current
    for (@$orig) {
        my $candidate = ( stat($_) )[9];
        return 0 if $candidate > $bubble_time;
    }

    # current!
    return 1;
}

# Quick 'n' dirty stripping of comments.  Will massacre stuff like comments
# embedded in string literals, so watch out.
sub strip_c_comments {
    my $c_code = shift;
    $c_code =~ s#/\*.*?\*/##gsm;
    return $c_code;
}

package Hierarchy;
use Carp;
use File::Find qw( find );
use File::Spec::Functions qw( catfile splitpath );
use File::Path qw( mkpath );

sub new {
    my $ignore = shift;
    my $self = bless {
        base_dir => undef,    # the directory we begin reading files from
        dest_dir => undef,    # the directory we write files to
        tree     => {},       # the hierarchy, with Obj at its base
        @_,
        },
        __PACKAGE__;
    return $self;
}

# Slurp all .h files.  For those which define a KINO_CLASS, create a Class
# object.  Discard the rest.
sub build {
    my $self = shift;
    my ( $base_dir, $dest_dir ) = @{$self}{qw( base_dir dest_dir )};

    # collect filenames
    my @all_h_paths;
    find(
        {   wanted => sub {
                if ( $File::Find::name =~ /\.h$/ ) {
                    push @all_h_paths, $File::Find::name;
                }
            },
            no_chdir => 1,
        },
        $self->{base_dir},
    );

    # process any file that has at least one class declaration
    my %classes;
    for my $h_path (@all_h_paths) {
        my ($r_path) = $h_path =~ /(.*)\.h/;
        $r_path .= '.r';
        $r_path =~ s/^$base_dir\W*//
            or die "'$h_path' doesn't start with '$base_dir'";
        $r_path = catfile( $dest_dir, $r_path );
        my ( undef, $dir, undef ) = splitpath($r_path);
        mkpath $dir unless -d $dir;

        my $content = Util::slurp_file($h_path);
        while (
            $content =~ s/.*?^(KINO_(FINAL_)?CLASS.*?KINO_END_CLASS\s*)//ms )
        {
            my $final       = defined $2;
            my $class_class = $final ? 'Class::Final' : 'Class';
            my $class       = $class_class->new(
                content => $1,
                h_path  => $h_path,
                r_path  => $r_path,
            );
            $classes{ $class->get_class_name } = $class;
        }
    }

    # wrangle the classes into a hierarchy and figure out inheritance
    while ( my ( $nickname, $class ) = each %classes ) {
        my $parent_name = $class->get_parent_class_name;
        next if $parent_name eq '';    # skip Obj, which has no parent
        $classes{$parent_name}->add_child($class);
    }
    $self->{tree} = $classes{"KinoSearch::Util::Obj"};
    $self->{tree}->bequeath;
}

sub write_all_modified {
    my $self = shift;

    #rewrite all if this file has changed.
    my $modified = !Util::current( $0, $self->{tree}->get_r_path );

    # seed the recursive write
    $self->_write_if_modified( $self->{tree}, $modified );
}

# recursive helper function
sub _write_if_modified {
    my ( $self, $class, $modified ) = @_;

    # if any parent is modified, rewrite all the kids
    $modified = $class->write_if_modified($modified);

    # proceed to the next generation
    for my $kid ( $class->get_children ) {
        if ( $class->is_final ) {
            confess(  "Attempt to inherit from final class "
                    . $class->get_class_name . " by "
                    . $kid->get_class_name );
        }
        $self->_write_if_modified( $kid, $modified );
    }
}

package Class;
use Carp;
use Config;

# Regex for matching a function (or "method") declaration.
my $func_re;

BEGIN {
    $func_re = qr{
            \s*
            ( [^(;]+  ? )       # return type $1
            \s*
            kino_[a-zA-Z]+_     # prefix and class_nick
            ([a-z_][a-z_0-9]+)  # micro name $2
            \s*\(               # opening paren
            ( [^;]*? )          # arg list $3
            \);                 # closing paren and terminating semicolon
        }xsm;
}

sub new {
    my $class_class = shift;
    my $self = bless {
        h_path            => undef,
        r_path            => undef,
        o_path            => undef,
        content           => undef,
        class_name        => undef,
        class_nick        => undef,
        struct_name       => undef,
        parent_class_name => undef,
        methods           => [],
        overridden        => {},
        functions         => [],
        member_vars       => [],
        children          => [],
        parent            => undef,
        @_
    }, $class_class;

    # derive some filepaths
    my ($path_minus_ext) = $self->{h_path} =~ /(.*)\.h/;
    $self->{o_path} = "$path_minus_ext$Config{_o}";

    # parse the source and extract class elements
    $self->{content} = Util::strip_c_comments( $self->{content} );
    $self->_extract_class_names;
    $self->_extract_functions;
    $self->_extract_methods;
    $self->_extract_member_vars;

    # verify that all text within class definition was consumed
    $self->{content} =~ s/KINO_END_CLASS\s*// or confess "no match";
    $self->_verify_complete_parse( $self->{content} );

    return $self;
}

# Accessors.
sub get_h_path            { shift->{h_path} }
sub get_o_path            { shift->{o_path} }
sub get_r_path            { shift->{r_path} }
sub get_class_name        { shift->{class_name} }
sub get_parent_class_name { shift->{parent_class_name} }
sub get_methods           { @{ shift->{methods} } }
sub get_member_vars       { @{ shift->{member_vars} } }
sub get_children          { @{ shift->{children} } }
sub get_parent            { shift->{parent} }

# Inheriting is allowed.
sub is_final {0}

# Add a child to this class.
sub add_child {
    my ( $self, $child ) = @_;
    push @{ $self->{children} }, $child;
}

# Bequeath all inherited methods and members to children.
sub bequeath {
    my $self = shift;

    for my $child ( @{ $self->{children} } ) {
        # This is a circular reference and thus a memory leak, but we don't
        # care, because we have to have everything in memory at once anyway.
        $child->{parent} = $self;

        # pass down member vars
        unshift @{ $child->{member_vars} }, @{ $self->{member_vars} };

        # pass down methods, with some being overridden
        my @common_methods;
        for my $method ( @{ $self->{methods} } ) {
            if ( exists $child->{meth_by_name}{ $method->get_micro_name } ) {
                my $child_method
                    = $child->{meth_by_name}{ $method->get_micro_name };
                $child_method->override($method);
                push @common_methods, $child_method;
            }
            else {
                push @common_methods, $method;
            }
        }
        unshift @{ $child->{methods} }, @common_methods;

        my @new_method_set;
        my %seen;
        for my $meth ( @common_methods, @{ $child->{methods} } ) {
            next if $seen{ $meth->get_micro_name };
            $seen{ $meth->get_micro_name } = 1;
            $meth = $meth->finalize if $child->is_final;
            push @new_method_set, $meth;
        }
        $child->{methods} = \@new_method_set;

        # pass it all down to the next generation
        $child->bequeath;
    }
}

# Parse KINO_CLASS section.
sub _extract_class_names {
    my $self    = shift;
    my $quot_re = qr/\s*"(.*?)"\s*/;
    $self->{content}
        =~ s/^KINO_(?:FINAL_)?CLASS\($quot_re,$quot_re,$quot_re\)\s*;//m
        or confess "Couldn't match KINO_CLASS definition";
    @{$self}{qw( class_name class_nick parent_class_name )} = ( $1, $2, $3 );
    ( $self->{struct_name} ) = $self->{class_name} =~ /(\w+)$/;
}

# Make sure that the parser consumed all non-whitespace characters.
sub _verify_complete_parse {
    my ( $self, $leftover ) = @_;
    if ( $leftover =~ /\S/ ) {
        confess "non-parseable content in $self->{h_path}:\n$leftover";
    }
}

# Parse KINO_FUNCTION directives.
sub _extract_functions {
    my $self = shift;

    # one Method object for each function declaration
    while ( $self->{content} =~ s/^KINO_FUNCTION\s*\(\s*(.*?)\);//sm ) {
        my $func_declaration = $1 . ';';

        # parse function declaration
        $func_declaration =~ $func_re
            or confess("Illegal function declaration: $func_declaration");
        my $function = Function->new(
            return_type => $1,
            micro_name  => $2,
            class_name  => $self->{class_name},
            class_nick  => $self->{class_nick},
            struct_name => $self->{struct_name},
            arg_list    => $3,
        );
        push @{ $self->{functions} }, $function;
    }
}

# Parse KINO_METHOD directives.
sub _extract_methods {
    my $self = shift;

    # one Method object for each function declaration
    while ( $self->{content}
        =~ s/^KINO_(FINAL_)?METHOD\s*\(\s*"(\w+)",(.*?)\);//ms )
    {
        my $final = $self->is_final || defined $1;
        my $meth_class       = $final ? "Method::Final" : "Method";
        my $full_method_name = $2;
        my $func_declaration = $3 . ';';

        # parse function declaration
        $func_declaration =~ $func_re
            or confess("Illegal function declaration: $func_declaration");
        my $return_type = $1;
        my $micro_name  = $2;
        my $arg_list    = $3;

        # get small macro name
        $full_method_name =~ /Kino_[A-Z][A-Za-z]*_(\w+)/
            or confess("Illegal method name: $full_method_name");
        my $macro_name = $1;

        my $method = $meth_class->new(
            return_type => $return_type,
            micro_name  => $micro_name,
            macro_name  => $macro_name,
            class_name  => $self->{class_name},
            class_nick  => $self->{class_nick},
            struct_name => $self->{struct_name},
            arg_list    => $arg_list,
        );
        $self->{meth_by_name}{$micro_name} = $method;
        push @{ $self->{methods} }, $method;
    }
}

# Parse struct definition.
sub _extract_member_vars {
    my $self        = shift;
    my $member_vars = $self->{member_vars};

    $self->{content} =~ s/(^\s*struct\s+kino_\w+\s+{.*?}\s*;)//ms
        or confess("Couldn't extract struct definition in $self->{h_path}");
    my $obj_section = $1;

    # remove the struct definition wrapper
    $obj_section =~ s/^\s*struct\s+kino_\w+\s+{\s+//
        or confess("Unrecognized text in object section: $obj_section");
    $obj_section =~ s/};\s*$//
        or confess("Unrecognized text in object section: $obj_section");

    # blow past the VTABLE and the refcount
    $obj_section =~ s/^\s*KINO_\w+\s+\*_;//
        or confess("Invalid object definition: $obj_section");
    $obj_section =~ s/^\s*kino_u32_t\s+refcount;//
        or confess("Invalid object definition: $obj_section");

    # ignore the macro indicating inherited member vars, if present
    $obj_section =~ s/^\s*KINO_\w+_MEMBER_VARS\s*//;

    # create one MemberVar object per var
    while ( $obj_section =~ s/\s*([^;]+?)(\b\w+);// ) {
        my ( $type, $name ) = ( $1, $2 );
        $type =~ s/\s+/ /g;    # collapse whitespace on the type
        push @$member_vars, MemberVar->new( type => $type, name => $name );
    }
    $self->_verify_complete_parse($obj_section);
}

# Print representation file if it's not up to date with either its own header
# or any ancestor.
sub write_if_modified {
    my ( $self, $modified ) = @_;

    # propagate modification status
    if ( !Util::current( $self->{h_path}, $self->{r_path} ) ) {
        $modified = 1;
    }

    # print boilerplate file if needed
    if ($modified) {
        my $boilerplate = $self->_gen_boilerplate;
        print "writing $self->{r_path}\n";
        open( my $fh, '>', $self->{r_path} )
            or confess "Can't open '$self->{r_path}': $!";
        print $fh $boilerplate;
    }

    return $modified;
}

# Generate boilerplate code.
sub _gen_boilerplate {
    my $self = shift;
    my ( $class_nick, $methods, $struct_name )
        = @{$self}{qw( class_nick methods struct_name )};
    my $uc_class_nick = uc($class_nick);
    my $uc_struct     = uc($struct_name);
    my $bp            = Util::autogen_header();

    # collect non-inherited methods
    my @native_methods
        = grep { $_->get_class_nick eq $self->{class_nick} } @$methods;

    # start the include guard
    my $include_guard_name = uc("R_KINO_$self->{class_nick}");
    $bp .= qq|#ifndef $include_guard_name\n#define $include_guard_name 1\n\n|;

    # include the header file
    my ($header_c_path) = $self->{h_path} =~ /^.*?(KinoSearch.*)/;
    $bp .= qq|#include "$header_c_path"\n\n|;

    # declare typedefs for native methods, to ease casting
    for my $method (@native_methods) {
        $bp .= $method->typedef_dec . "\n";
    }

    # define method macros
    for my $method (@$methods) {
        $bp .= $method->macro_def($class_nick) . "\n";
    }

    # define the virtual table
    $bp .= "struct KINO_${uc_struct}_VTABLE {\n";
    $bp .= "    KINO_OBJ_VTABLE *_;\n";
    $bp .= "    kino_u32_t refcount;\n";
    $bp .= "    KINO_OBJ_VTABLE *parent;\n";
    $bp .= "    const char *class_name;\n";
    for my $method (@$methods) {
        $bp .= "    kino_"
            . $method->typedef . " "
            . $method->get_micro_name . ";\n";
    }
    $bp .= "};\n\n";

    # declare the virtual table
    $bp .= "extern KINO_${uc_struct}_VTABLE KINO_$uc_struct;\n\n";

    # define short names
    $bp .= "#ifdef KINO_USE_SHORT_NAMES\n";
    $bp .= "  #define $struct_name kino_$struct_name\n";
    $bp .= "  #define $uc_struct KINO_$uc_struct\n";
    for my $function ( @{ $self->{functions} } ) {
        $bp .= $function->short_func;
    }
    for my $method (@native_methods) {
        $bp .= $method->short_typedef
            unless $method->isa("Method::Overridden");
        $bp .= $method->short_func;
    }
    for my $method (@$methods) {
        $bp .= $method->short_method_macro($class_nick);
    }
    $bp .= "  #define $uc_struct KINO_$uc_struct\n";
    $bp .= "#endif /* KINO_USE_SHORT_NAMES */\n\n";

    # define member vars macro
    $bp .= "#define KINO_${uc_struct}_MEMBER_VARS \\\n    ";
    my @declarations = map { $_->get_type . " " . $_->get_name . ";" }
        @{ $self->{member_vars} };
    $bp .= join( " \\\n    ", @declarations );
    $bp .= "\n\n";

    # define the vtable
    $bp .= $self->_vtable_boilerplate;

    # close the include guard
    $bp .= "#endif /* $include_guard_name */\n\n";

    # add a copyright notice
    $bp .= Util::copyright_footer();

    return $bp;
}

# Create the vtable.
sub _vtable_boilerplate {
    my $self      = shift;
    my $uc_struct = uc( $self->{struct_name} );

    # create a pointer to the parent class's vtable
    my $parent_ref = "NULL";    # Obj only
    if ( defined $self->{parent} ) {
        $parent_ref
            = "(KINO_OBJ_VTABLE*)" . uc("&KINO_$self->{parent}{struct_name}");
    }

    # spec functions which implment the methods, casting to quiet compiler
    my @implementing_funcs
        = map { '(kino_' . $_->typedef . ')' . $_->get_full_func_name }
        @{ $self->{methods} };

    # join the vtable's vtable, the vtable's refcount, the parent class's
    # vtable, the class name, and the funcs
    my $vtable = join( ",\n    ",
        '(KINO_OBJ_VTABLE*)&KINO_VIRTUALTABLE',
        '1', $parent_ref, qq|"$self->{class_name}"|, @implementing_funcs );

    return <<END_VTABLE

#ifdef KINO_WANT_${uc_struct}_VTABLE
KINO_${uc_struct}_VTABLE KINO_$uc_struct = {
    $vtable
};
#endif /* KINO_WANT_${uc_struct}_VTABLE */

END_VTABLE
}

package Class::Final;
use base qw( Class );

sub is_final {1}

package Function;
use Carp;

sub new {
    my $either     = shift;
    my $perl_class = ref($either) || $either;
    my $self       = bless {
        return_type => undef,
        class_name  => undef,
        class_nick  => undef,
        struct_name => undef,
        arg_list    => undef,
        micro_name  => undef,
        @_,
    }, $perl_class;
    return $self;
}

# Accessors
sub get_return_type { shift->{return_type} }
sub get_micro_name  { shift->{micro_name} }
sub get_class_nick  { shift->{class_nick} }
sub get_arg_list    { shift->{arg_list} }

# Reconstruct the full name of the function.
sub get_full_func_name {
    my $self = shift;
    return "kino_$self->{class_nick}_$self->{micro_name}";
}

# Return the function's short name.
sub short_func {
    my $self       = shift;
    my $short_name = "$self->{class_nick}_$self->{micro_name}";
    return "  #define $short_name kino_$short_name\n";
}

package Method;
use Carp;
use base qw( Function );

sub new {
    my $class = shift;
    my $self  = $class->SUPER::new(@_);

    # count the number of arguments and produce a symbolic list
    my @args         = split /,/, $self->{arg_list};
    my $sym_arg_list = "_self";
    my $arg_count    = 1;
    for ( 1 .. $#args ) {
        $sym_arg_list .= ", _arg" . $arg_count++;
    }
    $self->{sym_arg_list} = $sym_arg_list;

    if ( !defined $self->{macro_name} ) {
        # transform method_name to Method_Name
        $self->{macro_name} = $self->{micro_name};
        $self->{macro_name} =~ s/((?:^|_).)/\U$1/g;
    }

    if ( !defined $self->{typedef} ) {
        $self->{typedef}     = $self->_gen_typedef;
        $self->{typedef_dec} = $self->_gen_typedef_dec;
    }

    return $self;
}

sub override {
    my ( $self, $orig ) = @_;

    confess(  "Attempt to override final method '$orig->{micro_name}' from "
            . "$orig->{class_nick} by $self->{class_nick}" )
        if $orig->isa("Method::Final");

    # rebless the object
    bless $self, 'Method::Overridden';

    # remember the method we're overriding
    $self->set_orig($orig);
}

sub finalize {
    my $self = shift;
    return bless {
        %$self,
        # these are needed in case this method is overriding another
        typedef_dec => $self->typedef_dec,
        typedef     => $self->typedef,
        },
        'Method::Final';
}

# create the name of the function pointer typedef for the method's
# implementing function.
sub typedef { shift->{typedef} }

sub _gen_typedef {
    my $self = shift;
    return "$self->{class_nick}_$self->{micro_name}_t";
}

# Create a function pointer typedef.
sub typedef_dec { shift->{typedef_dec} }

sub _gen_typedef_dec {
    my $self = shift;
    return <<END_STUFF;
typedef $self->{return_type}
(*kino_$self->{class_nick}_$self->{micro_name}_t)($self->{arg_list});
END_STUFF
}

# Declare a method macro.  May use the class nick from an inheritor.
sub macro_def {
    my ( $self, $invoker ) = @_;
    my ( $micro_name, $struct_name, $sym_arg_list )
        = @{$self}{qw( micro_name struct_name sym_arg_list )};
    my $full_macro_name = "Kino_" . $invoker . "_" . $self->{macro_name};

    return <<END_STUFF;
#define $full_macro_name($sym_arg_list) \\
    (_self)->_->$micro_name((kino_$struct_name*)$sym_arg_list)
END_STUFF
}

# The typedef's short name.
sub short_typedef {
    my $self       = shift;
    my $short_name = "$self->{class_nick}_$self->{micro_name}_t";
    return "  #define $short_name kino_$short_name\n";
}

# The method macro's short name.
sub short_method_macro {
    my ( $self, $invoker ) = @_;
    my $short_name = $invoker . "_$self->{macro_name}";
    return "  #define $short_name Kino_$short_name\n";
}

package Method::Overridden;
use Carp;
use base qw( Method );

sub new { confess "Objects can only be reblessed into " . __PACKAGE__ }

sub set_orig {
    my ( $self, $orig ) = @_;
    $self->{orig} = $orig;
}

sub typedef       { shift->{orig}->typedef }
sub short_typedef { shift->{orig}->short_typedef }

# Create a method macro, using this class's nick, but inheriting everything
# else.
sub macro_def {
    my ( $self, $invoker ) = @_;
    confess("cant find ancestor for $self->{class_nick} $self->{micro_name}")
        unless defined $self->{orig};
    return $self->{orig}->macro_def($invoker);
}

package Method::Final;
use base qw( Method );
use Carp;

sub macro_def {
    my ( $self, $invoker ) = @_;
    my ( $macro_name, $struct_name, $sym_arg_list )
        = @{$self}{qw( macro_name struct_name sym_arg_list )};
    my $full_macro_name = "Kino_${invoker}_$macro_name";
    my $full_func_name  = $self->get_full_func_name;

    return <<END_STUFF;
#define $full_macro_name($sym_arg_list) \\
    $full_func_name((kino_$struct_name*)$sym_arg_list)
END_STUFF
}

package MemberVar;
use Carp;

sub new {
    my $either     = shift;
    my $perl_class = ref($either) || $either;
    my $self       = bless {
        type => undef,
        name => undef,
        @_,
    }, $perl_class;
    return $self;
}

sub get_type { shift->{type} }
sub get_name { shift->{name} }

__END__

=head1 NAME

boilerplater.pl - Generate boilerplate OO code for KinoSearch

=head1 SYNOPSIS

     perl boilerplater.pl INPUT_DIR DEST_DIR

=head1 PRIVATE API

boilerplater.pl is an implementation detail.  This documentation is partial --
enough for the curious hacker, but not a full API.

=head1 DESCRIPTION

The boilerplater.pl utility looks for special keywords in C header files and
generates boilerplate OO code when it finds them.  The keywords all begin with
C<KINO_>, and they must be the first item on a line to be recognized as valid.

As far as the C compiler is concerned, the keywords are simply macros defined
to do nothing -- so they only have meaning vis-a-vis boilerplater.

=head2 Rationale

Inheritance and polymorphism are messy in C, but basically required for
implementing Lucene.  Auto-generating code makes saves a lot of error-prone
typing.

=head2 Object Model

=over

=item *

Single inheritance.

=item *

Memory management via reference counting.

=item

Method dispatch using virtual tables.

=item *

All classes descend from KinoSearch::Util::Obj.

=back

=head2 Inclusion

The boilerplate code is written to a file with whose name is the same as the
header file, but with a ".r" extension (for "representation") rather than
".h".  Files should include "KinoSearch/Util/Foo.r" rather than
"KinoSearch/Util/Foo.h".

=head2 Short names

Short names -- names minus the prefix "kino_", "Kino_", or "KINO_" -- will be
auto-generated for all class symbols.

=head1 Header File 

Class declarations begin with a C<KINO_CLASS> directive and end with
C<KINO_END_CLASS>.  They must be prepared by including Obj.r and pre-declaring
the object and vtable typedefs.

    #ifndef H_KINO_FOO
    #define H_KINO_FOO 1

    #include "KinoSearch/Util/Obj.r"

    typedef struct kino_Foo kino_Foo;
    typedef struct KINO_FOO_VTABLE KINO_FOO_VTABLE;

    KINO_CLASS("KinoSearch::Util::Foo", "Foo", "KinoSearch::Util::Obj")

    struct kino_Foo {
        const KINO_FOO_VTABLE *_;
        kino_u32_t refcount;
        kino_u32_t num_widgets;
    };

    KINO_FUNCTION(
    kino_Foo*
    kino_Foo_new());
    
    KINO_METHOD("Kino_Foo_Do_Stuff",
    int
    kino_Foo_do_stuff(Foo *self));

    KINO_METHOD("Kino_Foo_Destroy",
    void
    kino_Foo_destroy(Foo *self));

    KINO_END_CLASS

    #endif /* H_KINO_FOO */

In between C<KINO_CLASS> and C<KINO_END_CLASS>, all code must adhere strictly
to all guidelines.  Comments are allowed, but auxiliary macros, functions and
such should be defined outside the class declaration, since boilerplater's
parser will throw an error if it finds something it doesn't understand.

=head2 Pre-declaring object and vtable structs

The name of the object struct must be in UpperCamelCase and be prepended with
C<kino_>:

    typedef kino_ClassName kino_ClassName;

The vtable struct's name must be an upper-cased version of the struct name
with _VTABLE appended.  (The vtable struct definition will be auto-generated.)

    typedef struct KINO_CLASSNAME_VTABLE KINO_CLASSNAME_VTABLE;

=head2 Object struct definition

The class's object struct definition must adhere to the following criteria:

=over

=item *

The first member must be a vtable pointer named C<_>.

=item *

The second member must be C<kino_u32_t refcount>.

=item *

If the class inherits anything besides Obj, it must match all members of its
parent other than the vtable.  To this end, a macro named
C<KINO_CLASSNAME_MEMBER_VARS> is auto-generated for each class which contains
all members save C<_> and C<refcount>. This macro should follow C<refcount> in
the child class's object struct definition:

    struct kino_FooJr {
        KINO_FOOJR_VTABLE *_;
        kino_u32_t refcount;
        KINO_FOO_MEMBER_VARS
        kino_i32_t another_variable;
    };

=back

=head1 Keywords

=head2 KINO_CLASS( [class_name], [class_nick], [parent_class] );

Begin a class declaration.  Three double-quoted string arguments are required,
and the struct definition for the object must follow on immediately
afterwards.

=over

=item *

B<class_name> - The name of this class.  The last word should match the
struct's short name.

=item *

B<class_nick> - A recognizable abbreviation of the class name, used as a
prefix for every function and method.

=item *

B<parent_class> - The full name of the parent class.

=back

=head2 KINO_END_CLASS

Terminate a class declaration.

=head2 KINO_FUNCTION(function_declaration);

All this directive does is generate a short name and perform some validation.
The function's name must follow this pattern:
    
    kino_ . $class_nick . '_' . $micro_name;

C<$class_nick> must be the one specified in KINO_CLASS.  $micro_name must
consist of only characters matching C<[a-z0-9_]>.

=head2 KINO_METHOD( "Kino_ClassNick_Method_Name", function_declaration );

This directive does everything KINO_FUNCTION does, but in addition assigns
method semantics to the first argument, which must be a double-quoted string.

Method declarations look identical to function declarations, and the function
names must match the same pattern.  However, as an additional constraint, the
first argument must be an object.

=head2 KINO_METHOD( "Kino_ClassNick_Method_Name", function_declaration );

As KINO_METHOD above, but method macro becomes a direct alias to the function
name, rather than a vtable invocation.

=head2 KINO_FINAL_CLASS( [class_name], [class_nick], [parent_class] );

As KINO_CLASS above, but causes all methods to become final.

=head1 Class definition file

There is only one requirement for the C file where the class is fully defined.
Before the pound-include directive for the ".r" file, the vtable must be
claimed:

    #define KINO_WANT_FOO_VTABLE
    #include "KinoSearch/Util/Foo.r"

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2007 Marvin Humphrey

This program is free software; you can redistribute it and/or modify under the
same terms as Perl itself.

=cut
