package Device::PaPiRus;
#---AUTOPRAGMASTART---
use 5.012;
use strict;
use warnings;
use diagnostics;
use mro 'c3';
use English qw( -no_match_vars );
use Carp;
our $VERSION = 1.1;
no if $] >= 5.017011, warnings => 'experimental::smartmatch';
use Fatal qw( close );
#---AUTOPRAGMAEND---

use GD;

sub new {
    my ($proto, %config) = @_;
    my $class = ref($proto) || $proto;

    my $self = {};

    bless $self, $class; # bless with our class

    # Let's load the display size
    open(my $ifh, '<', '/dev/epd/panel') or croak($OS_ERROR);
    my $line = <$ifh>;
    close $ifh;
    if($line =~ /\ (\d+)x(\d+)\ /) {
        ($self->{width}, $self->{height}) = ($1, $2);
    } else {
        croak("Can't read panel dimensions!");
    }

    # Default
    $self->{threshold} = 150;
    $self->{randomize_white} = 0;
    $self->{randomize_black} = 0;
    $self->{dithering} = 0;

    return $self;
}

sub getWidth {
    my ($self) = @_;

    return $self->{width};
}

sub getHeight {
    my ($self) = @_;

    return $self->{height};
}

sub randomizeWhite {
    my ($self, $val) = @_;

    $self->{randomize_white} = $val;
    return;
}

sub randomizeBlack {
    my ($self, $val) = @_;

    $self->{randomize_black} = $val;
    return;
}

sub useDithering {
    my ($self, $val) = @_;

    $self->{dithering} = $val;
    return;
}

sub setThreshold {
    my ($self, $threshold) = @_;

    $threshold = 0 + $threshold;
    if($threshold < 0) {
        croak("Threshold can not be less than zero");
    } elsif($threshold > 255) {
        croak("Threshold can not be larger than 255");
    }
    $self->{threshold} = $threshold;
    return;
}

sub fullUpdate {
    my ($self, $img) = @_;

    my $panelImage;
    if(!$self->{dithering}) {
        $panelImage = $self->calculateImage($img);
    } else {
        $panelImage = $self->calculateDitheringImage($img);
    }   
    return $self->writeImage($panelImage, 'U');
}

sub partialUpdate {
    my ($self, $img) = @_;

    my $panelImage;
    if(!$self->{dithering}) {
        $panelImage = $self->calculateImage($img);
    } else {
        $panelImage = $self->calculateDitheringImage($img);
    }   
    return $self->writeImage($panelImage, 'P');
}

sub writeImage {
    my ($self, $img, $mode) = @_;

    open(my $ofh, '>', '/dev/epd/display') or croak($!);
    binmode $ofh;
    print $ofh $img;
    close $ofh;

    open(my $cfh, '>', '/dev/epd/command') or croak($!);
    print $cfh $mode;
    close $cfh;

    return;
}

sub calculateImage {
    my ($self, $img) = @_;
    my $outimg = '';

    my ($sourcewidth, $sourceheight) = $img->getBounds();
    if($sourcewidth != $self->{width} || $sourceheight != $self->{height}) {
        croak('Image dimensions (' . $sourcewidth . 'x' . $sourceheight . ') do not match panel size (' . $self->{width} . 'x' . $self->{height} . ')!');
    }

    # We need to read 8 pixels of the image in one go, turn them into pure black&white bits and stuff the 8 of them into a single byte,
    # correcting for endianess and all that...
    for(my $y = 0; $y < $self->{height}; $y++) {
        for(my $x = 0; $x < ($self->{width} / 8); $x++) {
            my $buf = '';
            for(my $offs = 0; $offs < 8; $offs++) {
                my $index = $img->getPixel(($x*8) + $offs,$y);
                my ($r,$g,$b) = $img->rgb($index);
                my $grey = int(($r+$g+$b)/3);
                if($grey > $self->{threshold}) {
                    if($self->{randomize_white} && int(rand(10000)) % 4 == 0) {
                        $buf .= "1";
                    } else {
                        $buf .= "0";
                    }
                } else {
                    if($self->{randomize_black} && int(rand(10000)) % 4 == 0) {
                        $buf .= "0";
                    } else {
                        $buf .= "1";
                    }
                }
            }
            my $byte = pack('b8', $buf);
            $outimg .= $byte;
        }
    }

    return $outimg;
}

sub calculateDitheringImage {
    my ($self, $img) = @_;
    my $outimg = '';

    my ($sourcewidth, $sourceheight) = $img->getBounds();
    if($sourcewidth != $self->{width} || $sourceheight != $self->{height}) {
        croak('Image dimensions (' . $sourcewidth . 'x' . $sourceheight . ') do not match panel size (' . $self->{width} . 'x' . $self->{height} . ')!');
    }

    # Init array with the greyscale pixel value of the image
    my @npixel;
    for(my $x = 0; $x < $self->{width}; $x++) {
        my @nline;
        for(my $y = 0; $y < $self->{height}; $y++) {
            my $index = $img->getPixel($x, $y);
            my ($r,$g,$b) = $img->rgb($index);
            my $oldpixel = int(($r+$g+$b)/3);
            my $newpixel = $oldpixel / 255;
            push @nline, $newpixel;
        }
        $npixel[$x] = \@nline;
    }


    # Run dithering
    for(my $y = 0; $y < $self->{height}; $y++) {
        for(my $x = 0; $x < $self->{width}; $x++) {
            my $newpixel = $npixel[$x]->[$y];

            my $quant_error = (0.5 - $newpixel);

            # Correct neighboring pixels (if they exist)
            if(($x + 1) < $self->{width}) {
                $npixel[$x + 1]->[$y] = $npixel[$x + 1]->[$y] + ($quant_error * 7/16);
            }
            if($x > 0 && ($y + 1) < $self->{height}) {
                $npixel[$x - 1]->[$y + 1] = $npixel[$x - 1]->[$y + 1] + ($quant_error * 3/16);
            }
            if(($y + 1) < $self->{height}) {
                $npixel[$x]->[$y + 1] = $npixel[$x]->[$y + 1] + ($quant_error * 5/16);
            }
            if(($x + 1) < $self->{width} && ($y + 1) < $self->{height}) {
                $npixel[$x + 1]->[$y + 1] = $npixel[$x + 1]->[$y + 1] + ($quant_error * 1/16);
            }
        }
    }

    # We need to read 8 pixels of the image in one go, turn them into pure black&white bits and stuff the 8 of them into a single byte,
    # correcting for endianess and all that...
    for(my $y = 0; $y < $self->{height}; $y++) {
        for(my $x = 0; $x < ($self->{width} / 8); $x++) {
            my $buf = '';
            for(my $offs = 0; $offs < 8; $offs++) {
                my $raw = $npixel[($x * 8) + $offs]->[$y];
                if($raw >= 0.5) {
                    $buf .= '1';
                } else {
                    $buf .= '0';
                }
            }
            my $byte = pack('b8', $buf);
            $outimg .= $byte;
        }
    }

    return $outimg;
}


1;
__END__

=head1 NAME

Device::PaPiRus - Raspberry Pi "PaPiRus" e-paper display

=head1 SYNOPSIS

  use Device::PaPiRus;
  use GD;
  
  my $img = GD::Image->new('cvc.png');
  my $papirus = Device::PapiRus->new();

  $papirus->setThreshold(100);
  $papirus->fullUpdate($img);
  

=head1 DESCRIPTION

Device::PaPiRus is a library to use the PaPiRus e-paper display from Perl
with the help of the GD image library.

The Image must match the size of the panel exactly. Also the transformation to a single-bit black&white image in
this library is rather simple: take the average of R+G+B to make it greyscale, then see if it's above the given
threshold.

While the implementation is simple, it still allows you a few simple "animations" with whatever framerate you can get out of your panel. For example, if
you have a white-to-black gradient in the image, you can move the threshold and repaint the image to make a simple "moving" animation. Also, you can set
either randomize the white or black pixels while repainting the image over and over again. Nothing fancy, for more elaborate stuff look into the GD
library itself.

=head1 FUNCTIONS

=head2 new

Takes no arguments. Checks for a display panel and reads out its size.

=head2 getWidth

Returns the width of the panel in pixels.

=head2 getHeight

Returns the height of the panel in pixels.

=head2 setThreshold

Sets the threshold of where black ends and white begins. Default: 150

=head2 randomizeWhite

A true value means white pixels are randomized. Default: false

=head2 randomizeBlack

A true value means black pixels are randomized. Default: false

=head2 useDithering

Use an experimental dithering implementation, similar to Floyd Steinberg. Default: false

=head2 fullUpdate

Does a "full" update of the display panel (complete clearing by inverting pixels and stuff). Slow and annoying, but
guarantuees that all pixels are nice, shiny and in the correct color.

=head2 partialUpdate

Does a "partial" update of the display panel. This only overwrites pixels that have changed. A bit quicker than a full
update, no annoying going-to-black-and-back flicker, but may leave artifacts. If you need "quick" screen updates for a demo,
a partial update is the way to go. If you want very crisp text, you should choose a full update, at least every few screen
updates.

=head1 INSTALLATION NOTES

This module uses the EPD fuse module from the rePaper project for low level device access. Something like this should
get you going on Raspbian:

First, enable SPI in raspi-config (make sure it's loaded on boot).

Then:

  sudo apt-get install libfuse-dev python-imaging python-setuptools
  sudo easy_install pip

  git clone https://github.com/repaper/gratis.git
  cd gratis/PlatformWithOS
  make PANEL_VERSION=V231_G2 rpi-epd_fuse
  sudo make PANEL_VERSION=V231_G2 rpi-install

The next step is to edit /etc/default/epd-fuse. Make sure you got the correct panel size (EPD_SIZE) selected (for example 2.7).

Then reboot. If there is a problem, try 

  sudo service epd-fuse start

There's also a manual from Adafruit here: L<https://learn.adafruit.com/repaper-eink-development-board-arm-linux-raspberry-pi-beagle-bone-black?view=all>. Beware, the
Adafruit manual is based on a slightly older library and uses a slightly different panel.

=head1 AUTHOR

Rene Schickbauer, E<lt>rene.schickbauer@gmail.comE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2008-2016 by Rene Schickbauer

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.10.0 or,
at your option, any later version of Perl 5 you may have available.

=cut
