package MooX::ClassOnlyConstructor;

# ABSTRACT: Make Moo-based object constructors class-only methods

use Moo::Role ();
 
use constant ROLE_BASE => 'Method::Generate::Constructor::Role::';
{
  $MooX::ClassOnlyContructor::VERSION = 'v0.1_1';
}
 
# The gist of this code was taken from MooX::StrictConstructor
sub import {
    my $class  = shift;

    my $target = caller;
    die "MooX::ClassOnlyConstructor can only be used on Moo classes."
      unless $Moo::MAKERS{$target} && $Moo::MAKERS{$target}{is_class};
 
    my $con = Moo->_constructor_maker_for($target);

    my $con_role = ROLE_BASE;
    if ( $con->can('_generate_constructor') ) {
        $con_role .= 'ClassOnlyConstructor';
    } else {
        $con_role .= 'RetroClassOnlyConstructor';
    }
    Moo::Role->apply_roles_to_object($con, $con_role);
 
    return;
}
 
1;

__END__

=pod

=head1 NAME

MooX::ClassOnlyConstructor - Make Moo-based object constructors class-only methods

=head1 SYNOPSIS

In a module:

  
 package MyApp;
  
 use Moo;
 use MooX::ClassOnlyConstructor;
  
 
Then in a script:

  
 use MyApp;
  
 # this works as expected
 my $ma = MyApp->new();
  
 # and this will throw an exception via croak
 eval { my $ma2 = $ma->new(); };
 print "$#\n";
  
 # prints "'MyApp->new' must be called as a class method ..."
  

=head1 DESCRIPTION

Discussions about what calling a constuctor on an existing object should do
can come close to being holy wars at times. Often, the best course of action
is to simply not allow it so as to avoid the question entirely.

This module gives one the ability to prevent a L<Moo>-based object from
calling its own constructor. It does this by wrapping the constructor code
generated by L<Moo> with a simple check for C<$class> being a reference.

Similar behavior can be achieved with C<before new> but the class can not
be sub-classed properly if that is done. That was the motivation for this
module.

=head1 ERROR CONDITION

It is an error to use this module with a class that doesn't use L<Moo> and
an exception will be thrown in that case.

=head1 SUPPORT

Support is provided by the author. Please use the GitHub repository to report
bugs or make feature requests.

L<https://github.com/boftx/MooX-ClassOnlyConstructor>

=head1 ACKNOWLEDGEMENTS

This module draws heavily on L<MooX::StrictConstructor> as well as much
guidance and cooperation from C<mst> and C<haarg> on C<#moose>.

=head1 AUTHOR

Jim Bacon, E<lt>jim@nortx.comE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2014 by Jim Bacon

This library is free software; you can redistribute it and/or modify it
under the same terms as Perl itself, either Perl version 5.16 or,
at your option, any later version of Perl 5 you may have available.

=cut
