=head1 NAME

Sys::Statistics::Linux::MemStats - Collect linux memory informations.

=head1 SYNOPSIS

   use Sys::Statistics::Linux::MemStats;

   my $lxs   = new Sys::Statistics::Linux::MemStats;
   my $stats = $lxs->get;

=head1 DESCRIPTION

This module collects statistics by the virtual F</proc> filesystem (procfs) and is developed on default vanilla
kernels. It is tested on x86 hardware with the distributions SuSE (SuSE on s390 and s390x architecture as well),
Red Hat, Debian, Asianux, Slackware and Mandrake on kernel versions 2.4 and 2.6 and should run on all linux
kernels with a default vanilla kernel as well. It is possible that this module doesn't run on all distributions
if the procfs is too much changed.

Further it is necessary to run it as a user with the authorization to read the F</proc> filesystem.

=head1 MEMORY INFORMATIONS

Generated by F</proc/meminfo>.

   memused      -  Total size of used memory in kilobytes.
   memfree      -  Total size of free memory in kilobytes.
   memusedper   -  Total size of used memory in percent.
   memtotal     -  Total size of memory in kilobytes.
   buffers      -  Total size of buffers used from memory in kilobytes.
   cached       -  Total size of cached memory in kilobytes.
   slab         -  Total size of memory in kilobytes that used by kernel for data structure allocations.
   dirty        -  Total size of memory pages in kilobytes that waits to be written back to drive.
   mapped       -  Total size of memory in kilbytes that is mapped by devices or libraries with mmap.
   swapused     -  Total size of swap space is used is kilobytes.
   swapfree     -  Total size of swap space is free in kilobytes.
   swapusedper  -  Total size of swap space is used in percent.
   swaptotal    -  Total size of swap space in kilobytes.

=head1 METHODS

=head2 new()

Call C<new()> to create a new object.

   my $lxs = new Sys::Statistics::Linux::MemStats;

=head2 get()

Call C<get()> to get the statistics. C<get()> returns the statistics as a hash reference.

   my $stats = $lxs->get;

=head1 EXPORTS

No exports.

=head1 SEE ALSO

B<proc(5)>

=head1 REPORTING BUGS

Please report all bugs to <jschulz.cpan(at)bloonix.de>.

=head1 AUTHOR

Jonny Schulz <jschulz.cpan(at)bloonix.de>.

=head1 COPYRIGHT

Copyright (c) 2006, 2007 by Jonny Schulz. All rights reserved.

This program is free software; you can redistribute it and/or modify it under the same terms as Perl itself.

=cut

package Sys::Statistics::Linux::MemStats;
our $VERSION = '0.03';

use strict;
use warnings;
use IO::File;
use Carp qw(croak);

sub new {
   return bless {
      files => {
         meminfo => '/proc/meminfo',
      },
      stats => {},
   }, shift;
}

sub get {
   my $self  = shift;
   $self->{stats} = $self->_load;
   return $self->{stats};
}

#
# private stuff
#

sub _load {
   my $self  = shift;
   my $class = ref $self;
   my $file  = $self->{files};
   my $fh    = new IO::File;
   my %meminfo;

   $fh->open($file->{meminfo}, 'r') or croak "$class: unable to open $file->{meminfo} ($!)";
   while (my $line = <$fh>) {
      if ($line =~ /^(MemTotal|MemFree|Buffers|Cached|SwapTotal|SwapFree|Slab|Dirty|Mapped):\s*(\d+)/) {
         my ($n, $v) = ($1, $2);
         $n =~ tr/A-Z/a-z/;
         $meminfo{$n} = $v;
      }
   }
   $fh->close;

   $meminfo{memused}    = sprintf('%u', $meminfo{memtotal} - $meminfo{memfree});
   $meminfo{memusedper} = sprintf('%.2f', 100 * $meminfo{memused} / $meminfo{memtotal});
   $meminfo{swapused}   = sprintf('%u', $meminfo{swaptotal} - $meminfo{swapfree});

   # maybe there is no swap space on the machine
   if (!$meminfo{swaptotal}) {
      $meminfo{swapusedper} = 0;
   } else {
      $meminfo{swapusedper} = sprintf('%.2f', 100 * $meminfo{swapused} / $meminfo{swaptotal});
   }

   return \%meminfo;
}

1;
