#
#
#  Copyright (c) 2003 Andrew W. Speer <andrew.speer@isolutions.com.au>. All rights 
#  reserved.
#
#  This file is part of WebMod::Debug.
#
#  WebMod::Debug is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
#  $Id: Debug.pm,v 1.22 2004/06/08 07:52:18 aspeer Exp $


#
#  Package to manage debugging
#
package WebMod::Debug;


#  Compiler pragma
#
use strict qw(vars);
use vars   qw($VERSION $REVISION $PACKAGE @EXPORT %EXPORT_TAGS);


#  Use the constants module to get TRUE, FALSE etc
#
use WebMod::Log;
use Data::Dumper;
use IO::File;
use Carp;


#  Require the Exporter
#
require Exporter;


#  We don't use WebMod::Constants, define our own OK
#
use constant	OK  => 1;


#  Version Info, must be all one line for MakeMaker, CPAN.
#
$VERSION = eval { require WebMod::Debug::VERSION; do $INC{'WebMod/Debug/VERSION.pm'}};


#  Release info
#
$REVISION = (qw $Revision: 1.22 $)[1];


#  Package
#
$PACKAGE=__PACKAGE__;


#  Stub vars, fix later
#
my ($FILE_CONFIG, $Debug_hr);


#  Read in the debug file, if present, no error if not present. Default hander
#  var that can be set in import routine and read later;
#
#my $Debug_hr=do $FILE_CONFIG;
my %Debug_Handler;



#  Constant
#
my %Constant=(

    D_NONE                  =>      0,

    D_INIT                  =>      2**0,

    D_METHOD                =>      2**1,

    D_SUB                   =>      2**1,  #  Same as above

    D_UNDEF                 =>      2**2,

    D_UNDEF                 =>      2**3,

    D_UNDEF                 =>      2**4,

    D_INFO                  =>      2**4,

    D_INFO_0                =>      2**4,  #  Same as above

    D_INFO_1                =>      2**5,

    D_INFO_2                =>      2**6,

    D_INFO_3                =>      2**7,

    D_ALL                   =>      (2**8)-1,

    D_DEFAULT               =>      (2**8)-1,  # Default log level

    OK			    =>	    1,

);
my $D_DEFAULT=$Constant{'D_DEFAULT'};


#  All done, return OK
#
return 1;


#============================================================================


sub import {


    #  Custom import handler.
    #
    my ($class,$export)=@_;


    #  Export the debug, Dumper subs and $D_xxx vars by default.
    #
    @EXPORT=
	#(qw(debug Dumper), map { "\$$_" } keys %{$PACKAGE->{'Config'}});
	(qw(debug Dumper), map { "\$$_" } keys %Constant);


    #  Map to various tag names also, really not needed just lets us
    #  do special stuff with certain tags.
    #
    my @export_ok_tags=qw(all debug stdout null);
    map { $EXPORT_TAGS{$_}=\@EXPORT }  @export_ok_tags;
    &Exporter::export_ok_tags(@export_ok_tags);


    #  Get caller
    #
    my $caller=(caller(0))[0];


    #  Defult action on certain tags, really user can only pick one tag
    #
    $Debug_Handler{$caller}={

	':stdout'    =>	'WebMod::Log::Print',
	':null'	     =>	undef,

    }->{$export};


    #  And use the magical goto syntax to call the Export::import
    #  routine as if we were never here
    #
    goto &Exporter::import;

}


sub debug {



    #  Only called first time, means we are not init'd yet. Don't use
    #  prototyping here, as it may not be loaded yet.
    #
    #  This method is replaced in the caller by a closure after it is
    #  run once. See below.
    #
    my $param_hr=(ref($_[0]) eq 'HASH') ? $_[0] : { message => \@_ };


    #  Debugging call. Get the caller info first. Return if no caller info
    #
    my $caller=$param_hr->{'caller'} || (caller(0))[0] ||
	return OK;


    #  Get the debugging info for this caller. First local, then caller debug var,
    #  then caller constant file
    #
    my ($debug_param, @debug_param);
    switch: {


	#  Environment var overrides all
	#
	if (my $env=$ENV{"PERL_DEBUG_${caller}"} ||  $ENV{'PERL_DEBUG_ALL'} || ($ENV{'PERL_DEBUG'} eq $caller) ) {
	    if (-f $env) {
	    	$debug_param={
	    		handler		=>	'WebMod::Log::File',
	    		filename	=>	$env,
	    		mode		=>	O_CREAT|O_APPEND|O_WRONLY
	    	}
	    }
	    else {
	    		$debug_param={ handler=>'WebMod::Log::Print' };
	    }
	    last;
	}


	#  First option
	#
	($debug_param=$Debug_hr->{$caller}) && last;


	#  Second option
	#
	($debug_param=${"${caller}::DEBUG"}) && last;


	#  Third
	#
	$debug_param=${\"::$caller"}->{'Config'}{'DEBUG'} && last;


    }


    #  Plug into array
    #
    switch: {


	#  No string
	#
	$debug_param || do {
	    $Debug_Handler{$caller} &&
	    	(@debug_param={ handler=>$Debug_Handler{$caller} });
	    last;
	};


	#  Plain string
	#
	!(my $ref=ref($debug_param)) && do {
	    @debug_param={ handler=>$debug_param };
	    last
	};


	#  Hash ref
	#
	($ref eq 'HASH') && do {
	    (keys %{$debug_param}) && (@debug_param=$debug_param);
	    last
	};


	#  Array ref
	#
	($ref eq 'ARRAY') && do {
	    @debug_param=@{$debug_param};
	    last;
	};


	#  Fail
	#
	croak("unknown DEBUG reference type: $ref");

    };


    #  If it is empty, set to null routine and return
    #
    @debug_param || do {


	#  It is empty, set routines to null and quit, ensure warning turned
	#  off to avoid "subroutine redefined" type messages
	#
	local $^W=0;
	*{"${caller}::debug"} = sub {};
	*{"${caller}::Dumper"}= sub {};


	#  Returm
	#
	return OK;

    };
    #print Dumper(\@debug_param);


    #  Array to hold debug hash refs
    #
    my @debug_handler;


    #  Run through each debug handler, add to above array
    #
    foreach my $debug_hr (@debug_param) {


	#  Copy
	#
	my %debug=%{$debug_hr};


	#  Lower case everything
	#
	map { $debug{lc($_)}=delete($debug{$_}) } keys %debug;


	#  Xlate filter from array to hash
	#
	map { $debug{filter}{$_}++ } @{delete $debug{'filter'} || []};


	#  Get handler name
	#
	my $handler=$debug{'handler'} ||
	    croak('no handler supplied');


	#  Add handler
	#
	$debug{'log_or'}=WebMod::Log->new($debug_hr) ||
	    croak("could not add log handler$handler");


	#  Default level is all
	#
	$debug{'level'} ||= $D_DEFAULT;


	#  Push onto array
	#
	push @debug_handler, \%debug;


    };


    #  Create self ref to hold all of above
    #
    my %self=(

	debug_handler_ar    =>  \@debug_handler

       );


    #  Bless the self ref
    #
    my $self=bless (\%self, $PACKAGE);


    #  Now replace the caller debug routines with closures to our write routine
    #
    #unless (UNIVERSAL::can($caller, 'debug')) {
    *{"${caller}::debug"}= sub { $self->write(@_) };
    #}


    #  Done, call write routine so we do not lose the message we just received
    #
    $self->write($param_hr);


}


sub write {


    #  Get self ref
    #
    my $self=shift;


    #  Only called first time, means we are not init'd yet. Don't use
    #  prototyping here, as it may not be loaded yet.
    #
    my $param_hr=(ref($_[0]) eq 'HASH') ? $_[0] : { message => \@_ };


    #  Get the calling subname/method
    #
    my $method=$param_hr->{'method'} ||
	(split(/\:/, (caller(2))[3]))[-1] || 'main';


    #  Shortcut for message array ref, return if none
    #
    my $message_ar=$param_hr->{'message'} ||
	croak('no message');


    #  Message level is problematic. It can be spec'd explitely as last
    #  param, or left out for default. How to we know if last param is
    #  message level, or printf param. We have to do a crude check of the
    #  number of %'s, then count our params supplied
    #
    my $message_level=$param_hr->{'level'};
    unless ($message_level) {


	#  Need to work it out
	#
	my $message_param_ix=0;
	map { $message_param_ix++ } ($message_ar->[0]=~/%(?!%)/g);
	map { $message_param_ix-- } ($message_ar->[0]=~/%%/g);
	if (($message_param_ix+1) < @{$message_ar}) { $message_level=pop @{$message_ar} };


	#  Set to default if no level found
	#
	$message_level ||= $D_DEFAULT;

    }


    #  Formulate the actual message
    #
    my $message=sprintf(
	"[$method]  ". shift(@{$message_ar}), @{$message_ar});


    #  Get handler object, go through one by one
    #
    my $debug_handler_ar=$self->{'debug_handler_ar'} ||
	croak('could not get debug_handler array for debug routine');
    foreach my $debug_handler_hr (@{$debug_handler_ar}) {


	#  Skip if not at level we want
	#
	my $debug_level=$debug_handler_hr->{'level'};
	($debug_level & $message_level) || next;


	#  Skip if a method filter has been nominated, but this
	#  does not fit
	#
	if (defined(my $filter_hr=$debug_handler_hr->{'filter'})) {


	    #  Yes, a filter is defined. Do we match it, if not
	    #  then skip
	    #
	    $filter_hr->{$method} || next;

	}


	#  OK to log, get log object and write;
	#
	my $log_or=$debug_handler_hr->{'log_or'} ||
	    croak('unable to get log object in debug handler');
		     $log_or->write({ message=>$message });
    }


    #  And return
    #
    return $message;


}

