#
#
#  Copyright (c) 2003 Andrew W. Speer <andrew.speer@isolutions.com.au>. All rights 
#  reserved.
#
#  This file is part of WebMod::Log.
#
#  WebMod::Log is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
#  $Id: Log_File.pm,v 1.3 2004/03/12 02:35:05 aspeer Exp $

#
#  Log messages to a file
#
package WebMod::Log::File;


#  Compiler pragma
#
use strict 	qw(vars);
use vars 	qw($VERSION $REVISION $PACKAGE);


#  External modules
#
use POSIX qw(strftime);
use IO::File;
use Carp;


#  Version Info, must be all one line for MakeMaker, CPAN.
#
$VERSION = eval { require WebMod::Log::VERSION; do $INC{'WebMod/Log/VERSION.pm'}};


#  Release info
#
$REVISION = (qw $Revision: 1.3 $)[1];


#  Package
#
$PACKAGE=__PACKAGE__;


#  Valid optional handlers
#
my @Option=qw(caller timestamp);


#  All done, return OK
#
return 1;


#============================================================================


sub new {


    #  Start a new object
    #
    my ($class, $param_hr)=@_;
    ($param_hr && (ref($param_hr) eq 'HASH')) ||
	return err('params must be supplied as a HASH reference');


    #  Get details
    #
    my ($filename, $mode, $perms, $quiet)=delete @{$param_hr}
	{qw(filename mode perm quiet)};


    #  Open
    #
    my $self_fh=IO::File->new(grep {$_} ($filename, $mode, $perms)) || do {


	#  It failed. Return stuf if user spec'd quiet open
	#
	if ($quiet) {


	    #  Use the NULL driver
	    #
	    require WebMod::Log::Null; return WebMod::Log::Null->new();

	}
	else {


	    #  Error
	    #
	    return err("unable to open file $filename, $!");

	}

    };


    #  Turn on flush
    #
    $self_fh->autoflush(1);


    #  Options
    #
    my @option=grep { $param_hr->{$_} } @Option;


    #  Store away
    #
    my %self=(

	fh	    =>  $self_fh,
	option_ar   =>  \@option

       );


    #  Bless and return
    #
    return bless(\%self, $class);

}


sub write {


    #  Get self ref, param_hr
    #
    my ($self_or, $param_hr)=@_;
    my $message=$param_hr->{'message'};


    #  If options, do them now
    #
    map { $message=$self_or->$_($message) } @{$self_or->{'option_ar'}};


    #  Send
    #
    my $self_fh=$self_or->{'fh'};
    print $self_fh $message, "\n";


    #  Done
    #
    return 1;

}


sub close {


    #  Get self ref, param_hr
    #
    my ($self_or, $param_hr)=@_;


    #  Close
    #
    my $self_fh=$self_or->{'fh'};
    $self_fh->close();


    #  Undef
    #
    undef $self_fh;
    undef $self_or;


    #  Done
    #
    return 1;

}


sub timestamp {


    #  Timestamp the message
    #
    my ($self, $message)=@_;
    $message=strftime('%b %d %T ', localtime()) . $message;

}


sub caller {

    #  Get calling method
    #
    my ($self, $message)=@_;
    $message=sprintf('[%s] ', &method(3)) . $message;

}


sub err {


    #  Handle errors
    #
    my $errstr=&method(2).': '.sprintf(shift(), @_);
    croak($errstr);

}


sub method {


    #  Quick and dirty fetch of user method
    #
    my $caller=(CORE::caller(defined($_[0]) ? $_[0] : 1))[3];
    return (split(/:/, $caller))[-1] || $caller;

}


sub DESTROY {

    1;
    
    
}
