package Bencher::Scenario::StringFunctions::Indent;

use 5.010001;
use strict;
use warnings;

our $AUTHORITY = 'cpan:PERLANCAR'; # AUTHORITY
our $DATE = '2022-03-27'; # DATE
our $DIST = 'Bencher-Scenarios-StringFunctions'; # DIST
our $VERSION = '0.006'; # VERSION

our $scenario = {
    summary => "Benchmark string indenting (adding whitespace to lines of text)",
    participants => [
        {fcall_template=>'String::Nudge::nudge(<num_spaces>, <str>)'},
        {fcall_template=>'String::Indent::indent(<indent>, <str>)'},
        {fcall_template=>'String::Indent::Join::indent(<indent>, <str>)'},
        # TODO: Indent::String
        # TODO: Indent::Utils
        # TODO: Text::Indent
    ],
    datasets => [
        {name=>'empty'        , args=>{num_spaces=>4, indent=>'    ', str=>''}},
        {name=>'1line'        , args=>{num_spaces=>4, indent=>'    ', str=>join("", map {"line $_\n"} 1..1)}},
        {name=>'10line'       , args=>{num_spaces=>4, indent=>'    ', str=>join("", map {"line $_\n"} 1..10)}},
        {name=>'100line'      , args=>{num_spaces=>4, indent=>'    ', str=>join("", map {"line $_\n"} 1..100)}},
        {name=>'1000line'     , args=>{num_spaces=>4, indent=>'    ', str=>join("", map {"line $_\n"} 1..1000)}},
    ],
};

1;
# ABSTRACT: Benchmark string indenting (adding whitespace to lines of text)

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::StringFunctions::Indent - Benchmark string indenting (adding whitespace to lines of text)

=head1 VERSION

This document describes version 0.006 of Bencher::Scenario::StringFunctions::Indent (from Perl distribution Bencher-Scenarios-StringFunctions), released on 2022-03-27.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m StringFunctions::Indent

To run module startup overhead benchmark:

 % bencher --module-startup -m StringFunctions::Indent

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<String::Nudge> 1.0002

L<String::Indent> 0.03

L<String::Indent::Join>

=head1 BENCHMARK PARTICIPANTS

=over

=item * String::Nudge::nudge (perl_code)

Function call template:

 String::Nudge::nudge(<num_spaces>, <str>)



=item * String::Indent::indent (perl_code)

Function call template:

 String::Indent::indent(<indent>, <str>)



=item * String::Indent::Join::indent (perl_code)

Function call template:

 String::Indent::Join::indent(<indent>, <str>)



=back

=head1 BENCHMARK DATASETS

=over

=item * empty

=item * 1line

=item * 10line

=item * 100line

=item * 1000line

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.34.0 >>, CPU: I<< Intel(R) Core(TM) i5-7200U CPU @ 2.50GHz (2 cores) >>, OS: I<< GNU/Linux Ubuntu version 20.04 >>, OS kernel: I<< Linux version 5.4.0-91-generic >>.

Benchmark with default options (C<< bencher -m StringFunctions::Indent >>):

 #table1#
 {dataset=>"1000line"}
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                  | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Indent::indent       |      2570 |       390 |                 0.00% |                47.77% | 1.6e-07 |      20 |
 | String::Nudge::nudge         |      3650 |       274 |                42.25% |                 3.88% | 2.1e-07 |      20 |
 | String::Indent::Join::indent |      3790 |       264 |                47.77% |                 0.00% | 2.1e-07 |      20 |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

           Rate  SI:i  SN:n  SIJ:i 
  SI:i   2570/s    --  -29%   -32% 
  SN:n   3650/s   42%    --    -3% 
  SIJ:i  3790/s   47%    3%     -- 
 
 Legends:
   SI:i: participant=String::Indent::indent
   SIJ:i: participant=String::Indent::Join::indent
   SN:n: participant=String::Nudge::nudge

=for html <img src="data:image/png;base64,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" />


 #table2#
 {dataset=>"100line"}
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                  | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Indent::indent       |     24000 |      41.7 |                 0.00% |                72.08% | 1.1e-08 |      28 |
 | String::Nudge::nudge         |     33400 |      29.9 |                39.57% |                23.29% | 1.2e-08 |      26 |
 | String::Indent::Join::indent |     41200 |      24.3 |                72.08% |                 0.00% | 6.4e-09 |      22 |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

            Rate  SI:i  SN:n  SIJ:i 
  SI:i   24000/s    --  -28%   -41% 
  SN:n   33400/s   39%    --   -18% 
  SIJ:i  41200/s   71%   23%     -- 
 
 Legends:
   SI:i: participant=String::Indent::indent
   SIJ:i: participant=String::Indent::Join::indent
   SN:n: participant=String::Nudge::nudge

=for html <img src="data:image/png;base64,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" />


 #table3#
 {dataset=>"10line"}
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                  | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Indent::indent       |    188000 |      5.33 |                 0.00% |                95.85% | 1.7e-09 |      20 |
 | String::Nudge::nudge         |    249000 |      4.02 |                32.76% |                47.52% | 1.6e-09 |      23 |
 | String::Indent::Join::indent |    367000 |      2.72 |                95.85% |                 0.00% | 8.3e-10 |      20 |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

             Rate  SI:i  SN:n  SIJ:i 
  SI:i   188000/s    --  -24%   -48% 
  SN:n   249000/s   32%    --   -32% 
  SIJ:i  367000/s   95%   47%     -- 
 
 Legends:
   SI:i: participant=String::Indent::indent
   SIJ:i: participant=String::Indent::Join::indent
   SN:n: participant=String::Nudge::nudge

=for html <img src="data:image/png;base64,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" />


 #table4#
 {dataset=>"1line"}
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                  | rate (/s) | time (ns) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Indent::indent       |    724000 |      1380 |                 0.00% |               166.48% | 4.2e-10 |      20 |
 | String::Nudge::nudge         |    876000 |      1140 |                20.99% |               120.26% | 4.2e-10 |      20 |
 | String::Indent::Join::indent |   1930000 |       518 |               166.48% |                 0.00% | 1.7e-11 |      30 |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

              Rate  SI:i  SN:n  SIJ:i 
  SI:i    724000/s    --  -17%   -62% 
  SN:n    876000/s   21%    --   -54% 
  SIJ:i  1930000/s  166%  120%     -- 
 
 Legends:
   SI:i: participant=String::Indent::indent
   SIJ:i: participant=String::Indent::Join::indent
   SN:n: participant=String::Nudge::nudge

=for html <img src="data:image/png;base64,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" />


 #table5#
 {dataset=>"empty"}
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                  | rate (/s) | time (ns) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Indent::indent       |    745620 |    1341.2 |                 0.00% |               595.24% | 5.8e-12 |      20 |
 | String::Nudge::nudge         |   1020000 |     978   |                37.10% |               407.10% | 4.2e-10 |      20 |
 | String::Indent::Join::indent |   5180000 |     193   |               595.24% |                 0.00% | 9.5e-11 |      24 |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

              Rate  SI:i  SN:n  SIJ:i 
  SI:i    745620/s    --  -27%   -85% 
  SN:n   1020000/s   37%    --   -80% 
  SIJ:i  5180000/s  594%  406%     -- 
 
 Legends:
   SI:i: participant=String::Indent::indent
   SIJ:i: participant=String::Indent::Join::indent
   SN:n: participant=String::Nudge::nudge

=for html <img src="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAtAAAAH4CAMAAABUnipoAAAJJmlDQ1BpY2MAAEiJlZVnUJNZF8fv8zzphUASQodQQ5EqJYCUEFoo0quoQOidUEVsiLgCK4qINEWQRQEXXJUia0UUC4uCAhZ0gywCyrpxFVFBWXDfGZ33HT+8/5l7z2/+c+bec8/5cAEgiINlwct7YlK6wNvJjhkYFMwE3yiMn5bC8fR0A9/VuxEArcR7ut/P+a4IEZFp/OW4uLxy+SmCdACg7GXWzEpPWeGjy0wPj//CZ1dYsFzgMt9Y4eh/eexLzr8s+pLj681dfhUKABwp+hsO/4b/c++KVDiC9NioyGymT3JUelaYIJKZttIJHpfL9BQkR8UmRH5T8P+V/B2lR2anr0RucsomQWx0TDrzfw41MjA0BF9n8cbrS48hRv9/z2dFX73kegDYcwAg+7564ZUAdO4CQPrRV09tua+UfAA67vAzBJn/eqiVDQ0IgALoQAYoAlWgCXSBETADlsAWOAAX4AF8QRDYAPggBiQCAcgCuWAHKABFYB84CKpALWgATaAVnAad4Dy4Aq6D2+AuGAaPgRBMgpdABN6BBQiCsBAZokEykBKkDulARhAbsoYcIDfIGwqCQqFoKAnKgHKhnVARVApVQXVQE/QLdA66At2EBqGH0Dg0A/0NfYQRmATTYQVYA9aH2TAHdoV94fVwNJwK58D58F64Aq6HT8Id8BX4NjwMC+GX8BwCECLCQJQRXYSNcBEPJBiJQgTIVqQQKUfqkVakG+lD7iFCZBb5gMKgaCgmShdliXJG+aH4qFTUVlQxqgp1AtWB6kXdQ42jRKjPaDJaHq2DtkDz0IHoaHQWugBdjm5Et6OvoYfRk+h3GAyGgWFhzDDOmCBMHGYzphhzGNOGuYwZxExg5rBYrAxWB2uF9cCGYdOxBdhK7EnsJewQdhL7HkfEKeGMcI64YFwSLg9XjmvGXcQN4aZwC3hxvDreAu+Bj8BvwpfgG/Dd+Dv4SfwCQYLAIlgRfAlxhB2ECkIr4RphjPCGSCSqEM2JXsRY4nZiBfEU8QZxnPiBRCVpk7ikEFIGaS/pOOky6SHpDZlM1iDbkoPJ6eS95CbyVfJT8nsxmpieGE8sQmybWLVYh9iQ2CsKnqJO4VA2UHIo5ZQzlDuUWXG8uIY4VzxMfKt4tfg58VHxOQmahKGEh0SiRLFEs8RNiWkqlqpBdaBGUPOpx6hXqRM0hKZK49L4tJ20Bto12iQdQ2fRefQ4ehH9Z/oAXSRJlTSW9JfMlqyWvCApZCAMDQaPkcAoYZxmjDA+SilIcaQipfZItUoNSc1Ly0nbSkdKF0q3SQ9Lf5RhyjjIxMvsl+mUeSKLktWW9ZLNkj0ie012Vo4uZynHlyuUOy33SB6W15b3lt8sf0y+X35OQVHBSSFFoVLhqsKsIkPRVjFOsUzxouKMEk3JWilWqUzpktILpiSTw0xgVjB7mSJleWVn5QzlOuUB5QUVloqfSp5Km8oTVYIqWzVKtUy1R1WkpqTmrpar1qL2SB2vzlaPUT+k3qc+r8HSCNDYrdGpMc2SZvFYOawW1pgmWdNGM1WzXvO+FkaLrRWvdVjrrjasbaIdo12tfUcH1jHVidU5rDO4Cr3KfFXSqvpVo7okXY5upm6L7rgeQ89NL0+vU++Vvpp+sP5+/T79zwYmBgkGDQaPDamGLoZ5ht2GfxtpG/GNqo3uryavdly9bXXX6tfGOsaRxkeMH5jQTNxNdpv0mHwyNTMVmLaazpipmYWa1ZiNsulsT3Yx+4Y52tzOfJv5efMPFqYW6RanLf6y1LWMt2y2nF7DWhO5pmHNhJWKVZhVnZXQmmkdan3UWmijbBNmU2/zzFbVNsK20XaKo8WJ45zkvLIzsBPYtdvNcy24W7iX7RF7J/tC+wEHqoOfQ5XDU0cVx2jHFkeRk4nTZqfLzmhnV+f9zqM8BR6f18QTuZi5bHHpdSW5+rhWuT5z03YTuHW7w+4u7gfcx9aqr01a2+kBPHgeBzyeeLI8Uz1/9cJ4eXpVez33NvTO9e7zofls9Gn2eedr51vi+9hP0y/Dr8ef4h/i3+Q/H2AfUBogDNQP3BJ4O0g2KDaoKxgb7B/cGDy3zmHdwXWTISYhBSEj61nrs9ff3CC7IWHDhY2UjWEbz4SiQwNCm0MXwzzC6sPmwnnhNeEiPpd/iP8ywjaiLGIm0iqyNHIqyiqqNGo62ir6QPRMjE1MecxsLDe2KvZ1nHNcbdx8vEf88filhICEtkRcYmjiuSRqUnxSb7JicnbyYIpOSkGKMNUi9WCqSOAqaEyD0tandaXTlz/F/gzNjF0Z45nWmdWZ77P8s85kS2QnZfdv0t60Z9NUjmPOT5tRm/mbe3KVc3fkjm/hbKnbCm0N39qzTXVb/rbJ7U7bT+wg7Ijf8VueQV5p3tudATu78xXyt+dP7HLa1VIgViAoGN1tubv2B9QPsT8M7Fm9p3LP58KIwltFBkXlRYvF/OJbPxr+WPHj0t6ovQMlpiVH9mH2Je0b2W+z/0SpRGlO6cQB9wMdZcyywrK3BzcevFluXF57iHAo45Cwwq2iq1Ktcl/lYlVM1XC1XXVbjXzNnpr5wxGHh47YHmmtVagtqv14NPbogzqnuo56jfryY5hjmceeN/g39P3E/qmpUbaxqPHT8aTjwhPeJ3qbzJqamuWbS1rgloyWmZMhJ+/+bP9zV6tua10bo63oFDiVcerFL6G/jJx2Pd1zhn2m9az62Zp2WnthB9SxqUPUGdMp7ArqGjzncq6n27K7/Ve9X4+fVz5ffUHyQslFwsX8i0uXci7NXU65PHsl+spEz8aex1cDr97v9eoduOZ67cZ1x+tX+zh9l25Y3Th/0+LmuVvsW523TW939Jv0t/9m8lv7gOlAxx2zO113ze92D64ZvDhkM3Tlnv296/d5928Prx0eHPEbeTAaMip8EPFg+mHCw9ePMh8tPN4+hh4rfCL+pPyp/NP637V+bxOaCi+M24/3P/N59niCP/Hyj7Q/Fifzn5Ofl08pTTVNG02fn3Gcufti3YvJlykvF2YL/pT4s+aV5quzf9n+1S8KFE2+Frxe+rv4jcyb42+N3/bMec49fZf4bmG+8L3M+xMf2B/6PgZ8nFrIWsQuVnzS+tT92fXz2FLi0tI/QiyQvpNzTVQAAAAgY0hSTQAAeiYAAICEAAD6AAAAgOgAAHUwAADqYAAAOpgAABdwnLpRPAAAALFQTFRF////AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAlADUlQDVlADUAAAAAAAAAAAAlADUlQDVlADUlADUlQDWlgDXlADUKQA7MQBHFgAfPgBZQgBfQQBeAAAACwAQFgAfBgAIEQAYAAAAAAAAlADURQBj////ndVhNgAAADd0Uk5TABFEZiKIu6qZM8x33e5VTqnVysfSP+/89uzx+fR19+zf5O3wM6dEInUwiOTr1fj7+s7x9u/0W0ZfQOAAAAABYktHRACIBR1IAAAACXBIWXMAAABIAAAASABGyWs+AAAAB3RJTUUH5gMbDzA219EcJgAAEnRJREFUeNrt3Qt34sidhnHdESCRdGbSO5Od7Fx6N9nNfXNT9vt/sVVJAoMx5o8pm3qrnt85M6anbzr0M2qBwW+WAQAAAAAAAAAAAAAAAAAAAPhAebHcKPJHHwrwRmV1uFkMy41hX3ZWD6Pm0ccImDWHeF8KetUWRbF+9DECVuvNdjxFl3VduKCr6eMUdFHXY8hN+egDBG5Rdm2RNV1d9+UYdN+2Qz0Fve3rdiizYU4dUOEuOZrxJF2vxqC3WbYdqjHoahhPz2WfDd3UNaBivoZe7zbNcg09np6HouzGa+di+Fmdj4n3jz5GwMwFXQ9N2+yD7l3Qdd8408PBfOCiAzLGoHe9u+RwQedzvkOx6zL3vHThrjamyw9AQ7MbHxiO8U6XHO0YdueuOvLxMeJ4s3Att5tHHyNgturKfNM1XduXxWbTdf16uowu+81mvDlejHQdJ2joyIvxeqMo8sx9nG7s//t0syq4gAYAAAAAAIA/u27T8MYLxCLv8my1ffRRAJ6U7aOPAPCoXnVDyyUHpM2fpV27jmt3yVE/+oCA1x2/C/nsHcmVez1vtRmmF42NMRe8XRmBO34X8rN3JFe7jQu6WeVVV2frZo4aCNnxu5CX29OL0LdFVjYu6OnV6NuNexXvasM1NAJ3/C7k5Xbe77Jd79p1byGa3kY0/Wt99LrHn3+a/OIrwI+vp6K+/uW9QR+9C3l/e9dV/RSv67icg352bv78b984n78Nyje/evQRvOjfH30AEkf13VTU8Ov7eq6O3oX8dHvVzxfLLujtHHR1+vO+/cr/Xxb3C/QrXoT5UDrMo7o36Mnxu5Cn2+Vyzj695DhG0DcIM50wj+reoI/fhXy4nff1dAk9dVy5k7N71+cJgr5BmOmEeVR3B71/F3JZPt1u6my1mr83m741/XOCoG8QZjphHtXdlxz7dyE3zeH2tsuzqt9l+yc3+k139nwdQd8gzHTCPKr7r6GP34X88juS8xf+a5hBF9X9v8Y7CPNL2YV5VF4eFL5BmEFDHkEjKgSNqBA0okLQiApBIyoEjagQNKJC0IgKQSMqBI2oEDSiQtCICkEjKgSNqBA0okLQiApBIyoEjagQNKJC0Hjdf3x/qx8eebgEjdd9/69bff/IwyVovI6gTQhaBUGbELQKgjYhaBUEbULQKgj6kpNpZIJWQdAXnE4jE7QKgr7gdBqZoFUQ9AWn08gErYKgj12cRiZoFSkHfcM0MkGrSDnoG6aRCVpFykHfMI1M0CpSDvqGaeTPP9ZOmJsmOCITdDkV5Tdo+zTyT58KhzH74MkEvZ6K8hm0/jQyzskEPfP+tJ30NDLOJRy0/jQyzqUctPw0Ms4lHLT+NDLOpRy0/DQyziUdtB1BqyBoE4JWQdAmBK2CoE0IWgVBmxC0CoI2IWgVBG1C0CoI2oSgVRC0CUGrIGgTglZB0CYErYKgTQhaBUGbELQKgjYhaBUEbULQKgjahKBVELQJQasgaBOCVkHQJgStgqBNCFoFQZsQtAqCNiFoFQRtQtAqCNqEoFUQtAlBqyBoE4JWQdAmBK2CoC9hGlkSQV/ANLImgr6AaWRNBH0B08iaUgx6ffF7mEaWl2DQ9X7wmGnkCKUXdHFomGnkCCUXdN6v9kEzjRyh5IJe1YdLjhumkb98aZz17b8fPpZM0PVU1N1Bl5una+gbppE5Q6uQCXp2b9BVVx2CZho5RokFXW/GK46ufsqVaeTIJBZ0UT8FzTRyjBIL2pkuOcqSaeQopRp00zCNHKUEg37CNHJ8kg7ajqBVELQJQasgaBOCVkHQJgStgqBNCFoFQZsQtAqCNiFoFQRtQtAqCNqEoFUQtAlBqyBoE4JWQdAmBK2CoE0IWgVBmxC0CoI2IWgVBG1C0CoI2oSgVRC0CUGrIGgTglZB0CYErYKgTQhaBUGbELQKgjYhaBUEbULQKgjahKBVELQJQasg6EtYkpVE0BewJKuJoC9gSVYTQV/Akqym1IM+3bViSVZe4kHXx0OyLMlGIO2gi+NlZJZkY5B00POqLEuyMUk66HlV1rIk+9Onwsnf+Bvhw8gEvZ6K8hr0flXWsCT7+cfaqW7/TfCxZIIup6J8Bv20KsuSbDxkgp75DPppVZYl2XgkHPRhVZYl2YgkHLQzj3CyJBsPgmZJNiqJB/0SlmSVEbQJQasgaBOCVkHQJgStgqBNCFoFQZsQtAqCNiFoFQRtQtAqCNqEoFUQtAlBqyBoE4JWQdAmBK2CoE0IWgVBmxC0CoI2IWgVBG1C0CoI2oSgVRC0CUGrIGgTglZB0CYErYKgTQhaBUGbELQKgjYhaBUEbULQKgjahKBVELQJQasgaBOCVkHQJgStgqAvYRpZEkFfwDSyJoK+gGlkTQR9AdPImtILunhlyoppZHmpBV32w7B/rFcPo+NtZKaR9SUWdN6XWb5ZLo9XbVEUT+v1TCPHILGgpxWV/WJ9s4y5MY0ckcSCnsyTV+MvVtZ1YZtGZklWhUzQ3pZkm65bTrxDV7ductMwjczWtwqZoL1tfRdlN18aV/XY57bPmEaOiUzQMy+XHLuj4eN8KJhGjkliQU+PB5dcpweD7jEg08gRSSzowj2H0Y65luVye8M0clQSCzprh6brx5Ab9zTzeLtbM40cldSCzqqj4ePqhRFkppG1JRf02xC0CoI2IWgVBG1C0CoI2oSgVRC0CUGrIGgTglZB0CYErYKgTQhaBUGbELQKgjYhaBUEbULQKgjahKBVELQJQasgaBOCVkHQJgStgqBNCFoFQZsQtAqCNiFoFQRtQtAqCNqEoFUQtAlBqyBoE4JWQdAmBK0i0qDXa9uPsyJoFVEGXfZDU3Q+myZoFTEGvR7KosmXL8DoB0GriDHoup3mfjaF4ccaEbSKKIOufQTNNLKkGIMu+vUYdHnfJQfTyJpiDDrbDl3f9eVdvxPTyJqiDDqrynp350NCppE1xRh0NV88l9dH2J7PJDONLC++oKti6xZii1137UHh8Uzy/FOZRpYXX9Bls+kaZ3XlouNkJjljGjkO8QU9XjLYHg7uZ5KZRo5JjEEvDNfQ00yyZRr5y5fppO/5FSLwTyboeirK+FqOlfux/fVPrMwzyYZpZM7QKmSCnhk/sVJvmnrTGn7kPJPMNHI8Ygx6fCy3a6dP9V03zSQzjRyPSIN2z1E0Vy45DjPJTCNHJMagy67KxiyvPQ99mElmGjkiMQbtdo/rvttc+2HLTDLTyDGJMmhnV16/hGYaOT4xBl3c9zq7lxC0ihiD3l692LgZQauIMeisrd2rkzy+A4ugZcQYdDHMPP6+BK0ixqDfAUGrIGgTglZB0CYErYKgTQhaBUGbELQKgjYhaBUEbULQKgjahKBVELQJQasgaBOCVkHQJgStgqBNCFoFQZsQtAqCNiFoFQRtQtAqCNqEoFUQtAlBqyBoE4JWQdAmBK2CoE0IWgVBmxC0CoI2IWgVBH0JS7KSCPoClmQ1EfQFLMlqIugLWJLVlF7Q61fGsViSlZda0OtuGLplna12XwCvOfpOlmT1pRZ032Z5uyyorNyE8tP0IEuyMUgs6GmAcBqLHTXL9hVLshFJLOjcXSXvVwiHsq6LzLQk+9On6StOW4bi8FAyQa+nonw8y1HtJzmHrm7dQqFhSfbzj7VjGVvGQ8kEXU5F3R90Xg/LlXFVj9Vu+4wl2ZjIBD27/1mOzekCfT4ULMnGJLWgu6dPAE4PBt1jQJZkI5JY0LthmRMqy2VJdsOSbFQSC7rezwk17mnmoek6lmTjkljQp1iSjU/SQdsRtAqCNiFoFQRtQtAqCNqEoFUQtAlBqyBoE4JWQdAmBK2CoE0IWgVBmxC0CoI2IWgVBG1C0CoI2oSgVRC0CUGrIGgTglZB0CYErYKgTQhaBUGbELQKgjYhaBUEbULQKgjahKBVELQJQasgaBOCVkHQJgStgqBNCFoFQZsQtAqCNiFoFQRtQtAqCPoSppElEfQFTCNrIugLmEbWRNAXMI2sKe2gn88kM40sL+Wgj2eSJ0wj60s56OOZ5Ixp5DgkHPR+Jplp5JgkHPR+JtkyjcySrAqZoH0tyZ6aZpIN08hsfauQCdrf1veT/Uwy08jxkAl65vdZjv1MMtPI8Ug56P1MMtPIEUk46MNMMtPIEUk46P1MMtPIMUk46EuYRlZG0CYErYKgTQhaBUGbELQKgjYhaBUEbULQKgjahKBVELQJQasgaBOCVkHQJgStgqBNCFoFQZsQtAqCNiFoFQRtQtAqCNqEoFUQtAlBqyBoE4JWQdAmBK2CoE0IWgVBmxC0CoI2IWgVBG1C0CoI2oSgVRC0CUGrIGgTglZB0CYErYKgTQhaBUGbELQKgr6EaWRJBH0B08iaCPoCppE1EfQFTCNrSjDo4tp3MY0sLL2gq8Mg0PQl/Jvn38U0srTUgl72jyertiiK9fPvYhpZWmpBz/vHs2YZc2MaOSKpBX28QTiUdV24Vbfr08hfvjTO+ubfDR9MJuh6Kspv0F3duslNwzQyZ2gVMkHPfAZd1WO12z5jGjkmCQc9yYeCaeSYJBz09GDQPQZkGjkiqQZdluONtXsug2nkqKQadOOeZh6arlszjRyV9II+UhUvfRqcaWRlSQdtR9AqCNqEoFUQtAlBqyBoE4JWQdAmBK2CoE0IWgVBmxC0CoI2IWgVBG1C0CoI2oSgVRC0CUGrIGgTglZB0CYErYKgTQhaBUGbELQKgjYhaBUEbULQKgjahKBVELQJQasgaBOCVkHQJgStgqBNCFoFQZsQtAqCNiFoFQRtQtAqCNqEoFUQ9CUsyUoi6AtYktVE0BewJKuJoC9gSVZT4kEXL32TJVlhaQddDeffZElWWspBH6/KZizJxiHloJdVWZZk3+Q/f7jZf33AYaUc9DKqYlmSJegzv7m5nH/98AGHRdCmJVmmkc8Q9H28TSMfWzYLWZJ9A4L24X2CZkn2DQjah3cJmiXZtyBoH94laJZk34KgfXiPoFmSfROC9uEjXsvBkqwJQfvAC/yDQdA+EHQwCNoHgg4GQftA0MEgaB8IOhgE7QNBB4OgfSDoYBC0DwQdDIL2gaCDQdA+EHQwCNoHgg4GQftA0MEgaB8IOhgE7QNBB4OgfUgy6N/+983+5/2PiqB9SDLo392ezgd8BQyC9oGgCfp1BG1C0GcI2geCJujXEbQJQZ8haB8ImqBfR9AmBH2GoH0gaIJ+HUGbEPQZgvaBoAn6dQRtQtBnCNoHgibo1xG0CUGfIWgfCJqgX0fQJgR9hqB98Bn0Ord9dAj6DEH74CHoehg1+8HYqx9nBH2GoH3wEPSqLYpivR+MvfpxRtBnCNoHD0E30+LVfjD22scFQZ8haB88BD2UdV0c1tuufVwQ9BmC9sFH0F3dDuV+MPbax+Un/fSrT84vvnqE3//fzf7w/kf1x9uP6k8fcGf9+eaj+ssHHNULvp6Kuj/oqh4j3fb7wdhrH5ef9b/fTD5/+wh//dvN/vr+R/X324/qHx9wZ4V5VC/4birqu396OdHnw69vu+QAAlW4x4TV8LNlMLa68hEIW+Gevmg3h8HYa/8AYauHpuvWh8HYax9DVbX3/xqpiPy+qual2P1g7LWPodqsH30EOrivFORB/v0RZjph3lc4sQrwL9JdP4R1nVbP/4OFeF9h0Y5/OHlbZdUQ2jVRXnVFtQnqkXTTTHdWePcVDnbDOsu73Xj2aR59KHvL559a92quoq/u+8X8HtlQTndWOPcVzkxnnfFj3pePPpTZ2j0V6s6DvTs7r1aPPp5jdZe7OyuY+wpHqrZ3yayH5Q+n7AO5Xl25/8XcedB9JqrqQ/jrvV2umvN+vgQK5r7Cwbpvi12/m886k1A+81MNu+kvjbzbjt+qQzgqd2E2KZcX5IRyX2Ev710t87XG+IeT13m2fvjlatNMp+P9/2I7dx4M41TY7K+aN6vpznr8fYVTRy+V2g6hPMnRDL1LOt+/uad5/PNjzy/M3IsdwrizcMI9YC/qzeAe3nSrQE6E48VGPYxJl+65jSDOgycXZrmL2sUcxJ2FU+0wDJt6DGh99Erth1lOhOPjwKodk25WWRDnwdMLs/VQZ1seDoZqXUx/NO5TFw/veX8inK7nx6S7QM6Dzy7MymHowvxsPA66AJ5SfToRzs+HjUlvH31Mk+cXZhU5B8y9MiFfhfBqiaMT4WZ+HBjAQU3CujDDq9q+bfsmhHSOToRFAFfOx0K6MMMV5aoOpJ6jE2Eb5OskQrgwg5CnE2EV2IkwnAsz6AnwRBjOhRmUhHsiDOfCDEI4ESIunAgBAAAAAAAAAFD1/5eSbTeHoVzLAAAAJXRFWHRkYXRlOmNyZWF0ZQAyMDIyLTAzLTI3VDA4OjQ4OjU0KzA3OjAwcCqzvAAAACV0RVh0ZGF0ZTptb2RpZnkAMjAyMi0wMy0yN1QwODo0ODo1NCswNzowMAF3CwAAAAAhdEVYdHBzOkhpUmVzQm91bmRpbmdCb3gANTA0eDcyMCs1MCs1MNbiIsMAAAATdEVYdHBzOkxldmVsAEFkb2JlLTIuMAo5k3QNAAAAAElFTkSuQmCC" />


Benchmark module startup overhead (C<< bencher -m StringFunctions::Indent --module-startup >>):

 #table6#
 +----------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+
 | participant          | time (ms) | mod_overhead_time | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +----------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+
 | String::Indent       |       8.9 |               1.9 |                 0.00% |                20.55% | 1.2e-05   |      20 |
 | String::Nudge        |       8.9 |               1.9 |                 0.08% |                20.45% | 1.2e-05   |      21 |
 | String::Indent::Join |       8.8 |               1.8 |                 1.30% |                19.00% | 1.3e-05   |      20 |
 | perl -e1 (baseline)  |       7   |               0   |                20.55% |                 0.00% |   0.00019 |      20 |
 +----------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+


Formatted as L<Benchmark.pm|Benchmark> result:

                          Rate  S:I  S:N  SI:J  perl -e1 (baseline) 
  S:I                  112.4/s   --   0%   -1%                 -21% 
  S:N                  112.4/s   0%   --   -1%                 -21% 
  SI:J                 113.6/s   1%   1%    --                 -20% 
  perl -e1 (baseline)  142.9/s  27%  27%   25%                   -- 
 
 Legends:
   S:I: mod_overhead_time=1.9 participant=String::Indent
   S:N: mod_overhead_time=1.9 participant=String::Nudge
   SI:J: mod_overhead_time=1.8 participant=String::Indent::Join
   perl -e1 (baseline): mod_overhead_time=0 participant=perl -e1 (baseline)

=for html <img src="data:image/png;base64,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" />


To display as an interactive HTML table on a browser, you can add option C<--format html+datatables>.

=head1 BENCHMARK NOTES

Joining is faster than regex substitution for the datasets tested (0-1000
lines of short text).

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenarios-StringFunctions>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenarios-StringFunctions>.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 CONTRIBUTING


To contribute, you can send patches by email/via RT, or send pull requests on
GitHub.

Most of the time, you don't need to build the distribution yourself. You can
simply modify the code, then test via:

 % prove -l

If you want to build the distribution (e.g. to try to install it locally on your
system), you can install L<Dist::Zilla>,
L<Dist::Zilla::PluginBundle::Author::PERLANCAR>, and sometimes one or two other
Dist::Zilla plugin and/or Pod::Weaver::Plugin. Any additional steps required
beyond that are considered a bug and can be reported to me.

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2022, 2021, 2018 by perlancar <perlancar@cpan.org>.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenarios-StringFunctions>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=cut
