use strict;
use warnings;
use v5.10.0;
package Glib::FindMinVersion;

# ABSTRACT: Find minimum version of GLib needed to compile C source
our $VERSION = '0.004'; # VERSION

use Carp;
use List::Util qw(max);

=pod

=encoding utf8

=head1 NAME

Glib::FindMinVersion - Find minimum version of GLib needed to compile C source


=head1 SYNOPSIS

    use Glib::FindMinVersion;

    print Glib::FindMinVersion::with('g_get_num_processors'); # 2.36
    print scalar Glib::FindMinVersion::for_file('source.c'); # 2.18
    my %symbols = Glib::FindMinVersion::for_file('source.c' => 2.10);
    # (
    #   2.18 => [qw[g_set_error_literal]],
    #   2.12 => [qw[g_ascii_strtoll g_base64_decode]],
    # )
    

=head1 IMPLEMENTATION

Symbol use is detected quite crudely with C<< / \s* (?<symbol>\w+) \s* \( /x >>. Symbols introduced prior to 2.0 are labelled as 2.0.

=cut

my %GLib_version_of;

=head1 METHODS AND ARGUMENTS

=over 4

=item with(@symbols)

Returns minimum GLib version which exports all C<@symbols>.

=cut

sub with {
	my $symbol = shift;
    
    return $GLib_version_of{$symbol};
}

=item for_source($source [, $version])

Here, C<[]> denotes an optional argument.
In list context, returns a listing of all found glib symbols grouped by version. If C<$version> is given, the list is limited to symbols newer than C<$version>. C<$version> is a string of the form C</2\.\d\d?/>.

In scalar context, returns the maximum version in the list, which is the minimum version required to use all the symbols.

=cut

sub version_cmp($$) {
    my ($a, $b) = @_;
    my ($a_major, $a_minor) = $a =~ /(\d+)\.(\d+)/g;
    my ($b_major, $b_minor) = $b =~ /(\d+)\.(\d+)/g;

    return $a_major <=> $b_major || $a_minor <=> $b_minor;
}

sub for_source {
	my ($src, $current_version) = (shift, shift // '0.0');
    
    my @symbols = $src =~ / \s* (\w+) \s* \( /gx;
    return max map { $GLib_version_of{$_} // 0 } @symbols unless wantarray;

    my %version_of = map { ($_, $GLib_version_of{$_} // '0.0') } @symbols;
    my %versions;
    for (keys %version_of) {
        push (@{ $versions{ $version_of{$_} } }, $_) if version_cmp($version_of{$_}, $current_version) == 1;
    }

    return map { ($_, $versions{$_}) } reverse sort version_cmp keys %versions;
}

=item for_file($filename [, $version])

Helper function that reads in C<$filename> and passes it to C<for_source>.

=cut

sub for_file {
    open my $fh, '<', shift;
    local $/;
    return for_source(<$fh>, shift);
}

%GLib_version_of = (
    g_log_writer_supports_color => '2.50',
    g_log_writer_standard_streams => '2.50',
    g_log_writer_journald => '2.50',
    g_log_writer_is_journald => '2.50',
    g_log_writer_format_fields => '2.50',
    g_log_writer_default => '2.50',
    g_log_variant => '2.50',
    g_log_structured_array => '2.50',
    g_log_structured => '2.50',
    g_log_set_writer_func => '2.50',
    g_key_file_load_from_bytes => '2.50',
    g_compute_hmac_for_bytes => '2.50',
    G_VARIANT_BUILDER_INIT => '2.50',
    G_PID_FORMAT => '2.50',
    G_DEBUG_HERE => '2.50',
    GLogWriterOutput => '2.50',
    GLogWriterFunc => '2.50',
    GLogField => '2.50',
    GLIB_VERSION_2_50 => '2.50',

    g_uint_checked_mul => '2.48',
    g_uint_checked_add => '2.48',
    g_uint64_checked_mul => '2.48',
    g_uint64_checked_add => '2.48',
    g_size_checked_mul => '2.48',
    g_size_checked_add => '2.48',
    g_sequence_is_empty => '2.48',
    GLIB_VERSION_2_48 => '2.48',

    g_thread_pool_move_to_front => '2.46',
    g_log_set_handler_full => '2.46',
    g_async_queue_remove_unlocked => '2.46',
    g_async_queue_remove => '2.46',
    g_async_queue_push_front_unlocked => '2.46',
    g_async_queue_push_front => '2.46',
    g_assert_cmpmem => '2.46',
    GLIB_VERSION_2_46 => '2.46',

    g_win32_check_windows_version => '2.44',
    g_strv_contains => '2.44',
    g_steal_pointer => '2.44',
    g_option_group_unref => '2.44',
    g_option_group_ref => '2.44',
    g_option_context_set_strict_posix => '2.44',
    g_option_context_get_strict_posix => '2.44',
    g_mutex_locker_new => '2.44',
    g_mutex_locker_free => '2.44',
    g_autoptr => '2.44',
    g_autofree => '2.44',
    g_auto => '2.44',
    G_DEFINE_AUTO_CLEANUP_FREE_FUNC => '2.44',
    G_DEFINE_AUTO_CLEANUP_CLEAR_FUNC => '2.44',
    G_DEFINE_AUTOPTR_CLEANUP_FUNC => '2.44',
    GMutexLocker => '2.44',
    GLIB_VERSION_2_44 => '2.44',

    G_GNUC_CHECK_VERSION => '2.42',
    GLIB_VERSION_2_42 => '2.42',

    g_win32_get_command_line => '2.40',
    g_variant_parse_error_print_context => '2.40',
    g_variant_dict_unref => '2.40',
    g_variant_dict_remove => '2.40',
    g_variant_dict_ref => '2.40',
    g_variant_dict_new => '2.40',
    g_variant_dict_lookup_value => '2.40',
    g_variant_dict_lookup => '2.40',
    g_variant_dict_insert_value => '2.40',
    g_variant_dict_insert => '2.40',
    g_variant_dict_init => '2.40',
    g_variant_dict_end => '2.40',
    g_variant_dict_contains => '2.40',
    g_variant_dict_clear => '2.40',
    g_str_tokenize_and_fold => '2.40',
    g_str_to_ascii => '2.40',
    g_str_match_string => '2.40',
    g_str_is_ascii => '2.40',
    g_ptr_array_insert => '2.40',
    g_option_context_parse_strv => '2.40',
    g_key_file_save_to_file => '2.40',
    g_info => '2.40',
    g_hash_table_get_keys_as_array => '2.40',
    g_assert_nonnull => '2.40',
    GVariantDict => '2.40',
    GLIB_VERSION_2_40 => '2.40',

    g_variant_new_take_string => '2.38',
    g_variant_new_printf => '2.38',
    g_test_trap_subprocess => '2.38',
    g_test_subprocess => '2.38',
    g_test_skip => '2.38',
    g_test_set_nonfatal_assertions => '2.38',
    g_test_incomplete => '2.38',
    g_test_get_filename => '2.38',
    g_test_get_dir => '2.38',
    g_test_failed => '2.38',
    g_test_build_filename => '2.38',
    g_regex_get_max_lookbehind => '2.38',
    g_assert_true => '2.38',
    g_assert_null => '2.38',
    g_assert_false => '2.38',
    G_KEY_FILE_DESKTOP_KEY_DBUS_ACTIVATABLE => '2.38',
    G_KEY_FILE_DESKTOP_KEY_ACTIONS => '2.38',
    GTestFileType => '2.38',
    GLIB_VERSION_2_38 => '2.38',

    g_variant_new_from_bytes => '2.36',
    g_variant_get_data_as_bytes => '2.36',
    g_unix_fd_source_new => '2.36',
    g_unix_fd_add_full => '2.36',
    g_unix_fd_add => '2.36',
    g_test_initialized => '2.36',
    g_source_set_ready_time => '2.36',
    g_source_remove_unix_fd => '2.36',
    g_source_query_unix_fd => '2.36',
    g_source_modify_unix_fd => '2.36',
    g_source_add_unix_fd => '2.36',
    g_markup_parse_context_unref => '2.36',
    g_markup_parse_context_ref => '2.36',
    g_get_num_processors => '2.36',
    g_close => '2.36',
    GLIB_VERSION_2_36 => '2.36',

    g_variant_check_format_string => '2.34',
    g_test_expect_message => '2.34',
    g_test_assert_expected_messages => '2.34',
    g_test_add_data_func_full => '2.34',
    g_string_free_to_bytes => '2.34',
    g_spawn_check_exit_status => '2.34',
    g_slist_copy_deep => '2.34',
    g_regex_get_has_cr_or_lf => '2.34',
    g_mapped_file_get_bytes => '2.34',
    g_list_copy_deep => '2.34',
    g_datalist_id_replace_data => '2.34',
    g_datalist_id_dup_data => '2.34',
    g_compute_checksum_for_bytes => '2.34',
    g_clear_pointer => '2.34',
    G_DEFINE_QUARK => '2.34',
    GLIB_VERSION_2_34 => '2.34',

    g_variant_new_fixed_array => '2.32',
    g_thread_unref => '2.32',
    g_thread_try_new => '2.32',
    g_thread_ref => '2.32',
    g_thread_new => '2.32',
    g_rw_lock_writer_unlock => '2.32',
    g_rw_lock_writer_trylock => '2.32',
    g_rw_lock_writer_lock => '2.32',
    g_rw_lock_reader_unlock => '2.32',
    g_rw_lock_reader_trylock => '2.32',
    g_rw_lock_reader_lock => '2.32',
    g_rw_lock_init => '2.32',
    g_rec_mutex_unlock => '2.32',
    g_rec_mutex_trylock => '2.32',
    g_rec_mutex_lock => '2.32',
    g_rec_mutex_init => '2.32',
    g_queue_free_full => '2.32',
    g_private_replace => '2.32',
    g_mutex_init => '2.32',
    g_mapped_file_new_from_fd => '2.32',
    g_main_context_ref_thread_default => '2.32',
    g_key_file_unref => '2.32',
    g_key_file_ref => '2.32',
    g_hash_table_contains => '2.32',
    g_hash_table_add => '2.32',
    g_environ_unsetenv => '2.32',
    g_environ_setenv => '2.32',
    g_environ_getenv => '2.32',
    g_cond_wait_until => '2.32',
    g_cond_init => '2.32',
    g_cond_clear => '2.32',
    g_bytes_unref_to_data => '2.32',
    g_bytes_unref_to_array => '2.32',
    g_bytes_unref => '2.32',
    g_bytes_ref => '2.32',
    g_bytes_new_with_free_func => '2.32',
    g_bytes_new_take => '2.32',
    g_bytes_new_static => '2.32',
    g_bytes_new_from_bytes => '2.32',
    g_bytes_new => '2.32',
    g_bytes_hash => '2.32',
    g_bytes_get_size => '2.32',
    g_bytes_get_data => '2.32',
    g_bytes_equal => '2.32',
    g_bytes_compare => '2.32',
    g_byte_array_new_take => '2.32',
    g_byte_array_free_to_bytes => '2.32',
    g_array_set_clear_func => '2.32',
    G_UNICHAR_MAX_DECOMPOSITION_LENGTH => '2.32',
    G_UNAVAILABLE => '2.32',
    G_SOURCE_REMOVE => '2.32',
    G_SOURCE_CONTINUE => '2.32',
    G_PRIVATE_INIT => '2.32',
    G_GNUC_END_IGNORE_DEPRECATIONS => '2.32',
    G_GNUC_BEGIN_IGNORE_DEPRECATIONS => '2.32',
    G_DEPRECATED_FOR => '2.32',
    G_DEPRECATED => '2.32',
    GRecMutex => '2.32',
    GRWLock => '2.32',
    GLIB_VERSION_MIN_REQUIRED => '2.32',
    GLIB_VERSION_MAX_ALLOWED => '2.32',
    GLIB_VERSION_2_32 => '2.32',
    GLIB_VERSION_2_30 => '2.32',
    GLIB_VERSION_2_28 => '2.32',
    GLIB_VERSION_2_26 => '2.32',
    GBytes => '2.32',

    g_variant_new_objv => '2.30',
    g_variant_get_objv => '2.30',
    g_variant_dup_objv => '2.30',
    g_utf8_substring => '2.30',
    g_unix_signal_source_new => '2.30',
    g_unix_signal_add_full => '2.30',
    g_unix_signal_add => '2.30',
    g_unix_set_fd_nonblocking => '2.30',
    g_unix_open_pipe => '2.30',
    g_unicode_script_to_iso15924 => '2.30',
    g_unicode_script_from_iso15924 => '2.30',
    g_unichar_fully_decompose => '2.30',
    g_unichar_decompose => '2.30',
    g_unichar_compose => '2.30',
    g_test_fail => '2.30',
    g_regex_escape_nul => '2.30',
    g_ptr_array_new_full => '2.30',
    g_pointer_bit_unlock => '2.30',
    g_pointer_bit_trylock => '2.30',
    g_pointer_bit_lock => '2.30',
    g_mkdtemp_full => '2.30',
    g_mkdtemp => '2.30',
    g_match_info_unref => '2.30',
    g_match_info_ref => '2.30',
    g_hmac_update => '2.30',
    g_hmac_unref => '2.30',
    g_hmac_ref => '2.30',
    g_hmac_new => '2.30',
    g_hmac_get_string => '2.30',
    g_hmac_get_digest => '2.30',
    g_hmac_copy => '2.30',
    g_hash_table_iter_replace => '2.30',
    g_format_size_full => '2.30',
    g_format_size => '2.30',
    g_dir_make_tmp => '2.30',
    g_compute_hmac_for_string => '2.30',
    g_compute_hmac_for_data => '2.30',
    g_atomic_pointer_xor => '2.30',
    g_atomic_pointer_or => '2.30',
    g_atomic_pointer_and => '2.30',
    g_atomic_pointer_add => '2.30',
    g_atomic_int_xor => '2.30',
    g_atomic_int_or => '2.30',
    g_atomic_int_and => '2.30',
    G_VARIANT_TYPE_VARDICT => '2.30',
    G_STATIC_ASSERT_EXPR => '2.30',
    GHmac => '2.30',

    g_variant_lookup_value => '2.28',
    g_variant_lookup => '2.28',
    g_source_remove_child_source => '2.28',
    g_source_get_time => '2.28',
    g_source_add_child_source => '2.28',
    g_slist_free_full => '2.28',
    g_sequence_lookup_iter => '2.28',
    g_sequence_lookup => '2.28',
    g_main_context_invoke_full => '2.28',
    g_main_context_invoke => '2.28',
    g_list_free_full => '2.28',
    g_get_user_runtime_dir => '2.28',
    g_get_real_time => '2.28',
    g_get_monotonic_time => '2.28',
    g_get_locale_variants => '2.28',
    g_get_environ => '2.28',
    GTestFunc => '2.28',
    GTestFixtureFunc => '2.28',
    GTestDataFunc => '2.28',

    g_variant_new_bytestring_array => '2.26',
    g_variant_new_bytestring => '2.26',
    g_variant_is_floating => '2.26',
    g_variant_get_bytestring_array => '2.26',
    g_variant_get_bytestring => '2.26',
    g_variant_dup_bytestring_array => '2.26',
    g_variant_dup_bytestring => '2.26',
    g_variant_compare => '2.26',
    g_variant_builder_add_parsed => '2.26',
    g_time_zone_unref => '2.26',
    g_time_zone_ref => '2.26',
    g_time_zone_new_utc => '2.26',
    g_time_zone_new_local => '2.26',
    g_time_zone_new => '2.26',
    g_time_zone_is_dst => '2.26',
    g_time_zone_get_offset => '2.26',
    g_time_zone_get_abbreviation => '2.26',
    g_time_zone_find_interval => '2.26',
    g_time_zone_adjust_time => '2.26',
    g_source_set_name_by_id => '2.26',
    g_source_set_name => '2.26',
    g_source_get_name => '2.26',
    g_regex_get_match_flags => '2.26',
    g_regex_get_compile_flags => '2.26',
    g_key_file_set_uint64 => '2.26',
    g_key_file_set_int64 => '2.26',
    g_key_file_get_uint64 => '2.26',
    g_key_file_get_int64 => '2.26',
    g_dcgettext => '2.26',
    g_date_time_unref => '2.26',
    g_date_time_to_utc => '2.26',
    g_date_time_to_unix => '2.26',
    g_date_time_to_timezone => '2.26',
    g_date_time_to_timeval => '2.26',
    g_date_time_to_local => '2.26',
    g_date_time_ref => '2.26',
    g_date_time_new_utc => '2.26',
    g_date_time_new_now_utc => '2.26',
    g_date_time_new_now_local => '2.26',
    g_date_time_new_now => '2.26',
    g_date_time_new_local => '2.26',
    g_date_time_new_from_unix_utc => '2.26',
    g_date_time_new_from_unix_local => '2.26',
    g_date_time_new_from_timeval_utc => '2.26',
    g_date_time_new_from_timeval_local => '2.26',
    g_date_time_new => '2.26',
    g_date_time_is_daylight_savings => '2.26',
    g_date_time_hash => '2.26',
    g_date_time_get_ymd => '2.26',
    g_date_time_get_year => '2.26',
    g_date_time_get_week_of_year => '2.26',
    g_date_time_get_week_numbering_year => '2.26',
    g_date_time_get_utc_offset => '2.26',
    g_date_time_get_timezone_abbreviation => '2.26',
    g_date_time_get_seconds => '2.26',
    g_date_time_get_second => '2.26',
    g_date_time_get_month => '2.26',
    g_date_time_get_minute => '2.26',
    g_date_time_get_microsecond => '2.26',
    g_date_time_get_hour => '2.26',
    g_date_time_get_day_of_year => '2.26',
    g_date_time_get_day_of_week => '2.26',
    g_date_time_get_day_of_month => '2.26',
    g_date_time_format => '2.26',
    g_date_time_equal => '2.26',
    g_date_time_difference => '2.26',
    g_date_time_compare => '2.26',
    g_date_time_add_years => '2.26',
    g_date_time_add_weeks => '2.26',
    g_date_time_add_seconds => '2.26',
    g_date_time_add_months => '2.26',
    g_date_time_add_minutes => '2.26',
    g_date_time_add_hours => '2.26',
    g_date_time_add_full => '2.26',
    g_date_time_add_days => '2.26',
    g_date_time_add => '2.26',
    G_TIME_SPAN_SECOND => '2.26',
    G_TIME_SPAN_MINUTE => '2.26',
    G_TIME_SPAN_MILLISECOND => '2.26',
    G_TIME_SPAN_HOUR => '2.26',
    G_TIME_SPAN_DAY => '2.26',
    G_GNUC_DEPRECATED_FOR => '2.26',
    GTimeZone => '2.26',
    GTimeSpan => '2.26',

    g_variant_unref => '2.24',
    g_variant_type_string_scan => '2.24',
    g_variant_type_new => '2.24',
    g_variant_store => '2.24',
    g_variant_ref_sink => '2.24',
    g_variant_ref => '2.24',
    g_variant_print_string => '2.24',
    g_variant_print => '2.24',
    g_variant_new_variant => '2.24',
    g_variant_new_va => '2.24',
    g_variant_new_uint64 => '2.24',
    g_variant_new_uint32 => '2.24',
    g_variant_new_uint16 => '2.24',
    g_variant_new_tuple => '2.24',
    g_variant_new_strv => '2.24',
    g_variant_new_string => '2.24',
    g_variant_new_signature => '2.24',
    g_variant_new_object_path => '2.24',
    g_variant_new_maybe => '2.24',
    g_variant_new_int64 => '2.24',
    g_variant_new_int32 => '2.24',
    g_variant_new_int16 => '2.24',
    g_variant_new_handle => '2.24',
    g_variant_new_from_data => '2.24',
    g_variant_new_double => '2.24',
    g_variant_new_dict_entry => '2.24',
    g_variant_new_byte => '2.24',
    g_variant_new_boolean => '2.24',
    g_variant_new_array => '2.24',
    g_variant_new => '2.24',
    g_variant_n_children => '2.24',
    g_variant_iter_next_value => '2.24',
    g_variant_iter_next => '2.24',
    g_variant_iter_new => '2.24',
    g_variant_iter_n_children => '2.24',
    g_variant_iter_loop => '2.24',
    g_variant_iter_init => '2.24',
    g_variant_iter_free => '2.24',
    g_variant_iter_copy => '2.24',
    g_variant_is_signature => '2.24',
    g_variant_is_of_type => '2.24',
    g_variant_is_object_path => '2.24',
    g_variant_is_normal_form => '2.24',
    g_variant_is_container => '2.24',
    g_variant_hash => '2.24',
    g_variant_get_variant => '2.24',
    g_variant_get_va => '2.24',
    g_variant_get_uint64 => '2.24',
    g_variant_get_uint32 => '2.24',
    g_variant_get_uint16 => '2.24',
    g_variant_get_type_string => '2.24',
    g_variant_get_type => '2.24',
    g_variant_get_strv => '2.24',
    g_variant_get_string => '2.24',
    g_variant_get_size => '2.24',
    g_variant_get_normal_form => '2.24',
    g_variant_get_maybe => '2.24',
    g_variant_get_int64 => '2.24',
    g_variant_get_int32 => '2.24',
    g_variant_get_int16 => '2.24',
    g_variant_get_handle => '2.24',
    g_variant_get_fixed_array => '2.24',
    g_variant_get_double => '2.24',
    g_variant_get_data => '2.24',
    g_variant_get_child_value => '2.24',
    g_variant_get_child => '2.24',
    g_variant_get_byte => '2.24',
    g_variant_get_boolean => '2.24',
    g_variant_get => '2.24',
    g_variant_equal => '2.24',
    g_variant_dup_strv => '2.24',
    g_variant_dup_string => '2.24',
    g_variant_classify => '2.24',
    g_variant_byteswap => '2.24',
    g_variant_builder_unref => '2.24',
    g_variant_builder_ref => '2.24',
    g_variant_builder_open => '2.24',
    g_variant_builder_new => '2.24',
    g_variant_builder_init => '2.24',
    g_variant_builder_end => '2.24',
    g_variant_builder_close => '2.24',
    g_variant_builder_clear => '2.24',
    g_variant_builder_add_value => '2.24',
    g_variant_builder_add => '2.24',
    g_try_realloc_n => '2.24',
    g_try_malloc_n => '2.24',
    g_try_malloc0_n => '2.24',
    g_realloc_n => '2.24',
    g_malloc_n => '2.24',
    g_malloc0_n => '2.24',
    g_bit_unlock => '2.24',
    g_bit_trylock => '2.24',
    g_bit_lock => '2.24',
    GVariantClass => '2.24',
    GVariant => '2.24',

    g_tree_unref => '2.22',
    g_tree_ref => '2.22',
    g_test_log_set_fatal_handler => '2.22',
    g_reload_user_special_dirs_cache => '2.22',
    g_ptr_array_unref => '2.22',
    g_ptr_array_set_free_func => '2.22',
    g_ptr_array_ref => '2.22',
    g_ptr_array_new_with_free_func => '2.22',
    g_mkstemp_full => '2.22',
    g_mapped_file_ref => '2.22',
    g_main_context_push_thread_default => '2.22',
    g_main_context_pop_thread_default => '2.22',
    g_main_context_get_thread_default => '2.22',
    g_int64_hash => '2.22',
    g_int64_equal => '2.22',
    g_hostname_to_unicode => '2.22',
    g_hostname_to_ascii => '2.22',
    g_hostname_is_non_ascii => '2.22',
    g_hostname_is_ip_address => '2.22',
    g_hostname_is_ascii_encoded => '2.22',
    g_error_new_valist => '2.22',
    g_double_hash => '2.22',
    g_double_equal => '2.22',
    g_byte_array_unref => '2.22',
    g_byte_array_ref => '2.22',
    g_array_unref => '2.22',
    g_array_ref => '2.22',
    g_array_get_element_size => '2.22',
    G_GUINTPTR_FORMAT => '2.22',
    G_GINTPTR_MODIFIER => '2.22',
    G_GINTPTR_FORMAT => '2.22',
    GTestLogFatalFunc => '2.22',

    g_thread_get_initialized => '2.20',
    g_poll => '2.20',
    g_base64_decode_inplace => '2.20',
    g_assert_no_error => '2.20',
    g_assert_error => '2.20',
    G_STATIC_ASSERT => '2.20',
    G_PASTE => '2.20',
    G_GOFFSET_MODIFIER => '2.20',
    G_GOFFSET_FORMAT => '2.20',
    G_GOFFSET_CONSTANT => '2.20',

    guintptr => '2.18',
    gintptr => '2.18',
    g_utime => '2.18',
    g_set_error_literal => '2.18',
    g_markup_parse_context_push => '2.18',
    g_markup_parse_context_pop => '2.18',
    g_markup_parse_context_get_user_data => '2.18',
    g_dpgettext2 => '2.18',
    g_dngettext => '2.18',
    g_dgettext => '2.18',
    g_checksum_reset => '2.18',
    NC_ => '2.18',
    G_GNUC_ALLOC_SIZE2 => '2.18',
    G_GNUC_ALLOC_SIZE => '2.18',

    g_win32_get_package_installation_directory_of_module => '2.16',
    g_warn_if_reached => '2.16',
    g_warn_if_fail => '2.16',
    g_uri_unescape_string => '2.16',
    g_uri_unescape_segment => '2.16',
    g_uri_parse_scheme => '2.16',
    g_uri_escape_string => '2.16',
    g_test_trap_reached_timeout => '2.16',
    g_test_trap_has_passed => '2.16',
    g_test_trap_fork => '2.16',
    g_test_trap_assert_stdout_unmatched => '2.16',
    g_test_trap_assert_stdout => '2.16',
    g_test_trap_assert_stderr_unmatched => '2.16',
    g_test_trap_assert_stderr => '2.16',
    g_test_trap_assert_passed => '2.16',
    g_test_trap_assert_failed => '2.16',
    g_test_timer_start => '2.16',
    g_test_timer_last => '2.16',
    g_test_timer_elapsed => '2.16',
    g_test_suite_add_suite => '2.16',
    g_test_suite_add => '2.16',
    g_test_run_suite => '2.16',
    g_test_run => '2.16',
    g_test_rand_int_range => '2.16',
    g_test_rand_int => '2.16',
    g_test_rand_double_range => '2.16',
    g_test_rand_double => '2.16',
    g_test_rand_bit => '2.16',
    g_test_queue_unref => '2.16',
    g_test_queue_free => '2.16',
    g_test_queue_destroy => '2.16',
    g_test_minimized_result => '2.16',
    g_test_message => '2.16',
    g_test_maximized_result => '2.16',
    g_test_init => '2.16',
    g_test_get_root => '2.16',
    g_test_create_suite => '2.16',
    g_test_create_case => '2.16',
    g_test_bug_base => '2.16',
    g_test_bug => '2.16',
    g_test_add_func => '2.16',
    g_test_add_data_func => '2.16',
    g_test_add => '2.16',
    g_string_append_uri_escaped => '2.16',
    g_strcmp0 => '2.16',
    g_propagate_prefixed_error => '2.16',
    g_prefix_error => '2.16',
    g_markup_parse_context_get_element_stack => '2.16',
    g_markup_collect_attributes => '2.16',
    g_hash_table_iter_steal => '2.16',
    g_hash_table_iter_remove => '2.16',
    g_hash_table_iter_next => '2.16',
    g_hash_table_iter_init => '2.16',
    g_hash_table_iter_get_hash_table => '2.16',
    g_format_size_for_display => '2.16',
    g_dpgettext => '2.16',
    g_compute_checksum_for_string => '2.16',
    g_compute_checksum_for_data => '2.16',
    g_checksum_update => '2.16',
    g_checksum_type_get_length => '2.16',
    g_checksum_new => '2.16',
    g_checksum_get_string => '2.16',
    g_checksum_get_digest => '2.16',
    g_checksum_free => '2.16',
    g_checksum_copy => '2.16',
    g_async_queue_new_full => '2.16',
    g_assert_cmpuint => '2.16',
    g_assert_cmpstr => '2.16',
    g_assert_cmpint => '2.16',
    g_assert_cmphex => '2.16',
    g_assert_cmpfloat => '2.16',
    GChecksumType => '2.16',
    GChecksum => '2.16',
    C_ => '2.16',

    goffset => '2.14',
    g_unichar_iszerowidth => '2.14',
    g_unichar_ismark => '2.14',
    g_unichar_get_script => '2.14',
    g_unichar_combining_class => '2.14',
    g_timeout_source_new_seconds => '2.14',
    g_timeout_add_seconds_full => '2.14',
    g_timeout_add_seconds => '2.14',
    g_string_vprintf => '2.14',
    g_string_overwrite_len => '2.14',
    g_string_overwrite => '2.14',
    g_string_chunk_clear => '2.14',
    g_string_append_vprintf => '2.14',
    g_slice_dup => '2.14',
    g_slice_copy => '2.14',
    g_sequence_swap => '2.14',
    g_sequence_sort_iter => '2.14',
    g_sequence_sort_changed_iter => '2.14',
    g_sequence_sort_changed => '2.14',
    g_sequence_sort => '2.14',
    g_sequence_set => '2.14',
    g_sequence_search_iter => '2.14',
    g_sequence_search => '2.14',
    g_sequence_remove_range => '2.14',
    g_sequence_remove => '2.14',
    g_sequence_range_get_midpoint => '2.14',
    g_sequence_prepend => '2.14',
    g_sequence_new => '2.14',
    g_sequence_move_range => '2.14',
    g_sequence_move => '2.14',
    g_sequence_iter_prev => '2.14',
    g_sequence_iter_next => '2.14',
    g_sequence_iter_move => '2.14',
    g_sequence_iter_is_end => '2.14',
    g_sequence_iter_is_begin => '2.14',
    g_sequence_iter_get_sequence => '2.14',
    g_sequence_iter_get_position => '2.14',
    g_sequence_iter_compare => '2.14',
    g_sequence_insert_sorted_iter => '2.14',
    g_sequence_insert_sorted => '2.14',
    g_sequence_insert_before => '2.14',
    g_sequence_get_length => '2.14',
    g_sequence_get_iter_at_pos => '2.14',
    g_sequence_get_end_iter => '2.14',
    g_sequence_get_begin_iter => '2.14',
    g_sequence_get => '2.14',
    g_sequence_free => '2.14',
    g_sequence_foreach_range => '2.14',
    g_sequence_foreach => '2.14',
    g_sequence_append => '2.14',
    g_regex_unref => '2.14',
    g_regex_split_simple => '2.14',
    g_regex_split_full => '2.14',
    g_regex_split => '2.14',
    g_regex_replace_literal => '2.14',
    g_regex_replace_eval => '2.14',
    g_regex_replace => '2.14',
    g_regex_ref => '2.14',
    g_regex_new => '2.14',
    g_regex_match_simple => '2.14',
    g_regex_match_full => '2.14',
    g_regex_match_all_full => '2.14',
    g_regex_match_all => '2.14',
    g_regex_match => '2.14',
    g_regex_get_string_number => '2.14',
    g_regex_get_pattern => '2.14',
    g_regex_get_max_backref => '2.14',
    g_regex_get_capture_count => '2.14',
    g_regex_escape_string => '2.14',
    g_regex_check_replacement => '2.14',
    g_queue_init => '2.14',
    g_queue_clear => '2.14',
    g_option_context_get_help => '2.14',
    g_once_init_leave => '2.14',
    g_once_init_enter => '2.14',
    g_match_info_next => '2.14',
    g_match_info_matches => '2.14',
    g_match_info_is_partial_match => '2.14',
    g_match_info_get_string => '2.14',
    g_match_info_get_regex => '2.14',
    g_match_info_get_match_count => '2.14',
    g_match_info_free => '2.14',
    g_match_info_fetch_pos => '2.14',
    g_match_info_fetch_named_pos => '2.14',
    g_match_info_fetch_named => '2.14',
    g_match_info_fetch_all => '2.14',
    g_match_info_fetch => '2.14',
    g_match_info_expand_references => '2.14',
    g_key_file_load_from_dirs => '2.14',
    g_hash_table_get_values => '2.14',
    g_hash_table_get_keys => '2.14',
    g_get_user_special_dir => '2.14',
    G_REGEX_ERROR => '2.14',
    G_QUEUE_INIT => '2.14',
    G_MINSSIZE => '2.14',
    G_MAXSSIZE => '2.14',
    G_KEY_FILE_DESKTOP_TYPE_LINK => '2.14',
    G_KEY_FILE_DESKTOP_TYPE_DIRECTORY => '2.14',
    G_KEY_FILE_DESKTOP_TYPE_APPLICATION => '2.14',
    G_KEY_FILE_DESKTOP_KEY_VERSION => '2.14',
    G_KEY_FILE_DESKTOP_KEY_URL => '2.14',
    G_KEY_FILE_DESKTOP_KEY_TYPE => '2.14',
    G_KEY_FILE_DESKTOP_KEY_TRY_EXEC => '2.14',
    G_KEY_FILE_DESKTOP_KEY_TERMINAL => '2.14',
    G_KEY_FILE_DESKTOP_KEY_STARTUP_WM_CLASS => '2.14',
    G_KEY_FILE_DESKTOP_KEY_STARTUP_NOTIFY => '2.14',
    G_KEY_FILE_DESKTOP_KEY_PATH => '2.14',
    G_KEY_FILE_DESKTOP_KEY_ONLY_SHOW_IN => '2.14',
    G_KEY_FILE_DESKTOP_KEY_NO_DISPLAY => '2.14',
    G_KEY_FILE_DESKTOP_KEY_NOT_SHOW_IN => '2.14',
    G_KEY_FILE_DESKTOP_KEY_NAME => '2.14',
    G_KEY_FILE_DESKTOP_KEY_MIME_TYPE => '2.14',
    G_KEY_FILE_DESKTOP_KEY_ICON => '2.14',
    G_KEY_FILE_DESKTOP_KEY_HIDDEN => '2.14',
    G_KEY_FILE_DESKTOP_KEY_GENERIC_NAME => '2.14',
    G_KEY_FILE_DESKTOP_KEY_EXEC => '2.14',
    G_KEY_FILE_DESKTOP_KEY_COMMENT => '2.14',
    G_KEY_FILE_DESKTOP_KEY_CATEGORIES => '2.14',
    G_KEY_FILE_DESKTOP_GROUP => '2.14',
    G_GNUC_MAY_ALIAS => '2.14',
    GUserDirectory => '2.14',
    GRegexMatchFlags => '2.14',
    GRegexEvalCallback => '2.14',
    GRegexError => '2.14',
    GRegexCompileFlags => '2.14',
    GRegex => '2.14',

    g_unichar_iswide_cjk => '2.12',
    g_time_val_to_iso8601 => '2.12',
    g_time_val_from_iso8601 => '2.12',
    g_source_set_funcs => '2.12',
    g_source_is_destroyed => '2.12',
    g_option_context_set_translation_domain => '2.12',
    g_option_context_set_translate_func => '2.12',
    g_option_context_set_summary => '2.12',
    g_option_context_set_description => '2.12',
    g_option_context_get_summary => '2.12',
    g_option_context_get_description => '2.12',
    g_main_current_source => '2.12',
    g_key_file_set_double_list => '2.12',
    g_key_file_set_double => '2.12',
    g_key_file_get_double_list => '2.12',
    g_key_file_get_double => '2.12',
    g_hash_table_steal_all => '2.12',
    g_hash_table_remove_all => '2.12',
    g_bookmark_file_to_file => '2.12',
    g_bookmark_file_to_data => '2.12',
    g_bookmark_file_set_visited => '2.12',
    g_bookmark_file_set_title => '2.12',
    g_bookmark_file_set_modified => '2.12',
    g_bookmark_file_set_mime_type => '2.12',
    g_bookmark_file_set_is_private => '2.12',
    g_bookmark_file_set_icon => '2.12',
    g_bookmark_file_set_groups => '2.12',
    g_bookmark_file_set_description => '2.12',
    g_bookmark_file_set_app_info => '2.12',
    g_bookmark_file_set_added => '2.12',
    g_bookmark_file_remove_item => '2.12',
    g_bookmark_file_remove_group => '2.12',
    g_bookmark_file_remove_application => '2.12',
    g_bookmark_file_new => '2.12',
    g_bookmark_file_move_item => '2.12',
    g_bookmark_file_load_from_file => '2.12',
    g_bookmark_file_load_from_data_dirs => '2.12',
    g_bookmark_file_load_from_data => '2.12',
    g_bookmark_file_has_item => '2.12',
    g_bookmark_file_has_group => '2.12',
    g_bookmark_file_has_application => '2.12',
    g_bookmark_file_get_visited => '2.12',
    g_bookmark_file_get_uris => '2.12',
    g_bookmark_file_get_title => '2.12',
    g_bookmark_file_get_size => '2.12',
    g_bookmark_file_get_modified => '2.12',
    g_bookmark_file_get_mime_type => '2.12',
    g_bookmark_file_get_is_private => '2.12',
    g_bookmark_file_get_icon => '2.12',
    g_bookmark_file_get_groups => '2.12',
    g_bookmark_file_get_description => '2.12',
    g_bookmark_file_get_applications => '2.12',
    g_bookmark_file_get_app_info => '2.12',
    g_bookmark_file_get_added => '2.12',
    g_bookmark_file_free => '2.12',
    g_bookmark_file_add_group => '2.12',
    g_bookmark_file_add_application => '2.12',
    g_base64_encode_step => '2.12',
    g_base64_encode_close => '2.12',
    g_base64_encode => '2.12',
    g_base64_decode_step => '2.12',
    g_base64_decode => '2.12',
    g_ascii_strtoll => '2.12',

    g_thread_pool_set_sort_function => '2.10',
    g_thread_pool_set_max_idle_time => '2.10',
    g_thread_pool_get_max_idle_time => '2.10',
    g_thread_foreach => '2.10',
    g_slist_insert_sorted_with_data => '2.10',
    g_slist_free1 => '2.10',
    g_slice_new0 => '2.10',
    g_slice_new => '2.10',
    g_slice_free_chain_with_offset => '2.10',
    g_slice_free_chain => '2.10',
    g_slice_free1 => '2.10',
    g_slice_free => '2.10',
    g_slice_alloc0 => '2.10',
    g_slice_alloc => '2.10',
    g_main_context_is_owner => '2.10',
    g_list_insert_sorted_with_data => '2.10',
    g_intern_string => '2.10',
    g_intern_static_string => '2.10',
    g_hash_table_unref => '2.10',
    g_hash_table_ref => '2.10',
    g_date_set_time_val => '2.10',
    g_date_set_time_t => '2.10',
    g_async_queue_sort_unlocked => '2.10',
    g_async_queue_sort => '2.10',
    g_async_queue_push_sorted_unlocked => '2.10',
    g_async_queue_push_sorted => '2.10',
    G_GUINT64_CONSTANT => '2.10',
    G_GNUC_WARN_UNUSED_RESULT => '2.10',

    g_win32_locale_filename_from_utf8 => '2.8',
    g_utf8_collate_key_for_filename => '2.8',
    g_try_renew => '2.8',
    g_try_new0 => '2.8',
    g_try_new => '2.8',
    g_try_malloc0 => '2.8',
    g_mkdir_with_parents => '2.8',
    g_mapped_file_new => '2.8',
    g_mapped_file_get_length => '2.8',
    g_mapped_file_get_contents => '2.8',
    g_mapped_file_free => '2.8',
    g_listenv => '2.8',
    g_get_host_name => '2.8',
    g_file_set_contents => '2.8',
    g_datalist_unset_flags => '2.8',
    g_datalist_set_flags => '2.8',
    g_datalist_get_flags => '2.8',
    g_creat => '2.8',
    g_chmod => '2.8',
    g_chdir => '2.8',
    g_build_pathv => '2.8',
    g_build_filenamev => '2.8',
    g_access => '2.8',
    G_GNUC_NULL_TERMINATED => '2.8',

    glib_check_version => '2.6',
    g_uri_list_extract_uris => '2.6',
    g_unlink => '2.6',
    g_strv_length => '2.6',
    g_stat => '2.6',
    g_rmdir => '2.6',
    g_rename => '2.6',
    g_remove => '2.6',
    g_option_group_set_translation_domain => '2.6',
    g_option_group_set_translate_func => '2.6',
    g_option_group_set_parse_hooks => '2.6',
    g_option_group_set_error_hook => '2.6',
    g_option_group_new => '2.6',
    g_option_group_free => '2.6',
    g_option_group_add_entries => '2.6',
    g_option_context_set_main_group => '2.6',
    g_option_context_set_ignore_unknown_options => '2.6',
    g_option_context_set_help_enabled => '2.6',
    g_option_context_parse => '2.6',
    g_option_context_new => '2.6',
    g_option_context_get_main_group => '2.6',
    g_option_context_get_ignore_unknown_options => '2.6',
    g_option_context_get_help_enabled => '2.6',
    g_option_context_free => '2.6',
    g_option_context_add_main_entries => '2.6',
    g_option_context_add_group => '2.6',
    g_open => '2.6',
    g_mkdir => '2.6',
    g_lstat => '2.6',
    g_log_set_default_handler => '2.6',
    g_key_file_to_data => '2.6',
    g_key_file_set_value => '2.6',
    g_key_file_set_string_list => '2.6',
    g_key_file_set_string => '2.6',
    g_key_file_set_locale_string_list => '2.6',
    g_key_file_set_locale_string => '2.6',
    g_key_file_set_list_separator => '2.6',
    g_key_file_set_integer_list => '2.6',
    g_key_file_set_integer => '2.6',
    g_key_file_set_comment => '2.6',
    g_key_file_set_boolean_list => '2.6',
    g_key_file_set_boolean => '2.6',
    g_key_file_remove_key => '2.6',
    g_key_file_remove_group => '2.6',
    g_key_file_remove_comment => '2.6',
    g_key_file_new => '2.6',
    g_key_file_load_from_file => '2.6',
    g_key_file_load_from_data_dirs => '2.6',
    g_key_file_load_from_data => '2.6',
    g_key_file_has_key => '2.6',
    g_key_file_has_group => '2.6',
    g_key_file_get_value => '2.6',
    g_key_file_get_string_list => '2.6',
    g_key_file_get_string => '2.6',
    g_key_file_get_start_group => '2.6',
    g_key_file_get_locale_string_list => '2.6',
    g_key_file_get_locale_string => '2.6',
    g_key_file_get_keys => '2.6',
    g_key_file_get_integer_list => '2.6',
    g_key_file_get_integer => '2.6',
    g_key_file_get_groups => '2.6',
    g_key_file_get_comment => '2.6',
    g_key_file_get_boolean_list => '2.6',
    g_key_file_get_boolean => '2.6',
    g_key_file_free => '2.6',
    g_get_user_data_dir => '2.6',
    g_get_user_config_dir => '2.6',
    g_get_user_cache_dir => '2.6',
    g_get_system_data_dirs => '2.6',
    g_get_system_config_dirs => '2.6',
    g_get_language_names => '2.6',
    g_get_filename_charsets => '2.6',
    g_freopen => '2.6',
    g_fopen => '2.6',
    g_filename_display_name => '2.6',
    g_filename_display_basename => '2.6',
    g_debug => '2.6',
    g_date_get_iso8601_week_of_year => '2.6',
    G_WIN32_IS_NT_BASED => '2.6',
    G_WIN32_HAVE_WIDECHAR_API => '2.6',
    G_OPTION_REMAINING => '2.6',
    G_IS_DIR_SEPARATOR => '2.6',
    G_GSSIZE_MODIFIER => '2.6',
    G_GSSIZE_FORMAT => '2.6',
    G_GSIZE_MODIFIER => '2.6',
    G_GSIZE_FORMAT => '2.6',
    G_GNUC_MALLOC => '2.6',
    G_GNUC_INTERNAL => '2.6',

    g_vasprintf => '2.4',
    g_unsetenv => '2.4',
    g_unichar_get_mirror_char => '2.4',
    g_timer_continue => '2.4',
    g_strsplit_set => '2.4',
    g_strip_context => '2.4',
    g_string_chunk_insert_len => '2.4',
    g_setenv => '2.4',
    g_rand_set_seed_array => '2.4',
    g_rand_new_with_seed_array => '2.4',
    g_rand_copy => '2.4',
    g_queue_unlink => '2.4',
    g_queue_sort => '2.4',
    g_queue_reverse => '2.4',
    g_queue_remove_all => '2.4',
    g_queue_remove => '2.4',
    g_queue_push_nth_link => '2.4',
    g_queue_push_nth => '2.4',
    g_queue_pop_nth_link => '2.4',
    g_queue_pop_nth => '2.4',
    g_queue_peek_tail_link => '2.4',
    g_queue_peek_nth_link => '2.4',
    g_queue_peek_nth => '2.4',
    g_queue_peek_head_link => '2.4',
    g_queue_link_index => '2.4',
    g_queue_insert_sorted => '2.4',
    g_queue_insert_before => '2.4',
    g_queue_insert_after => '2.4',
    g_queue_index => '2.4',
    g_queue_get_length => '2.4',
    g_queue_foreach => '2.4',
    g_queue_find_custom => '2.4',
    g_queue_find => '2.4',
    g_queue_delete_link => '2.4',
    g_queue_copy => '2.4',
    g_ptr_array_remove_range => '2.4',
    g_ptr_array_foreach => '2.4',
    g_once => '2.4',
    g_node_copy_deep => '2.4',
    g_markup_vprintf_escaped => '2.4',
    g_markup_printf_escaped => '2.4',
    g_hash_table_find => '2.4',
    g_file_read_link => '2.4',
    g_completion_complete_utf8 => '2.4',
    g_child_watch_source_new => '2.4',
    g_child_watch_add_full => '2.4',
    g_child_watch_add => '2.4',
    g_byte_array_remove_range => '2.4',
    g_atomic_pointer_set => '2.4',
    g_atomic_pointer_get => '2.4',
    g_atomic_pointer_compare_and_exchange => '2.4',
    g_atomic_int_set => '2.4',
    g_atomic_int_inc => '2.4',
    g_atomic_int_get => '2.4',
    g_atomic_int_exchange_and_add => '2.4',
    g_atomic_int_dec_and_test => '2.4',
    g_atomic_int_compare_and_exchange => '2.4',
    g_atomic_int_add => '2.4',
    g_array_remove_range => '2.4',
    Q_ => '2.4',
    N_ => '2.4',
    G_STRFUNC => '2.4',
    G_ONCE_INIT => '2.4',
    G_MININT8 => '2.4',
    G_MININT32 => '2.4',
    G_MININT16 => '2.4',
    G_MAXUINT8 => '2.4',
    G_MAXUINT32 => '2.4',
    G_MAXUINT16 => '2.4',
    G_MAXSIZE => '2.4',
    G_MAXINT8 => '2.4',
    G_MAXINT32 => '2.4',
    G_MAXINT16 => '2.4',
    G_GINT64_MODIFIER => '2.4',
    G_GINT32_MODIFIER => '2.4',
    G_GINT16_MODIFIER => '2.4',
    GOnceStatus => '2.4',
    GOnce => '2.4',
    GCopyFunc => '2.4',

    g_vsprintf => '2.2',
    g_vprintf => '2.2',
    g_vfprintf => '2.2',
    g_utf8_strreverse => '2.2',
    g_str_has_suffix => '2.2',
    g_str_has_prefix => '2.2',
    g_sprintf => '2.2',
    g_set_application_name => '2.2',
    g_printf => '2.2',
    g_markup_parse_context_get_element => '2.2',
    g_get_application_name => '2.2',
    g_fprintf => '2.2',
    g_ascii_strtoull => '2.2',
    G_UNLIKELY => '2.2',
    G_LIKELY => '2.2',
    G_GNUC_DEPRECATED => '2.2',

    gushort => '2.0',
    gunichar2 => '2.0',
    gunichar => '2.0',
    gulong => '2.0',
    guint8 => '2.0',
    guint64 => '2.0',
    guint32 => '2.0',
    guint16 => '2.0',
    guint => '2.0',
    guchar => '2.0',
    gssize => '2.0',
    gsize => '2.0',
    gshort => '2.0',
    gpointer => '2.0',
    glong => '2.0',
    glib_minor_version => '2.0',
    glib_micro_version => '2.0',
    glib_mem_profiler_table => '2.0',
    glib_major_version => '2.0',
    glib_interface_age => '2.0',
    glib_binary_age => '2.0',
    gint8 => '2.0',
    gint64 => '2.0',
    gint32 => '2.0',
    gint16 => '2.0',
    gint => '2.0',
    gfloat => '2.0',
    gdouble => '2.0',
    gconstpointer => '2.0',
    gchar => '2.0',
    gboolean => '2.0',
    g_win32_getlocale => '2.0',
    g_win32_get_windows_version => '2.0',
    g_win32_get_package_installation_subdirectory => '2.0',
    g_win32_get_package_installation_directory => '2.0',
    g_win32_error_message => '2.0',
    g_warning => '2.0',
    g_vsnprintf => '2.0',
    g_variant_type_value => '2.0',
    g_variant_type_string_is_valid => '2.0',
    g_variant_type_peek_string => '2.0',
    g_variant_type_next => '2.0',
    g_variant_type_new_tuple => '2.0',
    g_variant_type_new_maybe => '2.0',
    g_variant_type_new_dict_entry => '2.0',
    g_variant_type_new_array => '2.0',
    g_variant_type_n_items => '2.0',
    g_variant_type_key => '2.0',
    g_variant_type_is_variant => '2.0',
    g_variant_type_is_tuple => '2.0',
    g_variant_type_is_subtype_of => '2.0',
    g_variant_type_is_maybe => '2.0',
    g_variant_type_is_dict_entry => '2.0',
    g_variant_type_is_definite => '2.0',
    g_variant_type_is_container => '2.0',
    g_variant_type_is_basic => '2.0',
    g_variant_type_is_array => '2.0',
    g_variant_type_hash => '2.0',
    g_variant_type_get_string_length => '2.0',
    g_variant_type_free => '2.0',
    g_variant_type_first => '2.0',
    g_variant_type_equal => '2.0',
    g_variant_type_element => '2.0',
    g_variant_type_dup_string => '2.0',
    g_variant_type_copy => '2.0',
    g_variant_take_ref => '2.0',
    g_variant_parse => '2.0',
    g_variant_new_parsed_va => '2.0',
    g_variant_new_parsed => '2.0',
    g_utf8_validate => '2.0',
    g_utf8_to_utf16 => '2.0',
    g_utf8_to_ucs4_fast => '2.0',
    g_utf8_to_ucs4 => '2.0',
    g_utf8_strup => '2.0',
    g_utf8_strrchr => '2.0',
    g_utf8_strncpy => '2.0',
    g_utf8_strlen => '2.0',
    g_utf8_strdown => '2.0',
    g_utf8_strchr => '2.0',
    g_utf8_prev_char => '2.0',
    g_utf8_pointer_to_offset => '2.0',
    g_utf8_offset_to_pointer => '2.0',
    g_utf8_normalize => '2.0',
    g_utf8_next_char => '2.0',
    g_utf8_get_char_validated => '2.0',
    g_utf8_get_char => '2.0',
    g_utf8_find_prev_char => '2.0',
    g_utf8_find_next_char => '2.0',
    g_utf8_collate_key => '2.0',
    g_utf8_collate => '2.0',
    g_utf8_casefold => '2.0',
    g_utf16_to_utf8 => '2.0',
    g_utf16_to_ucs4 => '2.0',
    g_usleep => '2.0',
    g_unicode_canonical_ordering => '2.0',
    g_unicode_canonical_decomposition => '2.0',
    g_unichar_xdigit_value => '2.0',
    g_unichar_validate => '2.0',
    g_unichar_type => '2.0',
    g_unichar_toupper => '2.0',
    g_unichar_totitle => '2.0',
    g_unichar_tolower => '2.0',
    g_unichar_to_utf8 => '2.0',
    g_unichar_isxdigit => '2.0',
    g_unichar_iswide => '2.0',
    g_unichar_isupper => '2.0',
    g_unichar_istitle => '2.0',
    g_unichar_isspace => '2.0',
    g_unichar_ispunct => '2.0',
    g_unichar_isprint => '2.0',
    g_unichar_islower => '2.0',
    g_unichar_isgraph => '2.0',
    g_unichar_isdigit => '2.0',
    g_unichar_isdefined => '2.0',
    g_unichar_iscntrl => '2.0',
    g_unichar_isalpha => '2.0',
    g_unichar_isalnum => '2.0',
    g_unichar_digit_value => '2.0',
    g_unichar_break_type => '2.0',
    g_ucs4_to_utf8 => '2.0',
    g_ucs4_to_utf16 => '2.0',
    g_tuples_index => '2.0',
    g_tuples_destroy => '2.0',
    g_try_realloc => '2.0',
    g_try_malloc => '2.0',
    g_tree_traverse => '2.0',
    g_tree_steal => '2.0',
    g_tree_search => '2.0',
    g_tree_replace => '2.0',
    g_tree_remove => '2.0',
    g_tree_nnodes => '2.0',
    g_tree_new_with_data => '2.0',
    g_tree_new_full => '2.0',
    g_tree_new => '2.0',
    g_tree_lookup_extended => '2.0',
    g_tree_lookup => '2.0',
    g_tree_insert => '2.0',
    g_tree_height => '2.0',
    g_tree_foreach => '2.0',
    g_tree_destroy => '2.0',
    g_trash_stack_push => '2.0',
    g_trash_stack_pop => '2.0',
    g_trash_stack_peek => '2.0',
    g_trash_stack_height => '2.0',
    g_timer_stop => '2.0',
    g_timer_start => '2.0',
    g_timer_reset => '2.0',
    g_timer_new => '2.0',
    g_timer_elapsed => '2.0',
    g_timer_destroy => '2.0',
    g_timeout_source_new => '2.0',
    g_timeout_add_full => '2.0',
    g_timeout_add => '2.0',
    g_time_val_add => '2.0',
    g_thread_yield => '2.0',
    g_thread_supported => '2.0',
    g_thread_set_priority => '2.0',
    g_thread_self => '2.0',
    g_thread_pool_unprocessed => '2.0',
    g_thread_pool_stop_unused_threads => '2.0',
    g_thread_pool_set_max_unused_threads => '2.0',
    g_thread_pool_set_max_threads => '2.0',
    g_thread_pool_push => '2.0',
    g_thread_pool_new => '2.0',
    g_thread_pool_get_num_unused_threads => '2.0',
    g_thread_pool_get_num_threads => '2.0',
    g_thread_pool_get_max_unused_threads => '2.0',
    g_thread_pool_get_max_threads => '2.0',
    g_thread_pool_free => '2.0',
    g_thread_join => '2.0',
    g_thread_init => '2.0',
    g_thread_exit => '2.0',
    g_thread_create_full => '2.0',
    g_thread_create => '2.0',
    g_test_verbose => '2.0',
    g_test_undefined => '2.0',
    g_test_thorough => '2.0',
    g_test_slow => '2.0',
    g_test_quiet => '2.0',
    g_test_quick => '2.0',
    g_test_perf => '2.0',
    g_strup => '2.0',
    g_strtod => '2.0',
    g_strstrip => '2.0',
    g_strstr_len => '2.0',
    g_strsplit => '2.0',
    g_strsignal => '2.0',
    g_strrstr_len => '2.0',
    g_strrstr => '2.0',
    g_strreverse => '2.0',
    g_strnfill => '2.0',
    g_strndup => '2.0',
    g_strncasecmp => '2.0',
    g_strlcpy => '2.0',
    g_strlcat => '2.0',
    g_strjoinv => '2.0',
    g_strjoin => '2.0',
    g_string_up => '2.0',
    g_string_truncate => '2.0',
    g_string_sprintfa => '2.0',
    g_string_sprintf => '2.0',
    g_string_sized_new => '2.0',
    g_string_set_size => '2.0',
    g_string_printf => '2.0',
    g_string_prepend_unichar => '2.0',
    g_string_prepend_len => '2.0',
    g_string_prepend_c => '2.0',
    g_string_prepend => '2.0',
    g_string_new_len => '2.0',
    g_string_new => '2.0',
    g_string_insert_unichar => '2.0',
    g_string_insert_len => '2.0',
    g_string_insert_c => '2.0',
    g_string_insert => '2.0',
    g_string_hash => '2.0',
    g_string_free => '2.0',
    g_string_erase => '2.0',
    g_string_equal => '2.0',
    g_string_down => '2.0',
    g_string_chunk_new => '2.0',
    g_string_chunk_insert_const => '2.0',
    g_string_chunk_insert => '2.0',
    g_string_chunk_free => '2.0',
    g_string_assign => '2.0',
    g_string_ascii_up => '2.0',
    g_string_ascii_down => '2.0',
    g_string_append_unichar => '2.0',
    g_string_append_printf => '2.0',
    g_string_append_len => '2.0',
    g_string_append_c => '2.0',
    g_string_append => '2.0',
    g_strfreev => '2.0',
    g_strescape => '2.0',
    g_strerror => '2.0',
    g_strdupv => '2.0',
    g_strdup_vprintf => '2.0',
    g_strdup_printf => '2.0',
    g_strdup => '2.0',
    g_strdown => '2.0',
    g_strdelimit => '2.0',
    g_strconcat => '2.0',
    g_strcompress => '2.0',
    g_strchug => '2.0',
    g_strchomp => '2.0',
    g_strcasecmp => '2.0',
    g_strcanon => '2.0',
    g_str_hash => '2.0',
    g_str_equal => '2.0',
    g_stpcpy => '2.0',
    g_static_rw_lock_writer_unlock => '2.0',
    g_static_rw_lock_writer_trylock => '2.0',
    g_static_rw_lock_writer_lock => '2.0',
    g_static_rw_lock_reader_unlock => '2.0',
    g_static_rw_lock_reader_trylock => '2.0',
    g_static_rw_lock_reader_lock => '2.0',
    g_static_rw_lock_init => '2.0',
    g_static_rw_lock_free => '2.0',
    g_static_rec_mutex_unlock_full => '2.0',
    g_static_rec_mutex_unlock => '2.0',
    g_static_rec_mutex_trylock => '2.0',
    g_static_rec_mutex_lock_full => '2.0',
    g_static_rec_mutex_lock => '2.0',
    g_static_rec_mutex_init => '2.0',
    g_static_rec_mutex_free => '2.0',
    g_static_private_set => '2.0',
    g_static_private_init => '2.0',
    g_static_private_get => '2.0',
    g_static_private_free => '2.0',
    g_static_mutex_unlock => '2.0',
    g_static_mutex_trylock => '2.0',
    g_static_mutex_lock => '2.0',
    g_static_mutex_init => '2.0',
    g_static_mutex_get_mutex => '2.0',
    g_static_mutex_free => '2.0',
    g_spawn_sync => '2.0',
    g_spawn_command_line_sync => '2.0',
    g_spawn_command_line_async => '2.0',
    g_spawn_close_pid => '2.0',
    g_spawn_async_with_pipes => '2.0',
    g_spawn_async => '2.0',
    g_spaced_primes_closest => '2.0',
    g_source_unref => '2.0',
    g_source_set_priority => '2.0',
    g_source_set_can_recurse => '2.0',
    g_source_set_callback_indirect => '2.0',
    g_source_set_callback => '2.0',
    g_source_remove_poll => '2.0',
    g_source_remove_by_user_data => '2.0',
    g_source_remove_by_funcs_user_data => '2.0',
    g_source_remove => '2.0',
    g_source_ref => '2.0',
    g_source_new => '2.0',
    g_source_get_ready_time => '2.0',
    g_source_get_priority => '2.0',
    g_source_get_id => '2.0',
    g_source_get_current_time => '2.0',
    g_source_get_context => '2.0',
    g_source_get_can_recurse => '2.0',
    g_source_destroy => '2.0',
    g_source_attach => '2.0',
    g_source_add_poll => '2.0',
    g_snprintf => '2.0',
    g_slist_sort_with_data => '2.0',
    g_slist_sort => '2.0',
    g_slist_reverse => '2.0',
    g_slist_remove_link => '2.0',
    g_slist_remove_all => '2.0',
    g_slist_remove => '2.0',
    g_slist_prepend => '2.0',
    g_slist_position => '2.0',
    g_slist_nth_data => '2.0',
    g_slist_nth => '2.0',
    g_slist_next => '2.0',
    g_slist_length => '2.0',
    g_slist_last => '2.0',
    g_slist_insert_sorted => '2.0',
    g_slist_insert_before => '2.0',
    g_slist_insert => '2.0',
    g_slist_index => '2.0',
    g_slist_free_1 => '2.0',
    g_slist_free => '2.0',
    g_slist_foreach => '2.0',
    g_slist_find_custom => '2.0',
    g_slist_find => '2.0',
    g_slist_delete_link => '2.0',
    g_slist_copy => '2.0',
    g_slist_concat => '2.0',
    g_slist_append => '2.0',
    g_slist_alloc => '2.0',
    g_shell_unquote => '2.0',
    g_shell_quote => '2.0',
    g_shell_parse_argv => '2.0',
    g_set_printerr_handler => '2.0',
    g_set_print_handler => '2.0',
    g_set_prgname => '2.0',
    g_set_error => '2.0',
    g_scanner_warn => '2.0',
    g_scanner_unexp_token => '2.0',
    g_scanner_thaw_symbol_table => '2.0',
    g_scanner_sync_file_offset => '2.0',
    g_scanner_set_scope => '2.0',
    g_scanner_scope_remove_symbol => '2.0',
    g_scanner_scope_lookup_symbol => '2.0',
    g_scanner_scope_foreach_symbol => '2.0',
    g_scanner_scope_add_symbol => '2.0',
    g_scanner_remove_symbol => '2.0',
    g_scanner_peek_next_token => '2.0',
    g_scanner_new => '2.0',
    g_scanner_lookup_symbol => '2.0',
    g_scanner_input_text => '2.0',
    g_scanner_input_file => '2.0',
    g_scanner_get_next_token => '2.0',
    g_scanner_freeze_symbol_table => '2.0',
    g_scanner_foreach_symbol => '2.0',
    g_scanner_error => '2.0',
    g_scanner_eof => '2.0',
    g_scanner_destroy => '2.0',
    g_scanner_cur_value => '2.0',
    g_scanner_cur_token => '2.0',
    g_scanner_cur_position => '2.0',
    g_scanner_cur_line => '2.0',
    g_scanner_add_symbol => '2.0',
    g_rw_lock_clear => '2.0',
    g_return_val_if_reached => '2.0',
    g_return_val_if_fail => '2.0',
    g_return_if_reached => '2.0',
    g_return_if_fail => '2.0',
    g_renew => '2.0',
    g_relation_select => '2.0',
    g_relation_print => '2.0',
    g_relation_new => '2.0',
    g_relation_insert => '2.0',
    g_relation_index => '2.0',
    g_relation_exists => '2.0',
    g_relation_destroy => '2.0',
    g_relation_delete => '2.0',
    g_relation_count => '2.0',
    g_rec_mutex_clear => '2.0',
    g_realloc => '2.0',
    g_random_set_seed => '2.0',
    g_random_int_range => '2.0',
    g_random_int => '2.0',
    g_random_double_range => '2.0',
    g_random_double => '2.0',
    g_random_boolean => '2.0',
    g_rand_set_seed => '2.0',
    g_rand_new_with_seed => '2.0',
    g_rand_new => '2.0',
    g_rand_int_range => '2.0',
    g_rand_int => '2.0',
    g_rand_free => '2.0',
    g_rand_double_range => '2.0',
    g_rand_double => '2.0',
    g_rand_boolean => '2.0',
    g_queue_push_tail_link => '2.0',
    g_queue_push_tail => '2.0',
    g_queue_push_head_link => '2.0',
    g_queue_push_head => '2.0',
    g_queue_pop_tail_link => '2.0',
    g_queue_pop_tail => '2.0',
    g_queue_pop_head_link => '2.0',
    g_queue_pop_head => '2.0',
    g_queue_peek_tail => '2.0',
    g_queue_peek_head => '2.0',
    g_queue_new => '2.0',
    g_queue_is_empty => '2.0',
    g_queue_free => '2.0',
    g_quark_try_string => '2.0',
    g_quark_to_string => '2.0',
    g_quark_from_string => '2.0',
    g_quark_from_static_string => '2.0',
    g_qsort_with_data => '2.0',
    g_ptr_array_sort_with_data => '2.0',
    g_ptr_array_sort => '2.0',
    g_ptr_array_sized_new => '2.0',
    g_ptr_array_set_size => '2.0',
    g_ptr_array_remove_index_fast => '2.0',
    g_ptr_array_remove_index => '2.0',
    g_ptr_array_remove_fast => '2.0',
    g_ptr_array_remove => '2.0',
    g_ptr_array_new => '2.0',
    g_ptr_array_index => '2.0',
    g_ptr_array_free => '2.0',
    g_ptr_array_add => '2.0',
    g_propagate_error => '2.0',
    g_private_set => '2.0',
    g_private_new => '2.0',
    g_private_get => '2.0',
    g_printf_string_upper_bound => '2.0',
    g_printerr => '2.0',
    g_print => '2.0',
    g_pattern_spec_new => '2.0',
    g_pattern_spec_free => '2.0',
    g_pattern_spec_equal => '2.0',
    g_pattern_match_string => '2.0',
    g_pattern_match_simple => '2.0',
    g_pattern_match => '2.0',
    g_path_skip_root => '2.0',
    g_path_is_absolute => '2.0',
    g_path_get_dirname => '2.0',
    g_path_get_basename => '2.0',
    g_parse_debug_string => '2.0',
    g_on_error_stack_trace => '2.0',
    g_on_error_query => '2.0',
    g_nullify_pointer => '2.0',
    g_ntohs => '2.0',
    g_ntohl => '2.0',
    g_node_unlink => '2.0',
    g_node_traverse => '2.0',
    g_node_reverse_children => '2.0',
    g_node_prev_sibling => '2.0',
    g_node_prepend_data => '2.0',
    g_node_prepend => '2.0',
    g_node_nth_child => '2.0',
    g_node_next_sibling => '2.0',
    g_node_new => '2.0',
    g_node_n_nodes => '2.0',
    g_node_n_children => '2.0',
    g_node_max_height => '2.0',
    g_node_last_sibling => '2.0',
    g_node_last_child => '2.0',
    g_node_is_ancestor => '2.0',
    g_node_insert_data_before => '2.0',
    g_node_insert_data_after => '2.0',
    g_node_insert_data => '2.0',
    g_node_insert_before => '2.0',
    g_node_insert_after => '2.0',
    g_node_insert => '2.0',
    g_node_get_root => '2.0',
    g_node_first_sibling => '2.0',
    g_node_first_child => '2.0',
    g_node_find_child => '2.0',
    g_node_find => '2.0',
    g_node_destroy => '2.0',
    g_node_depth => '2.0',
    g_node_copy => '2.0',
    g_node_children_foreach => '2.0',
    g_node_child_position => '2.0',
    g_node_child_index => '2.0',
    g_node_append_data => '2.0',
    g_node_append => '2.0',
    g_newa => '2.0',
    g_new0 => '2.0',
    g_new => '2.0',
    g_mutex_unlock => '2.0',
    g_mutex_trylock => '2.0',
    g_mutex_new => '2.0',
    g_mutex_lock => '2.0',
    g_mutex_free => '2.0',
    g_mutex_clear => '2.0',
    g_module_symbol => '2.0',
    g_module_supported => '2.0',
    g_module_open => '2.0',
    g_module_name => '2.0',
    g_module_make_resident => '2.0',
    g_module_error => '2.0',
    g_module_close => '2.0',
    g_module_build_path => '2.0',
    g_mkstemp => '2.0',
    g_message => '2.0',
    g_memmove => '2.0',
    g_memdup => '2.0',
    g_mem_set_vtable => '2.0',
    g_mem_profile => '2.0',
    g_mem_is_system_malloc => '2.0',
    g_mem_gc_friendly => '2.0',
    g_markup_parse_context_parse => '2.0',
    g_markup_parse_context_new => '2.0',
    g_markup_parse_context_get_position => '2.0',
    g_markup_parse_context_free => '2.0',
    g_markup_parse_context_end_parse => '2.0',
    g_markup_escape_text => '2.0',
    g_mapped_file_unref => '2.0',
    g_malloc0 => '2.0',
    g_malloc => '2.0',
    g_main_set_poll_func => '2.0',
    g_main_run => '2.0',
    g_main_quit => '2.0',
    g_main_pending => '2.0',
    g_main_new => '2.0',
    g_main_loop_unref => '2.0',
    g_main_loop_run => '2.0',
    g_main_loop_ref => '2.0',
    g_main_loop_quit => '2.0',
    g_main_loop_new => '2.0',
    g_main_loop_is_running => '2.0',
    g_main_loop_get_context => '2.0',
    g_main_iteration => '2.0',
    g_main_is_running => '2.0',
    g_main_destroy => '2.0',
    g_main_depth => '2.0',
    g_main_context_wakeup => '2.0',
    g_main_context_wait => '2.0',
    g_main_context_unref => '2.0',
    g_main_context_set_poll_func => '2.0',
    g_main_context_remove_poll => '2.0',
    g_main_context_release => '2.0',
    g_main_context_ref => '2.0',
    g_main_context_query => '2.0',
    g_main_context_prepare => '2.0',
    g_main_context_pending => '2.0',
    g_main_context_new => '2.0',
    g_main_context_iteration => '2.0',
    g_main_context_get_poll_func => '2.0',
    g_main_context_find_source_by_user_data => '2.0',
    g_main_context_find_source_by_id => '2.0',
    g_main_context_find_source_by_funcs_user_data => '2.0',
    g_main_context_dispatch => '2.0',
    g_main_context_default => '2.0',
    g_main_context_check => '2.0',
    g_main_context_add_poll => '2.0',
    g_main_context_acquire => '2.0',
    g_logv => '2.0',
    g_log_set_handler => '2.0',
    g_log_set_fatal_mask => '2.0',
    g_log_set_always_fatal => '2.0',
    g_log_remove_handler => '2.0',
    g_log_default_handler => '2.0',
    g_log => '2.0',
    g_locale_to_utf8 => '2.0',
    g_locale_from_utf8 => '2.0',
    g_list_sort_with_data => '2.0',
    g_list_sort => '2.0',
    g_list_reverse => '2.0',
    g_list_remove_link => '2.0',
    g_list_remove_all => '2.0',
    g_list_remove => '2.0',
    g_list_previous => '2.0',
    g_list_prepend => '2.0',
    g_list_position => '2.0',
    g_list_nth_prev => '2.0',
    g_list_nth_data => '2.0',
    g_list_nth => '2.0',
    g_list_next => '2.0',
    g_list_length => '2.0',
    g_list_last => '2.0',
    g_list_insert_sorted => '2.0',
    g_list_insert_before => '2.0',
    g_list_insert => '2.0',
    g_list_index => '2.0',
    g_list_free_1 => '2.0',
    g_list_free1 => '2.0',
    g_list_free => '2.0',
    g_list_foreach => '2.0',
    g_list_first => '2.0',
    g_list_find_custom => '2.0',
    g_list_find => '2.0',
    g_list_delete_link => '2.0',
    g_list_copy => '2.0',
    g_list_concat => '2.0',
    g_list_append => '2.0',
    g_list_alloc => '2.0',
    g_io_create_watch => '2.0',
    g_io_channel_write_unichar => '2.0',
    g_io_channel_write_chars => '2.0',
    g_io_channel_write => '2.0',
    g_io_channel_win32_new_socket => '2.0',
    g_io_channel_win32_new_messages => '2.0',
    g_io_channel_win32_new_fd => '2.0',
    g_io_channel_unref => '2.0',
    g_io_channel_unix_new => '2.0',
    g_io_channel_unix_get_fd => '2.0',
    g_io_channel_shutdown => '2.0',
    g_io_channel_set_line_term => '2.0',
    g_io_channel_set_flags => '2.0',
    g_io_channel_set_encoding => '2.0',
    g_io_channel_set_close_on_unref => '2.0',
    g_io_channel_set_buffered => '2.0',
    g_io_channel_set_buffer_size => '2.0',
    g_io_channel_seek_position => '2.0',
    g_io_channel_seek => '2.0',
    g_io_channel_ref => '2.0',
    g_io_channel_read_unichar => '2.0',
    g_io_channel_read_to_end => '2.0',
    g_io_channel_read_line_string => '2.0',
    g_io_channel_read_line => '2.0',
    g_io_channel_read_chars => '2.0',
    g_io_channel_read => '2.0',
    g_io_channel_new_file => '2.0',
    g_io_channel_init => '2.0',
    g_io_channel_get_line_term => '2.0',
    g_io_channel_get_flags => '2.0',
    g_io_channel_get_encoding => '2.0',
    g_io_channel_get_close_on_unref => '2.0',
    g_io_channel_get_buffered => '2.0',
    g_io_channel_get_buffer_size => '2.0',
    g_io_channel_get_buffer_condition => '2.0',
    g_io_channel_flush => '2.0',
    g_io_channel_error_from_errno => '2.0',
    g_io_channel_close => '2.0',
    g_io_add_watch_full => '2.0',
    g_io_add_watch => '2.0',
    g_int_hash => '2.0',
    g_int_equal => '2.0',
    g_idle_source_new => '2.0',
    g_idle_remove_by_data => '2.0',
    g_idle_add_full => '2.0',
    g_idle_add => '2.0',
    g_iconv_open => '2.0',
    g_iconv_close => '2.0',
    g_iconv => '2.0',
    g_htons => '2.0',
    g_htonl => '2.0',
    g_hook_unref => '2.0',
    g_hook_ref => '2.0',
    g_hook_prepend => '2.0',
    g_hook_next_valid => '2.0',
    g_hook_list_marshal_check => '2.0',
    g_hook_list_marshal => '2.0',
    g_hook_list_invoke_check => '2.0',
    g_hook_list_invoke => '2.0',
    g_hook_list_init => '2.0',
    g_hook_list_clear => '2.0',
    g_hook_insert_sorted => '2.0',
    g_hook_insert_before => '2.0',
    g_hook_get => '2.0',
    g_hook_free => '2.0',
    g_hook_first_valid => '2.0',
    g_hook_find_func_data => '2.0',
    g_hook_find_func => '2.0',
    g_hook_find_data => '2.0',
    g_hook_find => '2.0',
    g_hook_destroy_link => '2.0',
    g_hook_destroy => '2.0',
    g_hook_compare_ids => '2.0',
    g_hook_append => '2.0',
    g_hook_alloc => '2.0',
    g_hash_table_thaw => '2.0',
    g_hash_table_steal => '2.0',
    g_hash_table_size => '2.0',
    g_hash_table_replace => '2.0',
    g_hash_table_remove => '2.0',
    g_hash_table_new_full => '2.0',
    g_hash_table_new => '2.0',
    g_hash_table_lookup_extended => '2.0',
    g_hash_table_lookup => '2.0',
    g_hash_table_insert => '2.0',
    g_hash_table_freeze => '2.0',
    g_hash_table_foreach_steal => '2.0',
    g_hash_table_foreach_remove => '2.0',
    g_hash_table_foreach => '2.0',
    g_hash_table_destroy => '2.0',
    g_getenv => '2.0',
    g_get_user_name => '2.0',
    g_get_tmp_dir => '2.0',
    g_get_real_name => '2.0',
    g_get_prgname => '2.0',
    g_get_home_dir => '2.0',
    g_get_current_time => '2.0',
    g_get_current_dir => '2.0',
    g_get_codeset => '2.0',
    g_get_charset => '2.0',
    g_free => '2.0',
    g_find_program_in_path => '2.0',
    g_filename_to_utf8 => '2.0',
    g_filename_to_uri => '2.0',
    g_filename_from_utf8 => '2.0',
    g_filename_from_uri => '2.0',
    g_file_test => '2.0',
    g_file_open_tmp => '2.0',
    g_file_get_contents => '2.0',
    g_file_error_from_errno => '2.0',
    g_error_new_literal => '2.0',
    g_error_new => '2.0',
    g_error_matches => '2.0',
    g_error_free => '2.0',
    g_error_copy => '2.0',
    g_error => '2.0',
    g_dirname => '2.0',
    g_direct_hash => '2.0',
    g_direct_equal => '2.0',
    g_dir_rewind => '2.0',
    g_dir_read_name => '2.0',
    g_dir_open => '2.0',
    g_dir_close => '2.0',
    g_date_valid_year => '2.0',
    g_date_valid_weekday => '2.0',
    g_date_valid_month => '2.0',
    g_date_valid_julian => '2.0',
    g_date_valid_dmy => '2.0',
    g_date_valid_day => '2.0',
    g_date_valid => '2.0',
    g_date_to_struct_tm => '2.0',
    g_date_subtract_years => '2.0',
    g_date_subtract_months => '2.0',
    g_date_subtract_days => '2.0',
    g_date_strftime => '2.0',
    g_date_set_year => '2.0',
    g_date_set_time => '2.0',
    g_date_set_parse => '2.0',
    g_date_set_month => '2.0',
    g_date_set_julian => '2.0',
    g_date_set_dmy => '2.0',
    g_date_set_day => '2.0',
    g_date_order => '2.0',
    g_date_new_julian => '2.0',
    g_date_new_dmy => '2.0',
    g_date_new => '2.0',
    g_date_is_leap_year => '2.0',
    g_date_is_last_of_month => '2.0',
    g_date_is_first_of_month => '2.0',
    g_date_get_year => '2.0',
    g_date_get_weekday => '2.0',
    g_date_get_sunday_weeks_in_year => '2.0',
    g_date_get_sunday_week_of_year => '2.0',
    g_date_get_month => '2.0',
    g_date_get_monday_weeks_in_year => '2.0',
    g_date_get_monday_week_of_year => '2.0',
    g_date_get_julian => '2.0',
    g_date_get_days_in_month => '2.0',
    g_date_get_day_of_year => '2.0',
    g_date_get_day => '2.0',
    g_date_free => '2.0',
    g_date_days_between => '2.0',
    g_date_compare => '2.0',
    g_date_clear => '2.0',
    g_date_clamp => '2.0',
    g_date_add_years => '2.0',
    g_date_add_months => '2.0',
    g_date_add_days => '2.0',
    g_dataset_set_data_full => '2.0',
    g_dataset_set_data => '2.0',
    g_dataset_remove_no_notify => '2.0',
    g_dataset_remove_data => '2.0',
    g_dataset_id_set_data_full => '2.0',
    g_dataset_id_set_data => '2.0',
    g_dataset_id_remove_no_notify => '2.0',
    g_dataset_id_remove_data => '2.0',
    g_dataset_id_get_data => '2.0',
    g_dataset_get_data => '2.0',
    g_dataset_foreach => '2.0',
    g_dataset_destroy => '2.0',
    g_datalist_set_data_full => '2.0',
    g_datalist_set_data => '2.0',
    g_datalist_remove_no_notify => '2.0',
    g_datalist_remove_data => '2.0',
    g_datalist_init => '2.0',
    g_datalist_id_set_data_full => '2.0',
    g_datalist_id_set_data => '2.0',
    g_datalist_id_remove_no_notify => '2.0',
    g_datalist_id_remove_data => '2.0',
    g_datalist_id_get_data => '2.0',
    g_datalist_get_data => '2.0',
    g_datalist_foreach => '2.0',
    g_datalist_clear => '2.0',
    g_critical => '2.0',
    g_convert_with_iconv => '2.0',
    g_convert_with_fallback => '2.0',
    g_convert => '2.0',
    g_cond_wait => '2.0',
    g_cond_timed_wait => '2.0',
    g_cond_signal => '2.0',
    g_cond_new => '2.0',
    g_cond_free => '2.0',
    g_cond_broadcast => '2.0',
    g_completion_set_compare => '2.0',
    g_completion_remove_items => '2.0',
    g_completion_new => '2.0',
    g_completion_free => '2.0',
    g_completion_complete => '2.0',
    g_completion_clear_items => '2.0',
    g_completion_add_items => '2.0',
    g_clear_error => '2.0',
    g_cache_value_foreach => '2.0',
    g_cache_remove => '2.0',
    g_cache_new => '2.0',
    g_cache_key_foreach => '2.0',
    g_cache_insert => '2.0',
    g_cache_destroy => '2.0',
    g_byte_array_sort_with_data => '2.0',
    g_byte_array_sort => '2.0',
    g_byte_array_sized_new => '2.0',
    g_byte_array_set_size => '2.0',
    g_byte_array_remove_index_fast => '2.0',
    g_byte_array_remove_index => '2.0',
    g_byte_array_prepend => '2.0',
    g_byte_array_new => '2.0',
    g_byte_array_free => '2.0',
    g_byte_array_append => '2.0',
    g_build_path => '2.0',
    g_build_filename => '2.0',
    g_bit_storage => '2.0',
    g_bit_nth_msf => '2.0',
    g_bit_nth_lsf => '2.0',
    g_basename => '2.0',
    g_atexit => '2.0',
    g_async_queue_unref_and_unlock => '2.0',
    g_async_queue_unref => '2.0',
    g_async_queue_unlock => '2.0',
    g_async_queue_try_pop_unlocked => '2.0',
    g_async_queue_try_pop => '2.0',
    g_async_queue_timeout_pop_unlocked => '2.0',
    g_async_queue_timeout_pop => '2.0',
    g_async_queue_timed_pop_unlocked => '2.0',
    g_async_queue_timed_pop => '2.0',
    g_async_queue_ref_unlocked => '2.0',
    g_async_queue_ref => '2.0',
    g_async_queue_push_unlocked => '2.0',
    g_async_queue_push => '2.0',
    g_async_queue_pop_unlocked => '2.0',
    g_async_queue_pop => '2.0',
    g_async_queue_new => '2.0',
    g_async_queue_lock => '2.0',
    g_async_queue_length_unlocked => '2.0',
    g_async_queue_length => '2.0',
    g_assert_not_reached => '2.0',
    g_assert => '2.0',
    g_ascii_xdigit_value => '2.0',
    g_ascii_toupper => '2.0',
    g_ascii_tolower => '2.0',
    g_ascii_strup => '2.0',
    g_ascii_strtod => '2.0',
    g_ascii_strncasecmp => '2.0',
    g_ascii_strdown => '2.0',
    g_ascii_strcasecmp => '2.0',
    g_ascii_isxdigit => '2.0',
    g_ascii_isupper => '2.0',
    g_ascii_isspace => '2.0',
    g_ascii_ispunct => '2.0',
    g_ascii_isprint => '2.0',
    g_ascii_islower => '2.0',
    g_ascii_isgraph => '2.0',
    g_ascii_isdigit => '2.0',
    g_ascii_iscntrl => '2.0',
    g_ascii_isalpha => '2.0',
    g_ascii_isalnum => '2.0',
    g_ascii_formatd => '2.0',
    g_ascii_dtostr => '2.0',
    g_ascii_digit_value => '2.0',
    g_array_sort_with_data => '2.0',
    g_array_sort => '2.0',
    g_array_sized_new => '2.0',
    g_array_set_size => '2.0',
    g_array_remove_index_fast => '2.0',
    g_array_remove_index => '2.0',
    g_array_prepend_vals => '2.0',
    g_array_prepend_val => '2.0',
    g_array_new => '2.0',
    g_array_insert_vals => '2.0',
    g_array_insert_val => '2.0',
    g_array_index => '2.0',
    g_array_free => '2.0',
    g_array_append_vals => '2.0',
    g_array_append_val => '2.0',
    g_alloca => '2.0',
    TRUE => '2.0',
    NULL => '2.0',
    MIN => '2.0',
    MAXPATHLEN => '2.0',
    MAX => '2.0',
    G_WIN32_IS_NT_BASED => '2.0',
    G_WIN32_DLLMAIN_FOR_DLL_NAME => '2.0',
    G_VA_COPY => '2.0',
    G_VARIANT_TYPE_VARIANT => '2.0',
    G_VARIANT_TYPE_VARDICT => '2.0',
    G_VARIANT_TYPE_UNIT => '2.0',
    G_VARIANT_TYPE_UINT64 => '2.0',
    G_VARIANT_TYPE_UINT32 => '2.0',
    G_VARIANT_TYPE_UINT16 => '2.0',
    G_VARIANT_TYPE_TUPLE => '2.0',
    G_VARIANT_TYPE_STRING_ARRAY => '2.0',
    G_VARIANT_TYPE_STRING => '2.0',
    G_VARIANT_TYPE_SIGNATURE => '2.0',
    G_VARIANT_TYPE_OBJECT_PATH_ARRAY => '2.0',
    G_VARIANT_TYPE_OBJECT_PATH => '2.0',
    G_VARIANT_TYPE_MAYBE => '2.0',
    G_VARIANT_TYPE_INT64 => '2.0',
    G_VARIANT_TYPE_INT32 => '2.0',
    G_VARIANT_TYPE_INT16 => '2.0',
    G_VARIANT_TYPE_HANDLE => '2.0',
    G_VARIANT_TYPE_DOUBLE => '2.0',
    G_VARIANT_TYPE_DICT_ENTRY => '2.0',
    G_VARIANT_TYPE_DICTIONARY => '2.0',
    G_VARIANT_TYPE_BYTESTRING_ARRAY => '2.0',
    G_VARIANT_TYPE_BYTESTRING => '2.0',
    G_VARIANT_TYPE_BYTE => '2.0',
    G_VARIANT_TYPE_BOOLEAN => '2.0',
    G_VARIANT_TYPE_BASIC => '2.0',
    G_VARIANT_TYPE_ARRAY => '2.0',
    G_VARIANT_TYPE_ANY => '2.0',
    G_VARIANT_TYPE => '2.0',
    G_VARIANT_PARSE_ERROR => '2.0',
    G_USEC_PER_SEC => '2.0',
    G_URI_RESERVED_CHARS_SUBCOMPONENT_DELIMITERS => '2.0',
    G_URI_RESERVED_CHARS_GENERIC_DELIMITERS => '2.0',
    G_URI_RESERVED_CHARS_ALLOWED_IN_USERINFO => '2.0',
    G_URI_RESERVED_CHARS_ALLOWED_IN_PATH_ELEMENT => '2.0',
    G_URI_RESERVED_CHARS_ALLOWED_IN_PATH => '2.0',
    G_UNLOCK => '2.0',
    G_UNIX_ERROR => '2.0',
    G_UNICODE_COMBINING_MARK => '2.0',
    G_TRYLOCK => '2.0',
    G_THREAD_ERROR => '2.0',
    G_THREADS_IMPL_WIN32 => '2.0',
    G_THREADS_IMPL_POSIX => '2.0',
    G_STR_DELIMITERS => '2.0',
    G_STRUCT_OFFSET => '2.0',
    G_STRUCT_MEMBER_P => '2.0',
    G_STRUCT_MEMBER => '2.0',
    G_STRLOC => '2.0',
    G_STRINGIFY => '2.0',
    G_STMT_START => '2.0',
    G_STMT_END => '2.0',
    G_STATIC_RW_LOCK_INIT => '2.0',
    G_STATIC_REC_MUTEX_INIT => '2.0',
    G_STATIC_PRIVATE_INIT => '2.0',
    G_STATIC_MUTEX_INIT => '2.0',
    G_SQRT2 => '2.0',
    G_SPAWN_EXIT_ERROR => '2.0',
    G_SPAWN_ERROR => '2.0',
    G_SHELL_ERROR => '2.0',
    G_SEARCHPATH_SEPARATOR_S => '2.0',
    G_SEARCHPATH_SEPARATOR => '2.0',
    G_PRIORITY_LOW => '2.0',
    G_PRIORITY_HIGH_IDLE => '2.0',
    G_PRIORITY_HIGH => '2.0',
    G_PRIORITY_DEFAULT_IDLE => '2.0',
    G_PRIORITY_DEFAULT => '2.0',
    G_POLLFD_FORMAT => '2.0',
    G_PI_4 => '2.0',
    G_PI_2 => '2.0',
    G_PI => '2.0',
    G_PDP_ENDIAN => '2.0',
    G_OS_WIN32 => '2.0',
    G_OS_UNIX => '2.0',
    G_OPTION_ERROR => '2.0',
    G_N_ELEMENTS => '2.0',
    G_NODE_IS_ROOT => '2.0',
    G_NODE_IS_LEAF => '2.0',
    G_MODULE_SUFFIX => '2.0',
    G_MODULE_IMPORT => '2.0',
    G_MODULE_EXPORT => '2.0',
    G_MINSHORT => '2.0',
    G_MINOFFSET => '2.0',
    G_MINLONG => '2.0',
    G_MININT64 => '2.0',
    G_MININT => '2.0',
    G_MINFLOAT => '2.0',
    G_MINDOUBLE => '2.0',
    G_MEM_ALIGN => '2.0',
    G_MAXUSHORT => '2.0',
    G_MAXULONG => '2.0',
    G_MAXUINT64 => '2.0',
    G_MAXUINT => '2.0',
    G_MAXSHORT => '2.0',
    G_MAXOFFSET => '2.0',
    G_MAXLONG => '2.0',
    G_MAXINT64 => '2.0',
    G_MAXINT => '2.0',
    G_MAXFLOAT => '2.0',
    G_MAXDOUBLE => '2.0',
    G_MARKUP_ERROR => '2.0',
    G_LOG_LEVEL_USER_SHIFT => '2.0',
    G_LOG_FATAL_MASK => '2.0',
    G_LOG_DOMAIN => '2.0',
    G_LOG_2_BASE_10 => '2.0',
    G_LOCK_EXTERN => '2.0',
    G_LOCK_DEFINE_STATIC => '2.0',
    G_LOCK_DEFINE => '2.0',
    G_LOCK => '2.0',
    G_LN2 => '2.0',
    G_LN10 => '2.0',
    G_LITTLE_ENDIAN => '2.0',
    G_KEY_FILE_ERROR => '2.0',
    G_IO_CHANNEL_ERROR => '2.0',
    G_INLINE_FUNC => '2.0',
    G_IEEE754_FLOAT_BIAS => '2.0',
    G_IEEE754_DOUBLE_BIAS => '2.0',
    G_HOOK_IS_VALID => '2.0',
    G_HOOK_IS_UNLINKED => '2.0',
    G_HOOK_IN_CALL => '2.0',
    G_HOOK_FLAG_USER_SHIFT => '2.0',
    G_HOOK_FLAGS => '2.0',
    G_HOOK_ACTIVE => '2.0',
    G_HOOK => '2.0',
    G_HAVE_GNUC_VISIBILITY => '2.0',
    G_GUINT64_FORMAT => '2.0',
    G_GUINT32_FORMAT => '2.0',
    G_GUINT16_FORMAT => '2.0',
    G_GNUC_UNUSED => '2.0',
    G_GNUC_SCANF => '2.0',
    G_GNUC_PURE => '2.0',
    G_GNUC_PRINTF => '2.0',
    G_GNUC_PRETTY_FUNCTION => '2.0',
    G_GNUC_NO_INSTRUMENT => '2.0',
    G_GNUC_NORETURN => '2.0',
    G_GNUC_FUNCTION => '2.0',
    G_GNUC_FORMAT => '2.0',
    G_GNUC_EXTENSION => '2.0',
    G_GNUC_CONST => '2.0',
    G_GNUC_CHECK_VERSION => '2.0',
    G_GINT64_FORMAT => '2.0',
    G_GINT64_CONSTANT => '2.0',
    G_GINT32_FORMAT => '2.0',
    G_GINT16_FORMAT => '2.0',
    G_FILE_ERROR => '2.0',
    G_END_DECLS => '2.0',
    G_E => '2.0',
    G_DIR_SEPARATOR_S => '2.0',
    G_DIR_SEPARATOR => '2.0',
    G_DATE_BAD_YEAR => '2.0',
    G_DATE_BAD_JULIAN => '2.0',
    G_DATE_BAD_DAY => '2.0',
    G_DATALIST_FLAGS_MASK => '2.0',
    G_CSET_a_2_z => '2.0',
    G_CSET_LATINS => '2.0',
    G_CSET_LATINC => '2.0',
    G_CSET_DIGITS => '2.0',
    G_CSET_A_2_Z => '2.0',
    G_CONVERT_ERROR => '2.0',
    G_CONST_RETURN => '2.0',
    G_BYTE_ORDER => '2.0',
    G_BREAKPOINT => '2.0',
    G_BOOKMARK_FILE_ERROR => '2.0',
    G_BIG_ENDIAN => '2.0',
    G_BEGIN_DECLS => '2.0',
    G_ATOMIC_LOCK_FREE => '2.0',
    G_ASCII_DTOSTR_BUF_SIZE => '2.0',
    GWin32OSType => '2.0',
    GVoidFunc => '2.0',
    GVariantType => '2.0',
    GVariantParseError => '2.0',
    GVariantIter => '2.0',
    GVariantBuilder => '2.0',
    GUserDirectory => '2.0',
    GUnixFDSourceFunc => '2.0',
    GUnicodeType => '2.0',
    GUnicodeScript => '2.0',
    GUnicodeBreakType => '2.0',
    GULONG_TO_LE => '2.0',
    GULONG_TO_BE => '2.0',
    GULONG_FROM_LE => '2.0',
    GULONG_FROM_BE => '2.0',
    GUINT_TO_POINTER => '2.0',
    GUINT_TO_LE => '2.0',
    GUINT_TO_BE => '2.0',
    GUINT_FROM_LE => '2.0',
    GUINT_FROM_BE => '2.0',
    GUINT64_TO_LE => '2.0',
    GUINT64_TO_BE => '2.0',
    GUINT64_SWAP_LE_BE => '2.0',
    GUINT64_FROM_LE => '2.0',
    GUINT64_FROM_BE => '2.0',
    GUINT32_TO_LE => '2.0',
    GUINT32_TO_BE => '2.0',
    GUINT32_SWAP_LE_PDP => '2.0',
    GUINT32_SWAP_LE_BE => '2.0',
    GUINT32_SWAP_BE_PDP => '2.0',
    GUINT32_FROM_LE => '2.0',
    GUINT32_FROM_BE => '2.0',
    GUINT16_TO_LE => '2.0',
    GUINT16_TO_BE => '2.0',
    GUINT16_SWAP_LE_PDP => '2.0',
    GUINT16_SWAP_LE_BE => '2.0',
    GUINT16_SWAP_BE_PDP => '2.0',
    GUINT16_FROM_LE => '2.0',
    GUINT16_FROM_BE => '2.0',
    GTuples => '2.0',
    GTree => '2.0',
    GTraverseType => '2.0',
    GTraverseFunc => '2.0',
    GTraverseFlags => '2.0',
    GTrashStack => '2.0',
    GTranslateFunc => '2.0',
    GTokenValue => '2.0',
    GTokenType => '2.0',
    GTimer => '2.0',
    GTimeVal => '2.0',
    GTimeType => '2.0',
    GTime => '2.0',
    GThreadPriority => '2.0',
    GThreadPool => '2.0',
    GThreadFunc => '2.0',
    GThreadError => '2.0',
    GThread => '2.0',
    GTestTrapFlags => '2.0',
    GTestSuite => '2.0',
    GTestSubprocessFlags => '2.0',
    GTestCase => '2.0',
    GStrv => '2.0',
    GStringChunk => '2.0',
    GString => '2.0',
    GStaticRecMutex => '2.0',
    GStaticRWLock => '2.0',
    GStaticPrivate => '2.0',
    GStaticMutex => '2.0',
    GStatBuf => '2.0',
    GSpawnFlags => '2.0',
    GSpawnError => '2.0',
    GSpawnChildSetupFunc => '2.0',
    GSourceFuncs => '2.0',
    GSourceFunc => '2.0',
    GSourceDummyMarshal => '2.0',
    GSourceCallbackFuncs => '2.0',
    GSource => '2.0',
    GShellError => '2.0',
    GSequenceIterCompareFunc => '2.0',
    GSequenceIter => '2.0',
    GSequence => '2.0',
    GSeekType => '2.0',
    GScannerMsgFunc => '2.0',
    GScannerConfig => '2.0',
    GScanner => '2.0',
    GSSIZE_TO_LE => '2.0',
    GSSIZE_TO_BE => '2.0',
    GSSIZE_FROM_LE => '2.0',
    GSSIZE_FROM_BE => '2.0',
    GSList => '2.0',
    GSIZE_TO_POINTER => '2.0',
    GSIZE_TO_LE => '2.0',
    GSIZE_TO_BE => '2.0',
    GSIZE_FROM_LE => '2.0',
    GSIZE_FROM_BE => '2.0',
    GRelation => '2.0',
    GRand => '2.0',
    GQueue => '2.0',
    GQuark => '2.0',
    GPtrArray => '2.0',
    GPrivate => '2.0',
    GPrintFunc => '2.0',
    GPollFunc => '2.0',
    GPollFD => '2.0',
    GPid => '2.0',
    GPatternSpec => '2.0',
    GPOINTER_TO_UINT => '2.0',
    GPOINTER_TO_SIZE => '2.0',
    GPOINTER_TO_INT => '2.0',
    GOptionParseFunc => '2.0',
    GOptionGroup => '2.0',
    GOptionFlags => '2.0',
    GOptionErrorFunc => '2.0',
    GOptionError => '2.0',
    GOptionEntry => '2.0',
    GOptionContext => '2.0',
    GOptionArgFunc => '2.0',
    GOptionArg => '2.0',
    GNormalizeMode => '2.0',
    GNodeTraverseFunc => '2.0',
    GNodeForeachFunc => '2.0',
    GNode => '2.0',
    GMutex => '2.0',
    GModuleUnload => '2.0',
    GModuleFlags => '2.0',
    GModuleCheckInit => '2.0',
    GModule => '2.0',
    GMemVTable => '2.0',
    GMatchInfo => '2.0',
    GMarkupParser => '2.0',
    GMarkupParseFlags => '2.0',
    GMarkupParseContext => '2.0',
    GMarkupError => '2.0',
    GMarkupCollectType => '2.0',
    GMappedFile => '2.0',
    GMainLoop => '2.0',
    GMainContext => '2.0',
    GLogLevelFlags => '2.0',
    GLogFunc => '2.0',
    GList => '2.0',
    GLONG_TO_LE => '2.0',
    GLONG_TO_BE => '2.0',
    GLONG_FROM_LE => '2.0',
    GLONG_FROM_BE => '2.0',
    GLIB_MINOR_VERSION => '2.0',
    GLIB_MICRO_VERSION => '2.0',
    GLIB_MAJOR_VERSION => '2.0',
    GLIB_DISABLE_DEPRECATION_WARNINGS => '2.0',
    GLIB_CHECK_VERSION => '2.0',
    GKeyFileFlags => '2.0',
    GKeyFileError => '2.0',
    GKeyFile => '2.0',
    GIOStatus => '2.0',
    GIOFuncs => '2.0',
    GIOFunc => '2.0',
    GIOFlags => '2.0',
    GIOError => '2.0',
    GIOCondition => '2.0',
    GIOChannelError => '2.0',
    GIOChannel => '2.0',
    GINT_TO_POINTER => '2.0',
    GINT_TO_LE => '2.0',
    GINT_TO_BE => '2.0',
    GINT_FROM_LE => '2.0',
    GINT_FROM_BE => '2.0',
    GINT64_TO_LE => '2.0',
    GINT64_TO_BE => '2.0',
    GINT64_FROM_LE => '2.0',
    GINT64_FROM_BE => '2.0',
    GINT32_TO_LE => '2.0',
    GINT32_TO_BE => '2.0',
    GINT32_FROM_LE => '2.0',
    GINT32_FROM_BE => '2.0',
    GINT16_TO_LE => '2.0',
    GINT16_TO_BE => '2.0',
    GINT16_FROM_LE => '2.0',
    GINT16_FROM_BE => '2.0',
    GIConv => '2.0',
    GHookMarshaller => '2.0',
    GHookList => '2.0',
    GHookFunc => '2.0',
    GHookFlagMask => '2.0',
    GHookFindFunc => '2.0',
    GHookFinalizeFunc => '2.0',
    GHookCompareFunc => '2.0',
    GHookCheckMarshaller => '2.0',
    GHookCheckFunc => '2.0',
    GHook => '2.0',
    GHashTableIter => '2.0',
    GHashTable => '2.0',
    GHashFunc => '2.0',
    GHRFunc => '2.0',
    GHFunc => '2.0',
    GFunc => '2.0',
    GFreeFunc => '2.0',
    GFormatSizeFlags => '2.0',
    GFloatIEEE754 => '2.0',
    GFileTest => '2.0',
    GFileError => '2.0',
    GErrorType => '2.0',
    GError => '2.0',
    GEqualFunc => '2.0',
    GDuplicateFunc => '2.0',
    GDoubleIEEE754 => '2.0',
    GDir => '2.0',
    GDestroyNotify => '2.0',
    GDebugKey => '2.0',
    GDateYear => '2.0',
    GDateWeekday => '2.0',
    GDateTime => '2.0',
    GDateMonth => '2.0',
    GDateDay => '2.0',
    GDateDMY => '2.0',
    GDate => '2.0',
    GDataForeachFunc => '2.0',
    GData => '2.0',
    GConvertError => '2.0',
    GCond => '2.0',
    GCompletionStrncmpFunc => '2.0',
    GCompletionFunc => '2.0',
    GCompletion => '2.0',
    GCompareFunc => '2.0',
    GCompareDataFunc => '2.0',
    GChildWatchFunc => '2.0',
    GCacheNewFunc => '2.0',
    GCacheDupFunc => '2.0',
    GCacheDestroyFunc => '2.0',
    GCache => '2.0',
    GByteArray => '2.0',
    GBookmarkFileError => '2.0',
    GBookmarkFile => '2.0',
    GAsyncQueue => '2.0',
    GArray => '2.0',
    FALSE => '2.0',
    CLAMP => '2.0',
    ABS => '2.0',
)
;

1;
__END__

=back

=head1 NAMING

Official casing is GLib, but there's a Glib module already on CPAN, so this follows suit. This and the L<Glib> module are unrelated though.

=head1 GIT REPOSITORY

L<http://github.com/athreef/Glib-FindMinVersion>

=head1 SEE ALSO

L<GLib Reference Manual|https://developer.gnome.org/glib/stable/>

This module was inspired by Wireshark's L<tools/checkAPIs.pl|https://github.com/wireshark/wireshark/blob/master/tools/checkAPIs.pl>.

=head1 AUTHOR

Ahmad Fatoum C<< <athreef@cpan.org> >>, L<http://a3f.at>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2017 Ahmad Fatoum

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut
