package String::BooleanSimple; ## Gets the boolean representative of a string
$String::BooleanSimple::VERSION = '0.026';



use strict;

use vars qw(@ISA @EXPORT %EXPORT_TAGS $VERSION);
use Exporter; 
use boolean qw(true false);


@ISA = qw(Exporter);

%EXPORT_TAGS = ( all => [qw(
                      boolean
                      is_true
                      is_false
                      isTrue  
                      isFalse
                )] ); 

Exporter::export_ok_tags('all'); 


# Some static functions returning a boolean value (0 or 1) of a string like
# 'true', 'false', etc. In the background it returns the value from the module
# boolean, which holds a true/false method. What means, if you change that rules,
# it would not return a 0 or 1, but whatever you defined.
#
# SYNOPSIS
# ========
# 
#  # imports all functions
#  use String::BooleanSimple ':all';
#
#  # imports only is_true()
#  use String::BooleanSimple qw(is_true);
#  
#
# Matches
# =======
# Supports these strings:
#
#
#   true yes active enabled on y ok positive 1 2 3 4 5 6 7 8 9
#
#   false no inactive disabled off n not ok negative 0
#
# If the string does not match, it causes an error. Whitespace at the beginning or end will be automatically removed.
#
# Default values
# ==============
# You may set a default string as second parameter on the functions. It works if the first value can not be indentified
# as valid string. An empty string is also a non-valid string and will trigger the default value.
#
# Example:
#
#  if ( is_true("","false") ){};
#
# it will be false. Typical for reading config files, where a value does not allways exist.
#
# If the default value as well is not a valid string, it dies with an error.
#
# Module boolean
# ==============
# If the calling application using the module boolean, you may write it like that:
#
#  use boolean ':all';
#  use String::BooleanSimple 'boolean';
#
#  my $s='positive'
#  
#  if ( isTrue( boolean($s) ) ){...};
#
#  if ( boolean($s) == true ){...};
#
#
# Please note, here the isTrue() function is part of "use boolean"!
# It is not imported with ':all' from String::BooleanSimple because that is a conflict.
#
# The following example is possible and logical, but looks silly:
#
#  if ( is_true($s) == true ){...};
#
# Theoretically you must do like that, if "use boolean", because who knows what is realy "true" and "false"? But
# it is not very nice to read it and the simple way "if( is_true() )" worth to risk that maybe false is not '0'.
#
# In other words, if the calling app is not using perl's "boolean" but somehow in perl's module "boolean" the
# value of true/false does not fit anymore to 1/0, it might be, that using is_true/is_false with String::BooleanSimple
# will cause wrong behaviour.
#
#
# LICENSE
# =======   
# You can redistribute it and/or modify it under the conditions of LGPL.
# 
# AUTHOR
# ======
# Andreas Hernitscheck  ahernit(AT)cpan.org


# Returns 1 if the string matches to a positive pattern like "true".
sub is_true { # $boolean ($string,$defaultstring)
  my $value = shift;
  my $def_value = shift;
  my $ret = undef;

  $ret = boolean($value,$def_value);

  return $ret;
}



# Returns 1 if the string matches to a negative pattern like "false".
sub is_false { # $boolean ($string,$defaultstring)
  my $value = shift;
  my $def_value = shift;
  my $ret = undef;

  $ret = 1 - ( boolean($value,$def_value) == true ? true : 0 );

  $ret ? true : false;

  return $ret;
}


# Returns 1 if the string matches to a postive pattern like "true".
# Returns 0 if the string matches to a negative pattern like "false".
sub boolean { # $boolean ($string,$defaultstring)
  my $value = shift;
  my $def_value = shift;
  my $ret = 2;

  # trim
  $value = _trim( $value );

  # lower case
  $value = lc( $value );

  if ( $value =~ m/^(true|yes|active|enabled|on|y|ok|positive|[1-9])$/ ){ $ret = 1 };

  if ( $value =~ m/^(false|no|inactive|disabled|off|n|negative|not ok|[0])$/ ){ $ret = 0 };


  if ($ret == 2) {

    if ( $def_value ne '' ){
      $ret = boolean( $def_value );
    }else{

      die "String value \'$value\' does not match to a given true/false pattern.";

    }
  }

  $ret ? true : false;

  return $ret;
}


# trimming the string from whitespace
sub _trim {
  my $value = shift;

  $value =~ s/^\s*//s;
  $value =~ s/\s*$//s;
  
  return $value;
}


# If you like the java style, you may import that alias
sub isTrue{ # $boolean ($string,$defaultstring)
  return is_true(@_);
}

# If you like the java style, you may import that alias
sub isFalse{ # $boolean ($string,$defaultstring)
  return is_false(@_);
}


1;

#################### pod generated by Pod::Autopod - keep this line to make pod updates possible ####################

=head1 NAME

String::BooleanSimple - Gets the boolean representative of a string


=head1 SYNOPSIS


 # imports all functions
 use String::BooleanSimple ':all';

 # imports only is_true()
 use String::BooleanSimple qw(is_true);
 



=head1 DESCRIPTION

Some static functions returning a boolean value (0 or 1) of a string like
'true', 'false', etc. In the background it returns the value from the module
boolean, which holds a true/false method. What means, if you change that rules,
it would not return a 0 or 1, but whatever you defined.



=head1 REQUIRES

L<boolean> 

L<Exporter> 


=head1 METHODS

=head2 boolean

 my $boolean = boolean($string, $defaultstring);

Returns 1 if the string matches to a postive pattern like "true".
Returns 0 if the string matches to a negative pattern like "false".


=head2 isFalse

 my $boolean = isFalse($string, $defaultstring);

If you like the java style, you may import that alias


=head2 isTrue

 my $boolean = isTrue($string, $defaultstring);

If you like the java style, you may import that alias


=head2 is_false

 my $boolean = is_false($string, $defaultstring);

Returns 1 if the string matches to a negative pattern like "false".


=head2 is_true

 my $boolean = is_true($string, $defaultstring);

Returns 1 if the string matches to a positive pattern like "true".



=head1 Matches

Supports these strings:


  true yes active enabled on y ok positive 1 2 3 4 5 6 7 8 9

  false no inactive disabled off n not ok negative 0

If the string does not match, it causes an error. Whitespace at the beginning or end will be automatically removed.



=head1 Default values

You may set a default string as second parameter on the functions. It works if the first value can not be indentified
as valid string. An empty string is also a non-valid string and will trigger the default value.

Example:

 if ( is_true("","false") ){};

it will be false. Typical for reading config files, where a value does not allways exist.

If the default value as well is not a valid string, it dies with an error.



=head1 Module boolean

If the calling application using the module boolean, you may write it like that:

 use boolean ':all';
 use String::BooleanSimple 'boolean';

 my $s='positive'
 
 if ( isTrue( boolean($s) ) ){...};

 if ( boolean($s) == true ){...};


Please note, here the isTrue() function is part of "use boolean"!
It is not imported with ':all' from String::BooleanSimple because that is a conflict.

The following example is possible and logical, but looks silly:

 if ( is_true($s) == true ){...};

Theoretically you must do like that, if "use boolean", because who knows what is realy "true" and "false"? But
it is not very nice to read it and the simple way "if( is_true() )" worth to risk that maybe false is not '0'.

In other words, if the calling app is not using perl's "boolean" but somehow in perl's module "boolean" the
value of true/false does not fit anymore to 1/0, it might be, that using is_true/is_false with String::BooleanSimple
will cause wrong behaviour.




=head1 AUTHOR

Andreas Hernitscheck  ahernit(AT)cpan.org


=head1 LICENSE

You can redistribute it and/or modify it under the conditions of LGPL.



=cut

