(* Copyright (C) 1989, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* Last modified on Mon Nov 18 14:41:43 PST 1991 by muller         *)

INTERFACE RTScheduler;

IMPORT Thread, Unix, Utime, Time;

PROCEDURE IOSelect (nfds: INTEGER;
                  readfds, writefds, exceptfds: UNTRACED REF Unix.FDSet;
                  timeout: UNTRACED REF Utime.struct_timeval := NIL): INTEGER;

PROCEDURE IOAlertSelect (nfds: INTEGER;
                         readfds, writefds, exceptfds: UNTRACED REF Unix.FDSet;
                         timeout: UNTRACED REF Utime.struct_timeval := NIL)
        : INTEGER RAISES {Thread.Alerted};

(* These procedures are functionnally equivalent to select(2). The difference
   is that they does not prevent the other threads from running. Also, with
   a non zero-valued timeout, they may return an arbitrary time after the 
   timeout had expired (of course, we try to make this difference as small
   as possible). With a zero-valued timeout, they returns right away
   (unless some other thread preempts, say during the calling or return step).
   As usual, AlertSelect is the alertable version, Select is the non-alertable
   version.   

   There is no guarantee that the result of Select can be valid for some
   period of time. For example, an other thread may preempt just during the
   return sequence from Select and read some data from an input file.
   The guarantee is that a call to select(2) gave this result at some point
   after the call to Select.

   Therefore, the result of Select should be treated as a hint.  A standard
   mechanism is to make reads non-blocking on the file descriptor of interest,
   and to have a loop of the form:

      LOOP
	status := read (fd, buf, n);
        IF status = -1 AND errno # EWOULDBLOCK THEN
	  (* error to be handled *)
        ELSIF status = 0 THEN
          (* eof reached *)
        ELSIF status > 0 THEN
          (* status chars available in buf *)
        END;

        EVAL RTScheduler.Select (fd, FDSet {fd}, NIL, FDSet {fd});
      END;

*)

PROCEDURE Pause (until: Time.T; alertable := FALSE) RAISES {Thread.Alerted};
(* This procedure removes the calling thread from the list of running threads;
   it will be reinserted sometime after until, hopefully very near that time.
   This procedure is here for the benefit of the Time package; you 
   should probably use procedures of the Time interfaces instead. *)

PROCEDURE SetSwitchingInterval (interval: Time.T);
(* select the interval at which switching interrupts should occur. *)

END RTScheduler.
