(* Copyright (C) 1990, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* Last modified on Tue Sep  3 22:12:51 1991 by kalsow         *)
(*      modified on Tue Jan 30 10:57:41 1990 by muller         *)
(*      modified on Thu Jan 25 22:51:07 PST 1990 by stolfi     *)

UNSAFE MODULE RandomReal EXPORTS Random;

IMPORT Word;

(*  
  Implements Random.Real and Random.LongReal for the VAX architecture,
  assuming the DFLOAT format for LONGREAL. *)

PROCEDURE Real (self: T := Default): REAL =
  CONST ExponentBias = 129;  (* Exponent bias for REAL *)
        FractionBits = 23;   (* Number of explicit fraction bits *)
        WordSize = 32;       (* Size of INTEGER in bits *)
  VAR frac, exp: INTEGER;
  BEGIN
    (* Generate a random fraction and get its first non-zero word: *)
    exp := ExponentBias - 1;
    frac := Integer (self);
    WHILE (frac = 0) AND (exp >= WordSize) DO
      (* This loop is (almost) never executed: *)
      DEC (exp, WordSize);
      frac := Integer (self);
    END;
 
    (* Normalize: *)
    WHILE (frac > 0) AND (exp > 0) DO
      (* This loop is executed about once on the average. *)
      frac := Word.Shift (frac, 1);
      DEC (exp);
    END;
    IF exp <= 0 THEN RETURN 0.0 END;  (* Store zero on underflow *)
    IF ((ExponentBias - 1 - exp) MOD WordSize) >  WordSize - FractionBits THEN
      (* Needs more random bits *)
      frac := Integer (self)
    END;

    (* Repack as REAL: *)
    RETURN LOOPHOLE (
      Word.Rotate(
        Word.Or (
          Word.Shift (exp, FractionBits), 
          Word.Shift (
            Word.And (frac, 16_7fffffff),
            -(WordSize - 1 - FractionBits)
          )
        ),
        16
      ),
      REAL
    );
  END Real;

PROCEDURE LongReal (self: T := Default): LONGREAL =
  VAR dbl: ARRAY [0..1] OF Word.T;
  BEGIN
    dbl[0] := LOOPHOLE(Real(self), Word.T); (* Exp and high-order fraction bits *)
    dbl[1] := Integer (self);  (* Low-order fraction bits *)
    RETURN LOOPHOLE(dbl, LONGREAL)
  END LongReal;

BEGIN
END RandomReal.
