(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)

(* Created by stolfi on Wed Oct 24  2:21:32 PDT 1990           *)
(* Last modified on Fri Mar 13 14:53:47 PST 1992 by muller     *)
(*      modified on Mon Oct 29 17:46:24 PST 1990 by stolfi     *)

INTERFACE ColorMatch;

(* Color matching.
  
  This interface exports procedures for finding the nearest match
  to a given color among a set of RGB values.
  
  Index: colors; graphics, colors; display, color.
*)

IMPORT Color;

TYPE
  Table = ARRAY OF Color.T;
    (*
      An array of color values. *)

EXCEPTION EmptyTable;
  (*
    Raised if the color table to be searched contains no 
    entries that are not RGB.Undefined. *)

PROCEDURE Nearest (
    READONLY a: Color.T; 
    READONLY table: Table;
    dist: Color.DistFn;
    parm: REAL := 1.0;
  ): CARDINAL RAISES {EmptyTable};
  (*
    Returns the index p in [0..LAST(table)] such that the distance
    dist(a,table[p],parm) is minimized.  Ignores tables entries that
    are Color.Undefined.  *)

TYPE
  IndexPair = ARRAY [0..1] OF CARDINAL;

PROCEDURE NearestPair(
    a: Color.T; 
    READONLY table: Table; 
    dist: Color.DistFn;
    parm: REAL := 1.0;
  ): IndexPair RAISES {EmptyTable};
  (* 
    Finds two colors in table[0..LAST(table)] whose arithmentic
    average is a good approximation of the given triple.  
    Ignores tables entries that are Color.Undefined.   *)

TYPE
  IndexQuad = ARRAY [0..3] OF CARDINAL;

PROCEDURE NearestQuad(
    a: Color.T; 
    READONLY table: Table; 
    dist: Color.DistFn;
    parm: REAL := 1.0;
  ): IndexQuad RAISES {EmptyTable};
  (* 
    Finds four colors in table[0..LAST(table)] whose arithmentic
    average is a good approximation of the given triple.  
    Ignores tables entries that are Color.Undefined.   *)

END ColorMatch.
