(* emake PC-Version
|  ----------------
|
|
|  Carsten Weich, July 1993
   **************************************************************************)

MODULE emake EXPORTS Main;


IMPORT Wr, Rd, RdUtils, Scan, Text, Thread, Params, ParseParams,
       FileStream, Time, System, Modules, ProgFileIO;
FROM Stdio IMPORT stdout, stderr;

<* FATAL Thread.Alerted, Wr.Failure *>


CONST
    PcMode = TRUE;
    UsageString = "[-Include PublicDir] [-Source SourceDir] [-Library Libname]\n"
                      & "[-Lforms] [-Lpvm} [-list] [-verbose] [-search] [-help] [-makefile] RootFileName";
    FormsVbtLibs = "-lm3formsvbt -lm3vbtkit -lm3ui -lm3X11R4 -lX11 ";
    FormsVbtMake = "import_obj($(FORMSVBTLIB))\n";
    PvmLibs = "-lm3pvm -lpvm ";
    PvmMake = "import_lib(m3pvm)\nimport_lib(pvm)\n";
    PgmListFile = "pgm_list";


TYPE EmakeMode = {List, Makefile, Compile};


VAR
    rootName: TEXT;
    includestring := "-D/m3/local/include "; (* zus.  Interfaces *)
    libstring := "-lm3loc -lgpl "; (* zustzliche Libraries *)
    sourcePath := ".";           (* Directorypath fr Sourcen *)
    publicPath := "/m3/include:/m3/local/include"; (* Path fr
                                                              Interf. *)

    modules := NEW(Modules.T);
    progName := Params.Get(0);
    mode: EmakeMode;
    verbose, searchMode, formsVbtMode, pvmMode: BOOLEAN;


(*****************************************************************************)
(* Utility-Prozeduren *)
(*****************************************************************************)

PROCEDURE Log (l: TEXT) =
    BEGIN
        IF verbose THEN Wr.PutText(stdout, l); Wr.Flush(stdout); END;
    END Log;

PROCEDURE HelpMessage () =
    BEGIN
        WITH P = Wr.PutText, o = stdout DO
            P(o, "emake-Options:\n\n");
            P(o, "-I[nclude] PublicDir:\n");
            P(o, "     Search for public Interfaces also in PublicDir."
                     & " These won't be compiled.\n\n");
            P(o, "-S[ource] SourceDir:\n");
            P(o, "     Search for parts of the program also in Source"
                     & "Dir (.i3 and .m3 files).\n\n");
            P(o, "-L[ibrary] Libname:\n");
            P(o, "     Link Libfilename (this is only passed to the co"
                     & "mpiler). If the file is\n");
            P(o, "     'libm3rpc.a' you have to specify as Libname '-L"
                     & " m3rpc'.\n\n");
            P(o, "-Lforms:\n");
            P(o, "     Link all necessary libraries to use FormsVBT\n\n");
            P(o, "-Lpvm:\n");
            P(o, "     Link all necessary libraries to use PVM\n");
            P(o, "     Same as -L m3pvm -L pvm\n\n");
            P(o, "-l, -list:\n");
            P(o, "     Don't compile anything, just show modules\n\n");
            P(o, "-s, -search:\n");
            P(o, "     Search for Implementation-Modules that have a "
                     & "filename not equal\n");
            P(o, "     to the Interfacefilename.\n");
            P(o, "     By default, emake searches only for Implementa"
                     & "tions with a filename\n");
            P(o, "     either equal to the Interfacename.m3 or someth"
                     & "ing like\n");
            P(o, "     Interfacename_xy.m3.\n\n");
            P(o, "-makefile:\n");
            P(o, "     Produce a m3makefile (on stdout) instead of di"
                     & "rectly starting the compiler\n\n");
            P(o, "-v, -verbose:\n");
            P(o, "     emake and m3 will tell something about what they"
                     & " are doing\n");
            Wr.Flush(o);
        END;
    END HelpMessage;


PROCEDURE PutOptions () =
    BEGIN
        Wr.PutText(stdout, "emake-Settings:\n\t");
        IF verbose THEN Wr.PutText(stdout, "verbose, ") END;
        IF formsVbtMode THEN Wr.PutText(stdout, "formsVbtMode, ") END;
        IF pvmMode THEN Wr.PutText(stdout, "pvmMode, ") END;
        IF searchMode THEN Wr.PutText(stdout, "searchMode, ") END;
        CASE mode OF
          EmakeMode.Compile => Wr.PutText(stdout, "compile\n");
        | EmakeMode.Makefile => Wr.PutText(stdout, "make m3makefile\n");
        | EmakeMode.List => Wr.PutText(stdout, "list modules\n");
        END;                     (*CASE*)
        Wr.PutText(stdout, "\tLibstring:\t>" & libstring & "<\n");
        Wr.PutText(stdout, "\tPublics:\t>" & publicPath & "<\n");
        Wr.PutText(stdout, "\tSources:\t>" & sourcePath & "<\n");
    END PutOptions;



(*****************************************************************************)
(* Runstringparameter abarbeiten *)
(*****************************************************************************)

PROCEDURE ProcessParameters (VAR mode: EmakeMode; VAR rootName: TEXT)
    RAISES {Scan.BadFormat} =
    VAR par: TEXT;
    BEGIN
        ParseParams.BeginParsing();

        mode := EmakeMode.Compile; (* Default *)
        verbose := FALSE;
        searchMode := FALSE;
        formsVbtMode := FALSE;
        pvmMode := FALSE;

        IF ParseParams.KeywordPresent("-l")
               OR ParseParams.KeywordPresent("-list") THEN
            mode := EmakeMode.List;
        END;                     (*IF*)
        WHILE ParseParams.KeywordPresent("-I") DO
            par:= ParseParams.GetNext();
            publicPath := publicPath & ":" & par & " ";
            includestring:= includestring & " -D" & par & " ";
        END;                     (*WHILE*)
        WHILE ParseParams.KeywordPresent("-Include") DO
            par:= ParseParams.GetNext();
            publicPath := publicPath & ":" & par & " ";
            includestring:= includestring & " -D" & par & " ";
        END;                     (*WHILE*)
        WHILE ParseParams.KeywordPresent("-S") DO
            sourcePath := sourcePath & ":" & ParseParams.GetNext();
        END;                     (*WHILE*)
        WHILE ParseParams.KeywordPresent("-Source") DO
            sourcePath := sourcePath & ":" & ParseParams.GetNext();
        END;                     (*WHILE*)
        WHILE ParseParams.KeywordPresent("-L") DO
            libstring := libstring & "-l" & ParseParams.GetNext() & " ";
        END;                     (*WHILE*)
        WHILE ParseParams.KeywordPresent("-Library") DO
            libstring := libstring & "-l" & ParseParams.GetNext() & " ";
        END;                     (*WHILE*)
        IF ParseParams.KeywordPresent("-v")
               OR ParseParams.KeywordPresent("-verbose") THEN
            verbose := TRUE;
            Modules.verbose := TRUE;
        END;                     (*IF*)
        IF ParseParams.KeywordPresent("-Lforms") THEN
            formsVbtMode := TRUE;
        END;                     (*IF*)
        IF ParseParams.KeywordPresent("-Lpvm") THEN
            pvmMode := TRUE;
        END;                     (*IF*)
        IF ParseParams.KeywordPresent("-s")
               OR ParseParams.KeywordPresent("-search") THEN
            searchMode := TRUE;
        END;                     (*IF*)
        IF ParseParams.KeywordPresent("-makefile") THEN
            mode := EmakeMode.Makefile;
        END;                     (*IF*)
        IF ParseParams.KeywordPresent("-h")
               OR ParseParams.KeywordPresent("-help")
               OR ParseParams.KeywordPresent("-Help")
               OR ParseParams.KeywordPresent("-?") THEN
            HelpMessage();
            RAISE Scan.BadFormat;
        END;                     (*IF*)
        ParseParams.UnparsedTail();

        rootName := ParseParams.GetNext();
        ParseParams.EndParsing();
    END ProcessParameters;


(*****************************************************************************)
(* Produce - Prozeduren *)
(*****************************************************************************)

PROCEDURE ProduceMakefile (modules: Modules.T) =
    VAR modName, libName: TEXT;

    BEGIN
        Wr.PutText(stdout, "M3DEFPATH=" & includestring & "\n");
        modules.beginScann();
        modName := modules.next();
        IF modName # NIL THEN
            Wr.PutText(
                stdout, "program(" & Modules.Basename(modName) & ")\n");
        END;                     (*IF*)

        WHILE modName # NIL DO
            IF NOT modules.publicInterface() THEN
                IF Text.Equal(Modules.Extname(modName), "i3") THEN
                    Wr.PutText(
                        stdout, "interface(" & Text.Sub(modName, 0,
                                                        Text.FindCharR(
                                                            modName, '.'))
                                    & ")\n");
                ELSIF Text.Equal(Modules.Extname(modName), "m3") THEN
                    Wr.PutText(stdout,
                               "implementation("
                                   & Text.Sub(modName, 0,
                                              Text.FindCharR(modName, '.'))
                                   & ")\n");
                ELSE
                    Wr.PutText(stdout, "****unknown: " & modName & ")\n");
                END;             (*IF*)
            END;                 (*IF*)
            modName := modules.next();
        END;                     (*WHILE*)
        WHILE NOT Text.Equal(libstring, "") DO
            libName := Text.Sub(libstring, Text.Length("-l"),
                                Text.FindChar(libstring, ' ')
                                    - Text.Length("-l"));
            libstring :=
                Text.Sub(libstring,
                         Text.Length(libName) + Text.Length("-l") + 1,
                         Text.Length(libstring) - Text.Length(libName));
            Wr.PutText(stdout, "import_lib(" & libName & ")\n");
        END;                     (*WHILE*)
        IF formsVbtMode THEN Wr.PutText(stdout, FormsVbtMake); END; (*IF*)
        IF pvmMode THEN Wr.PutText(stdout, PvmMake); END; (*IF*)
        Wr.Flush(stdout);
    END ProduceMakefile;


PROCEDURE ProduceList (modules: Modules.T) =
    VAR modName: TEXT;
    BEGIN
        modules.beginScann();
        modName := modules.next();

        WHILE modName # NIL DO
            IF modules.publicInterface() THEN
                Wr.PutText(stdout, "\t");
            END;                 (*IF*)
            IF NOT modules.uptodate() THEN
                Wr.PutText(stdout, "*");
            END;                 (*IF*)
            Wr.PutText(stdout, modName & "\n\t");
            modName := modules.next();
        END;                     (*WHILE*)
        Wr.PutText(stdout, "\r");
        Wr.Flush(stdout);
    END ProduceList;



PROCEDURE CompileModules (modules: Modules.T) RAISES {System.Error} =
    VAR
        command := "m3 -why -w1 -make " & includestring & libstring;
        modName, aoutName: TEXT;
        aoutUpdate: Time.T;
        pgmList: Wr.T;
    BEGIN
        modules.beginScann();
        modName := modules.next();
        aoutName := Modules.Basename(modName);

        (* Monitor, if compilation was successfull: *)
        TRY
            aoutUpdate := System.FileModifyTime(aoutName);
        EXCEPT
          System.Error => aoutUpdate := Time.Epoch;
        END;

        (* put the compiler-commandstring together: *)
        IF verbose THEN command := command & "-verbose "; END; (*IF*)
        command := command & " -o " & aoutName & " ";
        IF PcMode THEN
            pgmList := FileStream.OpenWrite(PgmListFile);
            WHILE modName # NIL DO
                IF NOT modules.publicInterface() THEN
                    Wr.PutText(pgmList, modName & "\n");
                END;             (*IF*)
                modName := modules.next();
            END;                 (*WHILE*)
            Wr.Close(pgmList);
            command := command & " -F" & PgmListFile;
        ELSE
            WHILE modName # NIL DO
                IF NOT modules.publicInterface() THEN
                    command := command & modName & " ";
                END;             (*IF*)
                modName := modules.next();
            END;                 (*WHILE*)
        END;
        IF formsVbtMode THEN
            command := command & " " & FormsVbtLibs;
        END;                     (*IF*)
        IF pvmMode THEN command := command & " " & PvmLibs; END; (*IF*)
        IF NOT PcMode THEN
            command := command & "2>&1| more";
        END;

        Log(command & "\n");
        System.CallProg(command);
        TRY
            IF Time.Compare(aoutUpdate, System.FileModifyTime(aoutName))
                   < 1 THEN
                Log("new program " & aoutName & "\n");
                IF PcMode THEN
                    Log("converting to " & aoutName & ".exe\n");
                    System.CallProg("aout2exe " & aoutName);
                    Log("deleting " & aoutName & "\n");
                    System.CallProg("del " & aoutName);
                END;
            END;
        EXCEPT
          System.Error => Log("no new program\n");
        END;
    END CompileModules;


(*****************************************************************************)

BEGIN

    TRY
        ProcessParameters(mode, rootName);
        Log("searching for modules...\n");

        IF verbose THEN PutOptions() END;

        modules.sourcePath := sourcePath;
        modules.publicPath := publicPath;
        modules.searchMode := searchMode;
        modules.getModules(rootName);

        CASE mode OF
          EmakeMode.Makefile => ProduceMakefile(modules);
        | EmakeMode.List => ProduceList(modules);
        | EmakeMode.Compile => CompileModules(modules);
        END;                     (*CASE*)


    EXCEPT
      Rd.Failure (reason) =>
          Wr.PutText(
              stderr, progName & ": " & RdUtils.FailureText(reason) & "\n");
    | Rd.EndOfFile => Wr.PutText(stderr, progName & ": Rd.EndOfFile\n");
    | Scan.BadFormat =>
          Wr.PutText(
              stderr, "Usage: " & progName & " " & UsageString & "\n");
    | Modules.NotFound, System.Error, ProgFileIO.SyntaxError (ErrorText) =>
          Wr.PutText(stderr, progName & ": " & ErrorText & "\n");
    END;                         (*TRY*)
END emake.
