(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)

(* Created by Jorge Stolfi on Sat Oct  7  2:14:47 PDT 1989     *)
(* Last modified on Tue Jun  2 13:10:05 PDT 1992 by stolfi     *)
(*      modified on Tue Feb 11 21:39:43 PST 1992 by muller     *)


INTERFACE LDW;

(* A perceptual color model.

  The Euclidean distance between two RGB triples is a poor indicator
  of their distinctiveness to the human eye, beacuse the sensitivity
  of the latter varies considerably from point to point of RGB cube.
  This interface defines an alternative color coordinate system, the
  LDW space, in which Euclidean distances are more nearly proportional
  to perceptual differences.

  This color space is somewhat similar (but not identical) to the
  CIE La*b* space.

  Index: colors; graphics, colors; display, color.
*)

IMPORT Color, RGB;

TYPE
  T = Color.T; (* ARRAY [0..2] OF REAL; *)
  (* 
    An LDW.T value describes a color by the three attributes Lightness
    (l), reDness (d), and yelloWness (w), all real numbers.  Lightness
    is a non-linear (log-like) function of RGB.Brightness, normalized
    so that Black has l=0.0 and White has l=1.0.  Redness and Yellowness
    are non-linear choma coordinates, orthogonal to RGB.Brightness;
    they are non-linear functions of the P and Q coordinates (see the
    YPQ interface), and range on some sub-interval of [-1.0 _ +1.0] *)

PROCEDURE LDWFromRGB (READONLY rgb: RGB.T): T;
PROCEDURE RGBFromLDW (READONLY ldw: T): RGB.T;
  (*
    Conversion between RGB and LDW coordinates. *)

PROCEDURE YFromL (l: REAL): REAL;
PROCEDURE LFromY (y: REAL): REAL;
  (*
    Conversion between perceptual lightness (l) and linear brightness 
    (CIE Y coordinate). *)

PROCEDURE LFromRGB (READONLY rgb: RGB.T): REAL;
  (* 
    Lighness (perceptual brightness) of an RGB value.
    
    This number is a non-linear monotonic function of brightness
    such that L(Black)=0, L(White)=1, and a series of gray levels at
    equal L increments will look equally spaced.  *)

PROCEDURE RGBFromL (l: REAL): RGB.T;
  (*
    A grey RGB tuple with given lightness value *)

PROCEDURE DFromRGB (READONLY rgb: RGB.T): REAL;
  (* 
    A perceptual measure of color saturation in the red-green direction. *)

PROCEDURE WFromRGB (READONLY rgb: RGB.T): REAL;
  (* 
    A perceptual measure of color saturation in the yellow-blue direction *)

PROCEDURE ChromaFromRGB (READONLY rgb: RGB.T): REAL;
  (* 
    A perceptual measure of saturation, defined as sqrt(W^2 + D^2). *)

PROCEDURE Dist(READONLY a, b: T; lumWeight: REAL := 1.0): REAL;
  (*
    Perceptual distance between LDW triples, defined as 
    Euclidean distance between /a/ and /b/, with the 
    lightness component scaled by /lumWeight/:
|      sqrt( lumWeight^2*(a[0]-b[0])^2 + (a[1]-b[1])^2 + (a[2]-b[2])^2).
    *)

CONST Undefined = Color.Undefined;
    (*
      The "undefined" LDW triple *)
    
CONST Default = Color.Default;

TYPE Bytes = Color.Bytes;

CONST (*PROCEDURE*) ToBytes = Color.ToBytes;
(* A safe LOOPHOLE of /x/ into ARRAY OF BYTE. *)

END LDW.
