(* Copyright 1989 by AT&T Bell Laboratories *)
(* Copyright 1991 by Carnegie Mellon University *)

(* This is a modified version of process.sml from version 0.73 of the
   compiler.

   Modifications:
   10/7/91 (DRT):
        * Modified process to apply a transformation function to 
	  the name of a module to yield the final module name to use.
        * Modified bootenv to
	        - load but not generate code for the math structure.
		  This structure is being created in the runtime using
	          C math functions.
                - load and generate code for the Loader structure.
*)
     
signature CPROCESSFILE =
sig
  datatype result = RESULT of {entrypoints : string list,imports : string list}
  exception Stop
  val dumpMap : unit -> unit
  val prLambda : unit -> unit
  val prFun : int -> unit
  val process : {pervasive : Modules.env option, (* for imports *)
                 env : Modules.env,       (* for regular compilation *)
                 dest : outstream,
                 src:  string,
		 gencode : (CPS.function * ErrorMsg.complainer * string *
			    outstream -> unit) option,
		 cc : string -> unit,
		 transform : string -> string} -> Modules.env * result
  val load : (string -> string) -> Modules.env -> string -> Modules.env * result
  val primeEnv : Modules.env
  val getCore : unit -> int list
  val bootEnv : (Modules.env -> string -> Modules.env * result) ->
                    (Modules.env * (int * int * int) * result)
end

functor CProcessFile(structure Print : CPRINT) : CPROCESSFILE =

struct
 open Access Modules Types Variables PrintUtil ModuleUtil NewParse
      PrintBasics

 structure CGoptions = System.Control.CG

     fun debugmsg  (msg : string) =
	let val printit = !System.Control.debugging
	in  if printit then (print msg; print "\n")
	    else ();
	    printit
	end

  exception Stop

  fun timemsg (s : string) =
    if !System.Control.timings then (print s; newline(); true) else false

  val saveLambda = System.Control.saveLambda
  val lambda = ref (Lambda.RECORD [])
  (* really needed only for interactive version *)
  val _ = System.Control.prLambda := (fn () => (MCprint.printLexp (!lambda); newline()))
  fun prLambda() = (MCprint.printLexp(!lambda); newline())
  fun prFun lv = (MCprint.printFun(!lambda) lv; newline())

  open ErrorMsg BareAbsyn Lambda System.Timer

  fun for l f = app f l
  val update = System.Stats.update

  (* lvar -> string environment used by batch compiler to map module
     lvars to names of modules *)
  exception Modname
  val m : string Intmap.intmap = Intmap.new(32, Modname)
  val lookup = Intmap.map m
  val enterName = Intmap.add m
  fun lookupName v =
      lookup v 
      handle Modname => 
	 ErrorMsg.impossible ("Bad free variable: " ^ Access.lvarName v)
  fun dumpMap() =
      let fun p(i:int,s:string) = (print i; print " -> "; print s; print "\n")
      in  print "lvar -> structure mapping:\n"; Intmap.app p m
      end

  val is_core = ref false;

  fun getCore () = if !is_core then [] else tl(!CoreInfo.stringequalPath)

  structure Importer = CImporter(structure FilePaths = UnixPaths()
				 val fileExtension = ".c"
				 structure Link : LINK =
				     struct
				       type info = lvar * string
				       val project = fn (a,_) => a
				       val save = enterName
				       val rename = fn (n,(a,b)) => (n,b)
				     end)

  datatype result = RESULT of {entrypoints : string list,imports : string list}

  local
     val empty = RESULT{entrypoints=nil,imports=nil}
 in
     val compress = fn l =>
      fold (fn (RESULT {entrypoints=a,imports=b},
		RESULT {entrypoints=c,imports=d}) =>
                    RESULT{entrypoints=a@c,imports=b@c}) l empty
  end

  fun process({dest,pervasive,env,src,
	       gencode, transform, cc}) : Modules.env * result =
      let 
          val _ = print ("[opening " ^ src ^ "]\n")
	  val stream = open_in src
	  val inputSource as {anyErrors,...} =
			 ErrorMsg.newSource(src,stream,false,std_out,
					    Index.openIndexFile src)
	  val parser = NewParse.parse inputSource

	 fun convert lam =
		let val timer = start_timer()
	     	    val lam = Reorder.reorder lam
		    val function = Convert.convert lam
	     	    val time = check_timer timer
	     	in  update(System.Stats.convert,time);
	     	    timemsg("convert, " ^ makestring time ^ "s")
	     		  orelse debugmsg "convert";
	     	    function
	     	end
       
         exception Eof
       
         fun transStrb env sb =
	     	let val timer = start_timer()
	     	    val (sb,profil) = Prof.instrumStrb sb
	     	    val Absyn.STRB{strvar=STRvar{access=PATH[v],...},...} = sb
	     	    val lam = Translate.transDec env inputSource
					(Absyn.STRdec[sb]) (Lambda.VAR v)
	     	    val lam = Prof.bindLambda(lam,profil)
	     	    val time = check_timer timer
	     	 in update(System.Stats.translate,time);
	     	    timemsg("translate, " ^ makestring time ^ "s")
	     		  orelse debugmsg "translate";
	     	    if !anyErrors then raise Stop else ();
	     	    lam
	     	end
	       
         fun transFctb env fb =
	     	let val timer = start_timer()
	     	    val (fb,profil) = Prof.instrumFctb fb
	     	    val Absyn.FCTB{fctvar=FCTvar{access=PATH[v],...},...} = fb
	     	    val lam = Translate.transDec env inputSource
				(Absyn.FCTdec[fb]) (Lambda.VAR v)
	     	    val lam = Prof.bindLambda(lam,profil)
	     	    val time = check_timer timer
	     	 in update(System.Stats.translate,time);
	     	    timemsg("translate, " ^ makestring time ^ "s")
	     		  orelse debugmsg "translate";
	     	    if !anyErrors then raise Stop else ();
	     	    lam
	     	end

	  fun cleanup() = (print("[closing " ^ src ^ "]\n");
			   close_in stream)

	  fun proc (src,dest) (name,lvar,mkLam) : Access.lvar * string =
	    let fun complain _ s = (print(src^": "^s ^ "\n"); raise Stop)
		val name = transform name
	    in enterName(lvar, name);
	        case gencode of
		  NONE => ()
	        | SOME gencode =>
		 let val lam = Opt.closestr(lookupName, mkLam(), getCore())
		 in debugmsg "closed";
		    if !saveLambda then lambda := lam else ();
		    gencode(convert lam, complain, name, dest);
		    if !anyErrors then raise Stop else ()
		 end;
               (lvar,name)
            end

	  fun importcodegen (absyn,env,src) =
	        let val dest = open_out (src^".c")
		    val _ = Print.startFile (outputc dest)
		    val proc' = proc(src,dest)
		    fun f (SEQdec decs) = fold (op @) (map f decs) nil
		      | f (MARKdec(d,_,_)) = f d
		      | f (SIGdec _) = nil
		      | f (FCTdec fbs) =
                         map (fn (b as FCTB{fctvar=FCTvar{name,access=PATH[v],
							 ...},...}) =>
			      proc'(Symbol.name name, v,
				    fn ()=>transFctb env b)
			      ) fbs
		      | f _ = ErrorMsg.impossible "cprocess/179"
               in (f absyn handle e => (close_out dest; raise e))
		  before (close_out dest; cc src)
	       end

	  val empty = (nil,nil)

          fun flatten (l : 'a list list) : 'a list = fold (op @) l nil

          (* compress: convert a list of list pairs to a pair of lists *)

          fun compress (l : ('a list * 'b list) list) : 'a list * 'b list =
	      (flatten (map #1 l),flatten (map #2 l))

	  val proc' = #2 o proc(src,dest)

	  val env' = ref env

	  fun g absyn : string list * string list =
	    let val pr = fn () =>
	       PrintDec.printDec (!env') absyn (fn _ => impossible "Process.f")
	     in case absyn
		of (SEQdec decs) => compress (map g decs)
		 | (MARKdec(d,_,_)) => g d
		 | (SIGdec sl) => (pr (); empty)
		 | (OPENdec _) => (pr (); empty)
		 | (STRdec sbs) =>
		     (pr ();
		      (map (fn sb as
			  STRB{strvar as STRvar{name=n,access=PATH[v],...},...} =>
			    let val mkLam = fn () => transStrb (!env') sb
			    in  proc'(Symbol.name n, v, mkLam)
			    end) sbs,nil))
		| (ABSdec sbs) =>
		     (pr ();
		      (map (fn sb as
			   STRB{strvar as STRvar{name=n,access=PATH[v],...},...} =>

			       let val mkLam = fn () => transStrb (!env') sb
			      in  proc'(Symbol.name n, v, mkLam)
			      end) sbs,nil))
	        | (FCTdec fbs) =>
		       (pr ();
			(map (fn fb as
			   FCTB{fctvar as FCTvar{name,access=PATH[v],...},...} =>
			    let val mkLam = fn () =>
				transFctb (!env') fb
				handle Match =>
				    impossible "transFctb: match exception"
			    in  (proc'(Symbol.name name, v, mkLam))
			    end) fbs,nil))
	       | IMPORTdec fnames =>
		    let val stdPervEnv = 
			 case pervasive
			 of SOME e => e
	 	          | NONE => (print "Illegal import while booting!";
				     raise Stop)
			val r = map (fn n => Importer.getModule(stdPervEnv,
					    importcodegen,n)) fnames
			val entrypts = map #2 (flatten (map #2 r))
		        val imports = flatten (map #3 r)
		   in env' := revfold (fn (env,accenv) => Env.atop(accenv,env))
			              (map #1 r) (!env');
		      (entrypts,imports)
		   end
               | _ => (ErrorMsg.complain
		          "signature, functor, or structure expected";
		       empty)
              end

              fun loop (pair as (entrypts,imports)): Modules.env * result =
		  case parser(!env')
		  of EOF => (cleanup();
			     if !anyErrors
			     then raise Stop   
			     else (!env',RESULT{entrypoints=entrypts,
						imports=imports}))
		   | ABORT => raise Stop
		   | ERROR => raise Stop
		   | PARSE (x,envr) => 
			 (env' := Env.atop(envr,!env');
			  loop (compress ([pair,g x])))
      in loop empty handle e => (cleanup(); raise e)
      end

  fun load transform env fname =
         process{dest=std_out,cc=fn _ => (),env=env,pervasive=NONE,
		 src=fname,gencode=NONE,transform=transform}

  val load' = load (fn i => i) 

 (* initializing static environment *)

 (* priming structures: PrimTypes and InLine *)
  val nameofPT = Symbol.strSymbol "PrimTypes"
  val varofPT = STRvar{name=nameofPT,access=PATH[0],binding=Prim.primTypes}
  val nameofIL = Symbol.strSymbol "InLine"
  val varofIL = STRvar{name=nameofIL,access=PATH[0],binding=Prim.inLine}

  val primeEnv = Env.bind(nameofIL,STRbind varofIL, 
			  Env.bind(nameofPT,STRbind varofPT,
				   openStructureVar (Env.empty,varofPT)))

  fun bootEnv (loader:env -> string -> env * result) =
    let val err = fn _ => impossible "bootEnv"
	val sigSymbols =
           map Symbol.sigSymbol ["REF","LIST","ARRAY","BYTEARRAY","IO","BOOL",
 				 "ENVIRON", "COMPILE", 
		                 "STRING","INTEGER","REAL","GENERAL"]
        val NJsymbol = Symbol.strSymbol "NewJersey"
        val signatures = !System.Control.Print.signatures
	val _ = System.Control.Print.signatures := 0
        val _ = CoreInfo.resetCore();
        val (env,_) = load' primeEnv "boot/assembly.sig"
	val (env,coresml) = (is_core := true;  loader env "boot/core.sml" 
				handle e => (is_core := false; raise e))
        val _ = is_core := false;
        val (env,_) = load' env "boot/dummy.sml";
        val markabs = !System.Control.markabsyn
		       before System.Control.markabsyn := false
	val svCore as STRvar{access=PATH[lvCore],...} =
	         lookSTR (env,[Symbol.strSymbol "Core"],err)
	val _ = CoreInfo.setCore(env,[Symbol.strSymbol "Core"]);
  	val (env,_) = load' env "boot/perv.sig";
	val (env,_) = load' env "boot/system.sig";

(* don't generate code for math library; we're using the C math library *)

	val (env,mathsml) = load' env "boot/math.sml";
	val (env,pervsml) = loader env "boot/perv.sml";
	val (env,_) = load' env "boot/overloads.sml";

(* add the loader *)

        val (_,loadersml) = loader env "c/loader.sml";
	val _ = System.Control.Print.signatures := signatures
	val _ = System.Control.markabsyn := markabs;
	val STRvar{access=PATH[lvMath],...} =
		     lookSTR (env,[Symbol.strSymbol "Math"],err)
	and svInitial as STRvar{access=PATH[lvInitial],
			         binding=strInitial,...} =
		     lookSTR (env,[Symbol.strSymbol "Initial"],err)
        and overLoads = lookSTR (env,[Symbol.strSymbol "Overloads"],err)
	val env' = openStructureVar (openStructureVar(Env.empty,svInitial),
	                             overLoads)
	val env' = fold (fn (name,e) => Env.bind(name,Env.look(env,name),e))
                 sigSymbols env'
	val env' = Env.bind
			(NJsymbol, 
			 STRbind(STRvar{name=NJsymbol,access=PATH[lvInitial],
				        binding=strInitial}),
			 env')
     in  (env',(lvCore,lvInitial,lvMath),compress[coresml,pervsml,loadersml])
     end handle Cascade s => (print("Compiler Bug: "^s^"\n");
				 raise Stop)
end (* structure ProcessFile *)
