(* Copyright 1989 by AT&T Bell Laboratories *)
(* Copyright (c) 1991 by Carnegie Mellon University *)


(* Modifications:

   8/6/90 (DRT) Modified bootEnv to generate a Loader object file also, but
          not place it in the pervasive environment.
   8/6/90 (DRT) process modified to apply a transformation function to the 
          name of a module to yield the final file name to use.
   8/6/90 (DRT) Modified bootEnv to add environment information for math
          and not to generate code.  This structure is being created in the
          runtime using C math functions.
 10/10/90 (DRT) Modified bootEnv: BASICIO doesn't exist in pervasives
          anymore.
*)

signature CPROCESSFILE =
sig
  datatype result = RESULT of {entrypoints : string list,imports : string list}
  structure Env : ENV
  exception Stop
  val process : {env : Env.env option,
		 dest : outstream,
		 src : string,
		 gencode : (CPS.function * ErrorMsg.complainer * string *
		           outstream -> unit) option,
		 cc : string -> unit,
		 transform : string -> string} -> result
  val load : (string -> string) -> string -> result
  val bootEnv : (string -> result) -> result
end

functor CProcessFile(structure Print : CPRINT) : CPROCESSFILE =
struct
  open Access Basics PrintUtil EnvAccess NewParse
  open ErrorMsg BareAbsyn Lambda System.Timer

   fun debugmsg  (msg : string) =
    let val printit = !System.Control.debugging
    in  if printit then (print msg; print "\n")
	else ();
	printit
    end

  exception Stop

  fun timemsg (s : string) =
    if !System.Control.timings then (print s; newline(); true) else false

  val update = System.Stats.update
  val printDepth = System.Control.Print.printDepth

  (* lvar -> string environment used by batch compiler to map module
     lvars to names of modules *)

  exception Modname
  val m : string Intmap.intmap = Intmap.new(32, Modname)
  val lookup = Intmap.map m
  val enterName = Intmap.add m
  fun lookupName v =
      lookup v 
      handle Modname => 
	 ErrorMsg.impossible ("Bad free variable: " ^ Access.lvarName v)
(*
  fun dumpMap() =
      let fun p(i:int,s:string) = (print i; print " -> "; print s; print "\n")
      in  print "lvar -> structure mapping:\n"; Intmap.app p m
      end
*)
  structure Importer = CImporter(structure FilePaths = UnixPaths()
				 val fileExtension = ".c"
				 structure Link : LINK =
				     struct
				       type info = lvar * string
				       val project = fn (a,_) => a
				       val save = enterName
				       val rename = fn (n,(a,b)) => (n,b)
				     end)

  val is_core = ref false;

  fun getCore () = if !is_core then [] else tl(!CoreInfo.stringequalPath)

  datatype result = RESULT of {entrypoints : string list,imports : string list}

  local
     val empty = RESULT{entrypoints=nil,imports=nil}
 in
     val compress = fn l =>
      fold (fn (RESULT {entrypoints=a,imports=b},
		RESULT {entrypoints=c,imports=d}) =>
                    RESULT{entrypoints=a@c,imports=b@c}) l empty
  end

  fun process {dest,env,src,gencode,transform,cc} : result =
      let 
          val _ = print("[opening " ^ src ^ "]\n")
          val stream = open_in src
	  val inputSource as {anyErrors,...} =
			 ErrorMsg.newSource(src,stream,false,std_out)
	  val parser = NewParse.parse inputSource
	  fun convert lam =
		let val timer = start_timer()
	     	    val lam = Reorder.reorder lam
		    val function = Convert.convert lam
	     	    val time = check_timer timer
	     	in  update(System.Stats.convert,time);
	     	    timemsg("convert, " ^ makestring time ^ "s")
	     		  orelse debugmsg "convert";
	     	    function
	     	end
       
         fun transStrb sb =
	     	let val timer = start_timer()
	     	    val (sb,profil) = Prof.instrumStrb sb
	     	    val Absyn.STRB{strvar=STRvar{access=LVAR v,...},...} = sb
	     	    val lam = Translate.transDec inputSource
					(Absyn.STRdec[sb]) (Lambda.VAR v)
	     	    val lam = Prof.bindLambda(lam,profil)
	     	    val time = check_timer timer
	     	 in update(System.Stats.translate,time);
	     	    timemsg("translate, " ^ makestring time ^ "s")
	     		  orelse debugmsg "translate";
	     	    if !anyErrors then raise Stop else ();
	     	    lam
	     	end
	       
         fun transFctb fb =
	     	let val timer = start_timer()
	     	    val (fb,profil) = Prof.instrumFctb fb
	     	    val Absyn.FCTB{fctvar=FCTvar{access=LVAR v,...},...} = fb
	     	    val lam = Translate.transDec inputSource
				(Absyn.FCTdec[fb]) (Lambda.VAR v)
	     	    val lam = Prof.bindLambda(lam,profil)
	     	    val time = check_timer timer

	     	 in update(System.Stats.translate,time);
	     	    timemsg("translate, " ^ makestring time ^ "s")
	     		  orelse debugmsg "translate";
	     	    if !anyErrors then raise Stop else ();
	     	    lam
	     	end

	  val _ = Env.commit()
	  fun cleanup() = (print("[closing " ^ src ^ "]\n");
			   close_in stream)

	  fun proc (src,dest) (name,lvar,mklam) : lvar * string =
             let fun complain _ s = (print(src ^ ": "^ s ^ "\n"); raise Stop)
		 val name = transform name
	     in enterName(lvar,name);
		case gencode
	        of NONE => ()
	         | SOME gencode =>
	            let val lam = Opt.closestr(lookupName,mklam(),getCore())
	            in gencode(convert lam, complain, name, dest);
	               if !anyErrors then raise Stop else ()
		    end;
		(lvar,name)
             end

	  fun importcodegen (absyn,src) =
	        let val dest = open_out (src^".c")
		    val _ = Print.startFile (outputc dest)
		    val proc' = proc(src,dest)
		    fun f (SEQdec decs) = fold (op @) (map f decs) nil
		      | f (MARKdec(d,_,_)) = f d
		      | f (SIGdec _) = nil
		      | f (FCTdec fbs) =
                         map (fn (b as FCTB{fctvar=FCTvar{name,access=LVAR v,
							 ...},...}) =>
			      proc'(Symbol.name name, v,fn ()=>transFctb b)) fbs
		      | f _ = ErrorMsg.impossible "cprocess/179"
               in (f absyn handle e => (close_out dest; raise e))
		  before (close_out dest; cc src)
	       end

	  val empty = (nil,nil)
          val compress = fn l => (fold (op @) (map (fn (a,b) => a) l) nil,
				  fold (op @) (map #2 l) nil)
	  val proc' = proc(src,dest)
          val rec g =
	      fn SEQdec decs => compress (map g decs)
               | (absyn as SIGdec _) =>
		      (PrintAbsyn.printDec(absyn,0,!printDepth);
		       newline();
		       empty)
               | (absyn as OPENdec _) => 
		      (PrintAbsyn.printDec(absyn,0,!printDepth);
		       newline();
		       empty)
	       | STRdec sbs =>
		     (map (fn (b as STRB{strvar=STRvar{name=[n],access=LVAR v,
						      ...},...}) =>
		                (print "structure "; printSym n; newline();
				 proc'(Symbol.name n, v,fn () => transStrb b))) sbs,
		      nil)
	       | ABSdec sbs =>
		     (map (fn (b as STRB{strvar=STRvar{name=[n],access=LVAR v,
						      ...},...}) =>
		                (print "abstraction "; printSym n; newline();
				 proc'(Symbol.name n, v,fn () => transStrb b))) sbs,
		      nil)
               | FCTdec fbs =>
		     (map (fn (b as FCTB{fctvar=FCTvar{name,access=LVAR v,...},
				        ...}) =>
			        (print "functor "; printSym name; newline();
				 proc'(Symbol.name name, v, fn () => transFctb b))) fbs,
		      nil)
	       | IMPORTdec fnames =>
		     
		   (compress (map (fn n => Importer.getModule(
                                            case env of SOME e => e
						      | NONE => (print "Illegal import while booting!"; raise Stop),
					    importcodegen,n)) fnames)
		    handle Importer.Import =>
			           (print "IMPORT failed\n"; raise Stop))
	       | MARKdec(d,_,_) => g d
               | _ => (ErrorMsg.complain
			   "signature, functor, or structure expected"; empty)

              fun loop pair : (lvar * string) list * string list =
		  case parser()
		  of EOF => (cleanup();
		             if !anyErrors
		             then (Env.restore(); raise Stop)
		             else (Env.consolidate(); pair))
		   | ABORT => raise Stop
		   | ERROR => raise Stop
		   | PARSE x => loop (compress [pair,g x])
      in case loop empty
	 of (info,imports) =>
	   RESULT{entrypoints=map #2 info,imports=imports} handle e =>
	           (Env.restore(); cleanup(); raise e)
      end

  fun load transform fname =
         process{dest=std_out,cc=fn _ => (),env=NONE,src=fname,gencode=NONE,
		 transform=transform}

  val load' = load (fn i => i) 

 (* initializing static environment *)

 (* priming structures: PrimTypes and InLine *)
  val nameofPT = Symbol.symbol "PrimTypes"
  val varofPT = STRvar{name=[nameofPT],access=LVAR 0,binding=Prim.primTypes}
  val varofPT' = STRvar{name=[nameofPT],access=PATH[0],binding=Prim.primTypes}
  val nameofIL = Symbol.symbol "InLine"
  val varofIL = STRvar{name=[nameofIL],access=LVAR 0,binding=Prim.inLine}

  fun primeEnv() =
      (Env.reset();
       openStructureVar varofPT';
       bindSTR(nameofPT,varofPT);
       bindSTR(nameofIL,varofIL);
       ())

  fun bootEnv (loader : string -> result) : result =
      let val _ = primeEnv();
	  val _ =  CoreInfo.resetCore();
	  val _ = load' "boot/assembly.sig"
	  val _ = is_core := true
	  val coresml = loader "boot/core.sml" handle e => (is_core := false;
							    raise e)
	  val _ =  is_core := false
          val _ = load' "boot/dummy.sml"
          val markabs = !System.Control.markabsyn
			before System.Control.markabsyn := false
	  val svCore = lookSTR (Symbol.symbol "Core")
          val _ = CoreInfo.setCore(svCore);
	  val _ =  load' "boot/perv.sig";
	  val _ = load' "boot/system.sig";

(* don't generate code for math libarary; we're using the C math library *)

 	  val mathsml = load' "boot/math.sml";
	  val pervsml = loader "boot/perv.sml";
	  val _ = load' "boot/overloads.sml";

(* add the loader *)

          val loadersml = loader "c/loader.sml";
	  val _ = System.Control.markabsyn := markabs
	  val svInitial as STRvar{access=PATH[lvInitial],
				  binding=strInitial as STRstr{table,...},
				  ...} = lookSTR (Symbol.symbol "Initial")
	  and STRvar{binding=STRstr{table=otable,...},...} =
		     lookSTR (Symbol.symbol "Overloads")
	  val sigs = map (fn s => lookSIG(Symbol.symbol s))
			      ["REF","LIST","ARRAY","BYTEARRAY",
			       "IO","BOOL","STRING","INTEGER","REAL","GENERAL"]
	  val NJsymbol = Symbol.symbol "NewJersey"
       in Env.reset();
	        (* merge overload bindings into Initial's symtable *)
	       IntStrMap.app (IntStrMap.add table) otable;
	       openStructureVar(svInitial);
	       app (fn (sgn as SIGvar{name,...}) => bindSIG(name,sgn))
		   sigs;
	       bindSTR(NJsymbol, STRvar{name=[NJsymbol],access=LVAR(lvInitial),
				        binding=strInitial});
	   compress [coresml,pervsml,loadersml]
       end handle Cascade s => (print("Compiler Bug: "^s^"\n");
				 raise Stop)
end (* structure CProcessFile *)
