(*  Title: conv
    Author: Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1988  University of Cambridge

Conversions: meta-level rewriting rules and tactics

From Cambridge LCF;  see also

L C Paulson, A higher-order implementation of rewriting,
Science of Computer Programming 3 (1983), pages 119-149.

part_tmatch should check TYPES: otherwise ?a==t matchs terms of diff types,
	resulting in exn THM ("instantiate: types", 0, [?])
Should check that rhs is not an instance of lhs in lhs==rhs! (LOOPING rewrites)
*)


infix thenc orelsec;

signature CONV = 
sig
  structure Tactical: TACTICAL
  local open Tactical Tactical.Thm 
  in
    type conv
    exception CONV
    val all_conv: conv
    val beta_conv: conv
    val changed_conv: conv -> conv
    val conv_cterm: conv -> Sign.cterm -> thm
    val depth_conv: conv -> conv
    val every_conv: conv list -> conv
    val fconv_rule: conv -> thm -> thm
    val first_conv: conv list -> conv
    val fold_tac: thm list -> tactic
    val goals_conv: conv -> conv
    val immed_conv: thm list -> conv
    val no_conv: conv
    val orelsec: conv*conv->conv
    val part_tmatch: (term -> term) -> thm -> {sign: Sign.sg, nextfree: int} -> term -> thm
    val redepth_conv: conv -> conv
    val repeatc: conv -> conv
    val rewrite_conv: thm->conv
    val rewrite_cterm: thm list -> Sign.cterm -> thm
    val rewrite_goals_rule: thm list -> thm -> thm
    val rewrite_goal_tac: thm list -> int -> tactic
    val rewrite_goals_tac: thm list -> tactic
    val rewrite_rule: thm list -> thm -> thm
    val rewrite_tac: thm list -> tactic
    val sub_conv: conv -> conv
    val thenc: conv*conv->conv
    val top_depth_conv: conv -> conv
    val top_immed_conv: thm list -> conv
    val try_conv: conv -> conv
  end
end;



functor ConvFun (structure Logic: LOGIC and Tactic: TACTIC) : CONV = 
struct
structure Tactical = Tactic.Tactical;
local open Tactic Tactic.Tactical Tactic.Tactical.Thm 
in


(*A "crec" contains data needed during conversion: signature and var# *)
type convrec = {sign: Sign.sg, nextfree: int};

(*A conversion is applied to a "crec" and a term.*)
type conv = convrec -> term -> thm;

exception CONV;

(*Match a given part of the theorem "th" to a term, instantiating the theorem.
  The part should be free in "th", which is typically a rewrite rule. 
  Combines the signatures of "th" and the term (via convrec) *)
fun part_tmatch partfn th (crec: convrec) t =
    let val {sign=thsign,prop,...} = rep_thm th
	and {sign,...} = crec;
	val instl = term_match ([], partfn prop, t);
	val ctermf = Sign.cterm_of (Sign.merge (sign,thsign));
	val cinstl = map (pairself ctermf) instl
    in  instantiate cinstl th  end
    handle TERM_MATCH => raise CONV;



(*Conversion for rewrite rules of the form |- t == u
   matches x1 .. xn :  t' -> |- t' == u *)
val rewrite_conv = part_tmatch (fst o Logic.dest_equals);


(*Perform beta conversion -- and nothing else*)
val beta_conv : conv =
    fn {sign,...} => fn t => 
	case t of
	    (Abs(_,_,_) $ _) => beta_conversion (Sign.cterm_of sign t)
	  | _ => raise CONV;


(*Conversion that always fails, identity element for orelsec *)
fun no_conv _ = raise CONV;

(*Conversion that always succeeds, using reflexive law : t ---> |- t==t*)
val all_conv : conv =
    fn {sign,...} => fn t => reflexive (Sign.cterm_of sign t);


(* apply two conversions in succession, fail if either does*)
fun op thenc (cv1, cv2) (crec: convrec) t =
    let val th1 = cv1 crec t;
	val {sign=sign1, prop=prop1,...} = rep_thm th1
        and {nextfree,...} = crec;
	val u = snd (Logic.dest_equals prop1) 
    in  transitive th1 (cv2 {sign=sign1, nextfree=nextfree} u)  end;

(* apply cv1, if it fails then apply cv2*)
fun op orelsec (cv1, cv2) crec t  =  cv1 crec t handle CONV => cv2 crec t;


(* perform the first successful conversion of those in the list*)
fun first_conv cvs = itlist_right (op orelsec) (cvs,no_conv);

(* perform every conversion in the list *)
fun every_conv cvs = itlist_right (op thenc) (cvs,all_conv);


fun repeatc cv crec = ((cv thenc (repeatc cv)) orelsec all_conv) crec;


(*Cause the conversion to fail if it does not change its input*)
fun changed_conv cv crec t = 
    let val th = cv crec t
	val (l,r) = Logic.dest_equals (#prop(rep_thm th))
    in  if l aconv r then raise CONV else th  end;


(*Apply the conversion if possible*)
fun try_conv cv = cv orelsec all_conv;


(* apply conv to all top-level subterms of term*)
fun sub_conv cv crec = fn
    (Abs(a,T,t)) =>
      let val {sign,nextfree} = crec;
	  val v = Free("x" ^ string_of_int nextfree, T);
	  val thu = cv {sign=sign, nextfree=nextfree+1}
		       (subst_bounds ([v], t))
      in  abstract_rule a (Sign.cterm_of sign v) thu  end
  | (t$u) => combination (cv crec t) (cv crec u)
  | t => all_conv crec t;


(*In [A1,...,An]==>B apply cv to the A's only -- for rewrite_goals_tac*)
fun goals_conv cv crec = fn
    imp$A$B => if imp=implies then 
	let val thA = cv crec A
	    and thB = goals_conv cv crec B
	    and cimp = all_conv crec implies
	in  combination (combination cimp thA) thB  end
      else all_conv crec (imp$A$B)
  | B => all_conv crec B;


(*In [A1,...,An]==>B apply cv to Ai only*)
fun goal_conv cv i crec = fn
    imp$A$B => if imp=implies then 
	let val thA = (if i=1 then cv else all_conv) crec A
	    and thB = goal_conv cv (i-1) crec B
	    and cimp = all_conv crec implies
	in  combination (combination cimp thA) thB  end
      else all_conv crec (imp$A$B)
  | B => all_conv crec B;


(*Apply a conversion recursively to a term and its parts. *)
fun depth_conv cv x =
   (sub_conv (depth_conv cv) thenc (repeatc cv)) x;

(*Like depth_conv, but retraverses after each conversion
  -- diverges if the conversion never fails. *)
fun redepth_conv cv x =
   (sub_conv (redepth_conv cv) thenc
	((cv thenc (redepth_conv cv)) orelsec all_conv)) x;

(*Try conversions at top level before descending.
  Not really normal order, but may terminate where redepth_conv would not.
  More efficient than redepth_conv for rewrites like fst(x,y)=x *)
fun top_depth_conv cv x =
   (repeatc cv thenc (try_conv
	(changed_conv (sub_conv (top_depth_conv cv)) thenc
	  try_conv (cv thenc top_depth_conv cv)))) x;


(*Immediate conversions: rewrite rules, beta-conversion.*)
fun immed_conv rths = 
    first_conv (map rewrite_conv rths)  orelsec  beta_conv;

(*rewrite a formula using a list of theorems *)
fun top_immed_conv rths = top_depth_conv (immed_conv rths);

(*Use a conversion to transform a theorem*)
fun fconv_rule cv th =
    let val {sign,prop,...} = rep_thm th
 	val eqth = cv {sign=sign, nextfree=0} prop
    in  equal_props eqth th  end;

(*Use a conversion to transform a term*)
fun conv_cterm cv ct = 
    let val {sign,t,...} = Sign.rep_cterm ct
    in  cv {sign=sign, nextfree=0} t  end;

(*Rewrite a theorem*)
fun rewrite_rule rths = fconv_rule (top_immed_conv rths);

(*Rewrite a term*)
fun rewrite_cterm rths = conv_cterm (top_immed_conv rths);  

(*Rewrite the subgoals of a proof state (represented by a theorem) *)
fun rewrite_goals_rule rths = fconv_rule (goals_conv (top_immed_conv rths));

(*Rewrite subgoal i of a proof state*)
fun rewrite_goal_rule rths i = fconv_rule (goal_conv (top_immed_conv rths) i);


(*Tactic:  Rewrite throughout proof*)
fun rewrite_tac rths = PRIMITIVE (rewrite_rule rths);

(*Tactic:  Rewrite subgoals only*)
fun rewrite_goals_tac rths = PRIMITIVE (rewrite_goals_rule rths);

(*Tactic:  Rewrite subgoal i only.
  SELECT_GOAL o rewrite_goals_rule does not work unless subgoal has
    parameters!*)
fun rewrite_goal_tac rths i = PRIMITIVE (rewrite_goal_rule rths i);

(*Tactic:  Fold throughout proof, replacing right side by left*)
fun fold_tac rths = rewrite_tac (map symmetric rths);

end;
end;
