(*  Title: 	HOL/syntax
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1989  University of Cambridge

Natural deduction syntax for classical higher-order logic.
*)

signature HOL_SYNTAX =
sig
  structure Syntax : PURE_SYNTAX
  val Aform: typ
  val Aterm: typ
  val Atype: typ
  val const_decs: (string list * typ) list
  val prin: term -> unit
  val read: string -> term
  val syn: Syntax.syntax
end;

functor HOL_SyntaxFun (Syntax: PURE_SYNTAX) : HOL_SYNTAX = 
struct
structure Syntax = Syntax;
local open Syntax 
in

(*Meta-types for terms, formulae, types*)
val Aterm  = Ground "term";
val Aform  = Ground "form";
val Atype  = Ground "type";

fun qnt_tr q [Free(id,T),A,P] = Const(q,Adummy)$ A $absfree(id,T,P);

fun qnt_tr' q (_$A$Abs(id,T,P)) =
    let val (id',P') = variant_abs(id,T,P)
    in Const(q,Adummy) $ Free(id',T) $ A $ P' end;

fun class_tr [v,A,P] =
	qnt_tr "Lambda" [v,  A,  Const("term", Adummy) $ P];

fun class_tr' (_$A$Abs(id,T,t)) =
    let val (id',t') = variant_abs(id,T,t)
    in  case t' of
	  Const("term",_) $ P' => 
		Const(" class",Adummy) $ Free(id',T) $ A $ P'
	| _ =>	Const(" lam",Adummy) $ Free(id',T) $ A $ t' 
    end;


(** Translation of 'dependent' type operators **)

fun dependent_tr q [Free(id,T),A,B] = Const(q,Adummy) $ A $ absfree(id,T,B);

fun ndependent_tr q [A,B] = 
    Const(q,Adummy) $ A $ Abs("x",Ground "dummy",incr_boundvars 1 B);

(*Is there a dependence or not?*)
fun dependent_tr' (q,r) (_$A$Abs(id,T,B)) =
    if  0 mem (loose_bnos B)
    then 
      let val (id',B') = variant_abs(id,T,B)
      in Const(q,Adummy) $ Free(id',T) $ A $ B' end
    else Const(r,Adummy) $ A $ B;

val mixfix =
 [   (*Propositions*)
  Mixfix("(_)", Aform --> Aprop, "Trueprop", [0], 5),
  Mixfix("(_ /: _)", [Aterm, Atype]--->Aprop, "Elem", [6,6], 5),
      (*Connectives*)
  Mixfix("~_", Aform-->Aform, "Not", [40],40),
  Infixr("&", [Aform,Aform]--->Aform, 35),
  Infixr("|", [Aform,Aform]--->Aform, 30),
  Infixr("-->", [Aform,Aform]--->Aform, 25),
  Infixr("<->", [Aform,Aform]--->Aform, 25),
  Mixfix("(3_ =/ _ :/ _)", [Aterm,Aterm,Atype]--->Aform, "Eq", [20,20,20],20),
  Mixfix("(3PICK _:_./ _)", [SId,Atype,Aform]--->Aterm, " PICK", [], 10), 
  Mixfix("(3ALL _:_./ _)",    [SId,Atype,Aform]--->Aform, " ALL", [], 10),
  Mixfix("(3EX _:_./ _)", [SId,Atype,Aform]--->Aform, " EX", [], 10),
      (*Functions*)
  Mixfix("(3lam _:_./ _)", [SId, Atype, Aterm]--->Aterm, " lam", [], 10),
  Infixl("`", [Aterm,Aterm]--->Aterm, 55),
      (*Classes*)
  Delimfix("(1{|_:_./_|})", [SId,Atype,Aform] ---> Aterm, " class"),
  Infixr("<:", [Aterm,Aterm]--->Aform, 50),
      (*Natural numbers*)
  Delimfix("0", Aterm, "0"),
      (*Types*)
  Mixfix("(3PROD _:_./ _)",    [SId,Atype,Atype]--->Atype, " PROD", [], 10),
  Mixfix("(3SUM _:_./ _)", [SId,Atype,Atype]--->Atype, " SUM", [], 10),
  Infixr("+", [Atype,Atype]--->Atype, 30),
  (*Invisible infixes!*)
  Mixfix("(_ */ _)", [Atype,Atype]--->Atype, " *", [36,35], 35),
  Mixfix("(_ ->/ _)", [Atype,Atype]--->Atype, " ->", [26,25], 25),
  Delimfix("(1{_:_./_})", [SId,Atype,Aform] ---> Atype, " subtype"),
      (*Products*)
  Delimfix("(1<_,/_>)", [Aterm,Aterm]--->Aterm, "Pair")
];

val ext = {logical_types=[Aterm,Aform,Atype], mixfix=mixfix,
	parse_translation=
  [(" PICK", qnt_tr "Pick"),
   (" ALL", qnt_tr "Forall"),
   (" EX", qnt_tr "Exists"),
   (" subtype", qnt_tr "subtype"),
   (" lam", qnt_tr "Lambda"),
   (" class", class_tr),
   (" PROD", dependent_tr "Pi"),
   (" SUM", dependent_tr "Sigma"),
   (" ->", ndependent_tr "Pi"),
   (" *",  ndependent_tr "Sigma")
  ],
print_translation=
  [("Pick", qnt_tr' " PICK"),
   ("Forall", qnt_tr' " ALL"),
   ("Exists", qnt_tr' " EX"),
   ("subtype", qnt_tr' " subtype"),
   ("Lambda", class_tr'),
   ("Pi", dependent_tr'    (" PROD"," ->")),
   ("Sigma", dependent_tr' (" SUM"," *"))
]};

val const_decs = constants mixfix @
  [(["Reduce"], [Aterm, Aterm]--->Aform),
   (["Pick"], [Atype, Aterm-->Aform]--->Aterm),
   (["Forall","Exists"], [Atype, Aterm-->Aform]--->Aform),
      (*Reflection*)
   (["term"], Aform-->Aterm),
   (["form"], Aterm-->Aform),
      (*Types*)
   (["void","unit"], Atype),
   (["subtype"], [Atype, Aterm-->Aform] ---> Atype),
   (["Pi","Sigma"], [Atype,Aterm-->Atype]--->Atype),
      (*Booleans*)
   (["bool"], Atype),
   (["True","False"], Aterm),
   (["cond"], [Atype,Aterm,Aterm,Aterm]--->Aterm),
      (*Natural numbers*)
   (["nat"], Atype),
   (["Succ"], Aterm-->Aterm),
   (["rec"], [Aterm, Aterm, [Aterm,Aterm]--->Aterm] ---> Aterm),
      (*Unions*)
   (["Inl","Inr"], [Atype, Atype, Aterm]--->Aterm),
   (["when"], [Atype,Atype,Atype,Aterm,Aterm-->Aterm,Aterm-->Aterm]--->Aterm),
      (*Products*)
   (["fst","snd"], Aterm-->Aterm),
   (["split"], [Aterm, [Aterm,Aterm]--->Aterm] --->Aterm),
      (*Classes*)
   (["subset"], [Atype,Aterm,Aterm]--->Aform),
   (["un","int"], [Atype,Aterm,Aterm]--->Aterm),
   (["union","inter"], [Atype,Aterm]--->Aterm),
   (["pow"], [Atype,Aterm]--->Aterm),
      (*Functions*)
   (["Lambda"], [Atype, Aterm-->Aterm]--->Aterm)
];

val syn = Syntax.extend pure ext;

fun read a = Syntax.read syn Any a;
fun prin t = Syntax.print_top_level syn t;

end;
end;
