(*  Title: 	HOL/class
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1989  University of Cambridge

Russell's theory of classes in higher-order logic 
Builds provers for Predicate Calculus and Class Theory
*)


(*** Construction of the Predicate Calculus prover ***)

(*Propositional rules*)
val prop_rulepairs =  
  [ (True_intr,"safe_intr"),
    (asm_rl,"safe_elim"),	(*!solves goals by assumption!*)
    (conj_intr,"safe_intr"),
    (disj_cintr,"safe_intr"),
    (imp_intr,"safe_intr"),
    (not_intr,"safe_intr"),
    (iff_intr,"safe_intr"),
    (form_intr,"safe_intr"),
    (False_elim,"safe_elim"),
    (conj_elim,"safe_elim"),
    (disj_elim,"safe_elim"),
    (imp_celim,"safe_elim"),
    (iff_elim,"safe_elim"),
    (form_elim2,"safe_elim"),
    (split_intr,"safe_intr"),
    (split_elim,"safe_elim") ];


(*Quantifier rules*)
val quant_rulepairs =  
  [ (all_intr,"safe_intr"),
    (all_elim,"haz_elim"),
    (exists_intr,"haz_intr"), (exists_cintr,"haz_cintr"),
    (exists_elim,"safe_elim") ];


structure Pc_Rule_tab = struct
    structure Pure = Pure
    val type_rls = type_rls
    val pairs = prop_rulepairs@quant_rulepairs
    end;

structure Pc = ProverFun
	(structure HOL_Lemmas=HOL_Lemmas and Rule_tab=Pc_Rule_tab);


val class_defs =
    [member_def, subset_def, un_def, int_def, union_def, inter_def, pow_def];

(** Subset relation *)

val subset_intr = prove_goal HOL_Rule.thy
    "(!x.[| x:A;  x<:R |] ==> x<:S) ==> R: A->bool ==> subset(A,R,S)"
 (fn asms=>
  [ (rewrite_goals_tac [subset_def]),
    (Pc.fast_tac asms 1) ]);

val subset_intr_prod = prove_goal HOL_Rule.thy
    "(!x y.[| x:A;  y:B;  <x,y> <: R |] ==> <x,y> <: S) ==> \
\    R: A*B->bool ==> subset(A*B,R,S)"
 (fn asms=>
  [ (rewrite_goals_tac [subset_def]),
    (resolve_tac [all_intr] 1),
    (eresolve_tac [sigma_elim] 1),
    (Pc.fast_tac asms 1) ]);


val subset_elim = prove_goal HOL_Rule.thy
    "[| subset(A,R,S);  ~(c<:R) ==> P;  c<:S ==> P;  R:A->bool; c: A |] ==> P"
 (fn asm1::asms=>
  [ (metacut_tac (rewrite_rule [subset_def] asm1) 1),
    (Pc.fast_tac asms 1) ]);


(*Thinning of 0<=A on the left.
  Avoids subset_elim, which would cause a needless case split.*)
val subset_null_elim = prove_goal HOL_Rule.thy
    "[| subset(A, lam x:A.False, S);  P |] ==> P"
 (fn asms=>
  [ (resolve_tac asms 1) ]);


(** Equality by extensionality, in terms of subsets.  *)

val equal_intr = prove_goal HOL_Rule.thy
    "[| subset(A,R,S);  subset(A,S,R);  R: A->bool;  S: A->bool |] \
\    ==> R=S : A->bool"
 (fn asms=>
  [ (cut_facts_tac asms 1),
    (rewrite_goals_tac class_defs),
    (resolve_tac [extensionality] 1),
    (resolve_tac [eqterm_intr] 3),
    (Pc.best_tac []) ]);


val equal_elim = prove_goal HOL_Rule.thy
    "[| R=S : A->bool;  [| subset(A,R,S);  subset(A,S,R) |] ==> P |] ==> P"
 (fn asms=>
  [ (metacut_tac eq_type1 1),
    (metacut_tac eq_type2 1),
    (ALLGOALS (resolve_tac asms)),
    (ALLGOALS (resolve_tac [asms RSN (1, subst)])),
    (REPEAT (ares_tac (asms@[subset_intr]) 1)) ]);


(*Membership in a class defined as a lambda-abstraction*)


(*General versions of class intro/elim where p(x) is any term
  Useful e.g. for lam x:A. un(A,f(x),g(x))    *)

val class_intr = prove_goal HOL_Rule.thy
    "[| form(p(a));  !x.x: A ==> p(x): bool;  a: A |] ==> a <: (lam x:A.p(x))"
 (fn asms=>
  [ (rewrite_goals_tac class_defs),
    (resolve_tac [beta_conv RS subst] 1),
    (Pc.fast_tac asms 1) ]);


val class_elim = prove_goal HOL_Rule.thy
    "[| a <: (lam x:A.p(x));  form(p(a)) ==> Q;  !x.x: A ==> p(x): bool |] \
\    ==> a: A ==> Q"
 (fn asms=>
  [ (resolve_tac asms 1),
    (resolve_tac [sym RS subst] 1),
    (resolve_tac (map (rewrite_rule class_defs) asms) 2),
    (Pc.fast_tac ([beta_conv]@asms) 1) ]);


(*Class intro/elim with abstraction over a formula 
  Less general but simpler than class_intr/class_elim.  *)

val cla_intr = prove_goal HOL_Rule.thy
    "[| P(a);  a: A |] ==> a <: {| x:A.P(x) |}"
 (fn asms=>
  [ (REPEAT (resolve_tac [class_intr] 1  ORELSE  Pc.step_tac asms 1)) ]);

val cla_elim = prove_goal HOL_Rule.thy
    "[| a <: {| x:A.P(x) |};  P(a) ==> Q;  a: A |] ==> Q"
 (fn asms=>
  [ (resolve_tac asms 1),
    (resolve_tac (reslist(asms,1,class_elim)) 1),
    (REPEAT (Pc.step_tac asms 1)) ]);



(** Binary Union *)

val un_type = prove_goal HOL_Rule.thy
    "[| R: A->bool;  S: A->bool |] ==> un(A,R,S): A->bool"
 (fn asms=>
  [ (rewrite_goals_tac class_defs),
    (Pc.typechk_tac asms) ]);


val un_cintr = prove_goal HOL_Rule.thy
    "[| ~(c<:S) ==> c<:R;  R:A->bool;  S:A->bool;  c:A |] ==> c <: un(A,R,S)"
 (fn asms=>
  [ (rewrite_goals_tac [un_def]),
    (Pc.fast_tac (class_intr::asms) 1) ]);


val un_elim = prove_goal HOL_Rule.thy
    "[| c <: un(A,R,S);  c<:R ==> P;  c<:S ==> P; \
\       R: A->bool;  S: A->bool;  c: A |]     ==> P"
 (fn asms=>
  [ (resolve_tac [class_elim] 1),
    (resolve_tac (map (rewrite_rule [un_def]) asms) 1),
    (Pc.best_tac asms) ]);



(** Binary Intersection *)

val int_type = prove_goal HOL_Rule.thy
    "[| R: A->bool;  S: A->bool |] ==> int(A,R,S): A->bool"
 (fn asms=>
  [ (rewrite_goals_tac class_defs),
    (Pc.typechk_tac asms) ]);


val int_intr = prove_goal HOL_Rule.thy
    "[| c<:R;  c<:S;  R: A->bool;  S: A->bool;  c: A |] ==> c <: int(A,R,S)"
 (fn asms=>
  [ (rewrite_goals_tac [int_def]),
    (Pc.fast_tac (class_intr::asms) 1) ]);


val int_elim = prove_goal HOL_Rule.thy
    "[| c <: int(A,R,S);  [| c<:R;  c<:S |] ==> P; \
\       R: A->bool;  S: A->bool;  c: A |]            ==> P"
 (fn asms=>
  [ (resolve_tac [class_elim] 1),
    (resolve_tac (map (rewrite_rule [int_def]) asms) 1),
    (Pc.best_tac asms) ]);


(** Big union *)

val union_type = prove_goal HOL_Rule.thy
    "F : (A->bool)->bool ==> union(A,F) : A->bool"
 (fn asms=>
  [ (rewrite_goals_tac [union_def]),
    (Pc.typechk_tac asms) ]);


(*A "hazardous" rule.  S is new variable.
  Note that subgoal S<:F should be tackled before c<:S.  *)
val union_intr = prove_goal HOL_Rule.thy
    "[| S<:F;  c<:S;  F : (A->bool)->bool;  S: A->bool;  c: A |] \
\    ==> c<:union(A,F)"
 (fn asms=>
  [ (rewrite_goals_tac [union_def]),
    (Pc.fast_tac (class_intr::asms) 1) ]);


val union_elim = prove_goal HOL_Rule.thy
    "[| c<:union(A,F);  !x.[| x: A->bool;  c<:x;  x<:F |] ==> P; \
\       F : (A->bool)->bool;  c: A |] ==> P"
 (fn asms=>
  [ (resolve_tac [class_elim] 1),
    (resolve_tac (map (rewrite_rule [union_def]) asms) 1),
    (REPEAT_FIRST (Pc.step_tac asms)) ]);


(** Big intersection *)

val inter_type = prove_goal HOL_Rule.thy
    "F : (A->bool)->bool ==> inter(A,F) : A->bool"
 (fn asms=>
  [ (rewrite_goals_tac [inter_def]),
    (Pc.typechk_tac asms) ]);


val inter_intr = prove_goal HOL_Rule.thy
    "[| !x.[| x: A->bool;  x<:F |] ==> c<:x;  F: (A->bool)->bool;  c: A |] \
\    ==> c<:inter(A,F)"
 (fn asms=>
  [ (rewrite_goals_tac [inter_def]),
    (Pc.fast_tac (class_intr::asms) 1) ]);


(*A "hazardous" rule.  S is new variable.
  The subgoal with S<:F should be tackled before the one with c<:S.  *)
val inter_elim = prove_goal HOL_Rule.thy
    "[| c<:inter(A,F);  ~(S<:F) ==> P;  c<:S ==> P; \
\       F : (A->bool)->bool;  S: A->bool;  c: A |]     ==> P"
 (fn asms=>
  [ (resolve_tac [class_elim] 1),
    (resolve_tac (map (rewrite_rule [inter_def]) asms) 1),
    (Pc.best_tac asms) ]);


(** powersets *)

val pow_type = prove_goal HOL_Rule.thy
    "[| S: A->bool |] ==> pow(A,S): (A->bool)->bool"
 (fn asms=>
  [ (rewrite_goals_tac class_defs),
    (Pc.typechk_tac asms) ]);

val pow_intr = prove_goal HOL_Rule.thy
    "[| subset(A,R,S);  R: A->bool;  S: A->bool |] ==> R <: pow(A,S)"
 (fn asms=>
  [ (cut_facts_tac asms 1),
    (rewrite_goals_tac [pow_def]),
    (resolve_tac [cla_intr] 1),
    (REPEAT (assume_tac 1)) ]);

(*simple elimination rule for powerset*)
val pow_elim_lemma = prove_goal HOL_Rule.thy
    "[| R <: pow(A,S);  R: A->bool;  S: A->bool |] ==> subset(A,R,S)"
 (fn asms=>
  [ (cut_facts_tac asms 1),
    (rewrite_goals_tac [pow_def]),
    (eresolve_tac [cla_elim] 1),
    (REPEAT (assume_tac 1)) ]);

(*sequent-style elimination rule for powerset*)
val pow_elim = prove_goal HOL_Rule.thy
    "[| R <: pow(A,S);  subset(A,R,S) ==> P;  R:A->bool;  S:A->bool |] ==> P"
 (fn asms=>
  [ (REPEAT (resolve_tac ([pow_elim_lemma]@asms) 1)) ]);


(*** Construction of prover for classes *)

val class_rulepairs =  
  [ (class_intr,"safe_intr"),
    (class_elim,"safe_elim"),
    (int_intr,"safe_intr"),
    (int_elim,"safe_elim"),
    (un_cintr,"safe_intr"),
    (un_elim,"safe_elim"),
    (inter_intr,"safe_intr"),
    (inter_elim,"haz_elim"),
    (union_intr,"haz_intr"),
    (union_intr,"haz_cintr"),    (*a fib!  lets us use comp_tac *)
    (union_elim,"safe_elim"),
    (subset_intr,"safe_intr"),
    (subset_elim,"haz_elim"),
    (pow_intr,"safe_intr"),
    (pow_elim,"safe_elim") ];

(***RECENTLY REMOVED: (equal_intr,"safe_intr"), (equal_elim,"safe_elim")  ***)

val class_type_rls = [un_type, int_type, union_type, inter_type, pow_type];


(*** Construction of the Class Theory prover ***)

structure Class_Rule_tab = struct
    structure Pure = Pure
    val type_rls = class_type_rls @ type_rls
    val pairs = class_rulepairs @ prop_rulepairs @ quant_rulepairs
    end;


structure Class = ProverFun
	(structure HOL_Lemmas=HOL_Lemmas and Rule_tab=Class_Rule_tab);

