signature Sequence =

(* GENERAL POLYMORPHIC CONSTANT SEQUENCES

   Created by:  Dave Berry, LFCS, University of Edinburgh
                db@lfcs.ed.ac.uk
   Date:        21 Feb 1989

   Maintenance: Author


   DESCRIPTION

   These functions form a general interface for one-dimensional constant
   sequences, such as lists and vectors.

   The comparison functions take a function to compare elements.

   The conversion functions include the simple parse, parse' read and string
   functions of the Object signature.  In addition there are versions
   that allow you to specify the start, finish and separating symbols.
   This lets yopu parse simple text layouts.  Too parse something that is
   too complicated to be handled by the functions given here, try using
   ML-YACC and ML-LEX.

   The stringSep function takes arbitrary strings as its start, finish
   and separating symbols.  The other functions are more limited.


   The start and finish symbols must be single non-blank characters or
   empty strings.  If they are not empty, then the input must contain
   the appropriate characters.  If they are empty they are ignored.
   
   The separating symbol may be a non-blank character, an empty string,
   or a blank.  If it is a blank, then the elements of the sequence may
   be separated by zero or more formatting characters.  If it is a single
   non-blank character, then elements must be separated by that character
   plus an arbitrary amount of whitespace.  If the separating symbol is
   the empty string, then nothing seperates elements - for example, if
   the elements are bytes then formatting characters will be read as the
   appropriate byte values.

   Other versions of the functions let you specify the length of the
   sequence to be read.  This is especially useful when there is no
   finish symbol or when the finish symbol is the same as the separating
   symbol.

   
   SEE ALSO

   Object, RefSequence, MonoSequence, SequenceToObject.
*)

sig

(* TYPES *)

  type 'a T


(* OBSERVERS *)

  val eq: ('a -> 'a -> bool) -> 'a T -> 'a T -> bool
   (* eq x y; returns true if (size x = size y) and for all i,
      0 <= i <= size x, (p (x sub i) (y sub i)). *)

  val ne: ('a -> 'a -> bool) -> 'a T -> 'a T -> bool
   (* ne x y; returns true if (size x <> size y) or for all i,
      0 <= i <= size x, (p (x sub i) (y sub i)). *)

  val lt: ('a -> 'a -> bool) -> 'a T -> 'a T -> bool
   (* lt p x y; returns true if (p (x sub i) (y sub i)) for
      0 <= i <= size x or if x is empty and y is not empty. *)

  val le: ('a -> 'a -> bool) -> 'a T -> 'a T -> bool
   (* le p x y; returns true if (p (x sub i) (y sub i)) for
      0 <= i <= size x or if x is empty. *)

  val gt: ('a -> 'a -> bool) -> 'a T -> 'a T -> bool
   (* gt p x y; returns true if (p (x sub i) (y sub i)) for
      0 <= i <= size y or if y is empty and x is not empty. *)

  val ge: ('a -> 'a -> bool) -> 'a T -> 'a T -> bool
   (* lt p x y; returns true if (p (x sub i) (y sub i)) for
      0 <= i <= size x or if y is empty. *)


(* CONVERTORS *)

  val stringSep: string -> string -> string ->
		 ('a -> string) -> 'a T -> string
  (* stringSep start finish sep p s; returns the string representation of s,
      beginning with start, ending with finish, and with the elements
      separated by sep. *)

  val string: ('a -> string) -> 'a T -> string
   (* string p l; returns the canonical string representation of l. *)

  
  exception Sep of string * string * string * string
   (* Sep (fn, start, finish, sep); raised if the function named fn is called
      with arguments start, finish and sep and one or more of these does not
      meet the requirements given above. *)

  val parseSepN: string -> string -> string ->
		 (string list -> ('a * string list, 'b) Result) ->
		 Nat -> string -> ('a T, 'a T Option) Result
   (* parseSepN start finish sep p n s; reads a sequence of 'a of length n
      that begins with start, ends with finish, in which the elements are
      separated by sep, and which forms a prefix of s, if there is one.
      Raises (Sep ("parseSep", start, finish, sep)) unless start, finish and
      sep fit the requirements listed above. *)

  val parseSep: string -> string -> string ->
                (string list -> ('a * string list, 'b) Result) ->
                string -> ('a T, 'a T Option) Result
   (* parseSep start finish sep p s; reads a sequence of 'a that begins with
      start, ends with finish, in which the elements are separated by sep,
      and which forms a prefix of s, if there is one.
      Raises (Sep ("parseSep'", start, finish, sep))
      unless start, finish and sep fit the requirements listed above. *)

  val parse: (string list -> ('a * string list, 'b) Result) ->
             string -> ('a T, 'a T Option) Result
   (* parse p s; reads a list of 'a that forms a prefix of s, if there
      is one, using default start, finish and separation symbols. *)

  val parseN: (string list -> ('a * string list, 'b) Result) ->
              Nat -> string -> ('a T, 'a T Option) Result
   (* parse p n s; reads a list of 'a of length n that forms a prefix of s,
      if there is one, using default start, finish and separation symbols. *)

  val parseSepN': string -> string -> string ->
                 (string list -> ('a * string list, 'b) Result) ->
                 Nat -> string list ->
                 ('a T * string list, 'a T Option * string list) Result
   (* parseSep' start finish sep p n l; reads a sequence of 'a of length n
      that begins with start, ends with finish, in which the elements are
      separated by sep, and which forms a prefix of l, if there is one, and
      returns that list with the remainder of l.
      Raises (Sep ("parseSep'", start, finish, sep))
      unless start, finish and sep fit the requirements listed above. *)

  val parseSep': string -> string -> string ->
                 (string list -> ('a * string list, 'b) Result) ->
                 string list ->
                 ('a T * string list, 'a T Option * string list) Result
   (* parseSep' start finish sep p l; reads a sequence of 'a that begins with
      start, ends with finish, in which the elements are separated by sep,
      and which forms a prefix of l, if there is one, and returns that list
      with the remainder of l.  Raises (Sep ("parseSep'", start, finish, sep))
      unless start, finish and sep fit the requirements listed above. *)


  val parse': (string list -> ('a * string list, 'b) Result) ->
              string list ->
              ('a T * string list, 'a T Option * string list) Result
   (* parse' p l; reads a list of 'a that forms a prefix of l, if there
      is one, and returns that list with the remainder of l. *)

  val parseN': (string list -> ('a * string list, 'b) Result) ->
               Nat -> string list ->
               ('a T * string list, 'a T Option * string list) Result
   (* parse' p n l; reads a list of 'a of length n that forms a prefix of l,
      if there is one, and returns that list with the remainder of l. *)

  val readSep: string -> string -> string ->
               (InStream -> ('a, 'b) Result) -> InStream ->
               ('a T, 'a T Option) Result
   (* readSep start finish sep p i; reads a sequence of 'a that begins with
      start, ends with finish, in which the elements are separated by sep,
      from i, if it begins with one.
      Raises (Sep ("readSep", start, finish, sep))
      unless start, finish and sep fit the requirements listed above. *)

  val readSepN: string -> string -> string ->
               (InStream -> ('a, 'b) Result) -> Nat -> InStream ->
               ('a T, 'a T Option) Result
   (* readSepN start finish sep p i; reads a sequence of 'a of length n that
      begins with start, ends with finish, in which the elements are separated
      by sep, from i, if it begins with one.
      Raises (Sep ("readSep", start, finish, sep))
      unless start, finish and sep fit the requirements listed above. *)

  val read: (InStream -> ('a, 'b) Result) -> InStream ->
            ('a T, 'a T Option) Result
   (* read p i; reads a list of 'a from i, if it begins with one. *)

  val readN: (InStream -> ('a, 'b) Result) -> Nat -> InStream ->
             ('a T, 'a T Option) Result
   (* read p n i; reads a list of 'a of length n from i, if it begins
      with one. *)

end
