signature Real =
sig

(* REALS

   Created by:	Dave Berry, LFCS, University of Edinburgh
		db@lfcs.ed.ac.uk
   Date:	8 Nov 1989

   Maintenance:	Author


   DESCRIPTION

   Standard functions on the built-in type "real".

   This signature is based on the draft ISO/ANSI Language Compatible
   Arithmetic Standard, Version 2.2, ANSI X3T2/89-179, 
   ISO/IEC JTC1.SC22/WG11 N144


   NOTES

   The arithmetic exceptions are defined to raise Overflow for all
   overflow operations and Div for attempts to divide by zero.

   The "trunc" function is intended for use when printing reals, but this might
   be an inefficient approach.  It does reduce the number of converters.

   Possible extra functions (from the Hope+ library):
   arccos, arccosh, arcsin, arcsinh, arctanh, cosh, sinh, tan, tanh, log10,
   log2, exp10, exp2

*)

  val version: real
   (* version = 0.1 *)


(* PERVASIVES *)

  exception Overflow
  and Div

  val + : real * real -> real
  val - : real * real -> real
  val * : real * real -> real
  val / : real * real -> real
  val ~ : real -> real
  val abs: real -> real
  val floor: real -> int
  val sin: real -> real
  val cos: real -> real
  val arctan: real -> real
  val exp: real -> real

  exception Ln
  val ln: real -> real

  exception Sqrt
  val sqrt: real -> real


(* SYSTEM *)

  val radix: int
   (* radix; the "base" of the implementation. *)

  val precBits: int Option
   (* precBits; the number of radix digits provided by the implementation.
      This is None if the system supports arbitrary precision real numbers. *)

  val minExp: int Option
   (* minExp; the smallest exponent value, in terms of the radix. *)

  val maxExp: int Option
   (* maxExp; the largest exponent value, in terms of the radix. *)

  val denorm: bool
   (* denorm; true if the implementation support denormalised values,
      false otherwise. *)

  val minReal: real Option
   (* minReal; the smallest real that can be stored on the system, or None
      if the system supports arbitrary length reals. *)

  val minReal': real Option
   (* minReal'; the smallest normalised real that can be stored on the system,
      or None if the system supports arbitrary length reals.
      If denorm = false then minReal' = minReal. *)

  val maxReal: real Option
   (* maxReal; the largest real that can be stored on the system, or None
      if the system supports arbitrary length reals. *)

  val precReal: int Option
   (* precReal; the largest (decimal) precision that a real can have on the
      system, or None if the system supports reals with arbitrary precision. *)

  val epsilon: real Option
   (* epsilon; the largest relative representation error for the set of
      normalised values provided by the implementation. *)

(* CONSTANTS *)

  val pi: real
   (* pi = 3.14159265 *)

  val e: real
   (* e = 2.71828183 *)


(* OBJECT *)

  structure Object: Object
    sharing type Object.T = real

  type T
     sharing type T = real

  (* The parse, parse' and read functions can handle any real constants,
     as defined in The Definition Of Standard ML, provided that the
     implementation is of sufficient precision to hold them. *)

  val parse': string list ->
              (real * string list, real Option * string list) Result

  val parse:  string -> (real, real Option) Result

  val read: InStream -> (real, real Option) Result

  val string: real -> string
   (* string n; returns the string representation of n, in the most convenient
      form.  The result must be a real constant as defined in The Definition Of
      Standard ML. *)

  val fixedWidth: bool
   (* fixedWidth = false. *)

  val lt: real -> real -> bool

  val le: real -> real -> bool

  val gt: real -> real -> bool

  val ge: real -> real -> bool

  val eq: real -> real -> bool

  val ne: real -> real -> bool


(* CONVERTERS *)

  val string': real -> string
  (* string n; returns the string representation of n, without exponent.
     The results alway contains a decimal point and at least one digit after
     the decimal point. *)

  val stringE: real -> string
  (* stringE n; returns the string representation of n, with exponent.  If all
     digits after the decimal point are 0, they are omitted, as is the decimal
     point itself.  0.0 is printed "0.0". *)

  val stringE': int -> real -> string
  (* stringE' w n; as stringE n except that the exponent must contain
     at least w characters. *)

  val trunc: int -> real -> real
  (* trunc p n; returns n with (decimal) precision p.  If p is greater than
     the existing precision, trunc has no effect. *) 

  val round: real -> int
  (* round n; returns the nearest integer to n, halves rounded up. *)

  val int: real -> int
  (* int n; returns n truncated to an integer, truncated towards 0. *)

  val ceiling: real -> int
  (* ceiling n; returns n rounded up to the next integer. *)


(* MANIPULATORS *)

  val max: real -> real -> real
   (* max x y; returns the greater of x and y.  *)

  val min: real -> real -> real
   (* min x y; returns the lesser of x and y.  *)

  val maxMin: real -> real -> real * real
   (* maxMin x y = (max (x, y), min (x, y)).  *)

  val sinCos: real -> real * real
   (* sinCos x = (sin x, cos x). *)

  (* infix 8 ** *)
  exception **! of real * int
  val ** : real * int -> real
   (* x ** y; x raised to the power y. *)
end
