(*$RefVector: GeneralTypes List InStreamType OutStreamType *)

loadSig "RefVector";

structure RefVector: RefVector =

(* VARIABLE VECTORS

   Created by:	Dave Berry, LFCS, University of Edinburgh
   Date:	30 Oct 1989

   This is a reference implementation.  Most systems will provide most of
   the following as built-in functions.

   Maintenance:	Author


   SEE ALSO

   Vector.
*)

struct

  val version = 0.1


(* TYPES *)

  datatype 'a RefVector = RefVector of 'a ref list * int * (unit ref)
   (* The first field contains the elements, the second contains the number of
      elements, and the third is a unique tag. *)

  type 'a T = 'a RefVector

(* LOCAL *)

  val emptyVector = RefVector ([], 0, ref ());


(* CREATORS *)

  fun create size init =
        if size < 0 then raise General.Nat ("create", size)
        else if size = 0 then emptyVector
        else RefVector (List.map ref (List.create size init),
			size,
			ref ())

  fun generate size f =
        if size < 0 then raise General.Nat ("generate", size)
        else if size = 0 then emptyVector
        else RefVector (List.generate size (ref o f), size, ref ())

  fun generate' size f base =
        if size < 0 then raise General.Nat ("generate'", size)
        else if size = 0 then emptyVector
	else RefVector (List.map ref (List.generate' size f base),
			size,
			ref ())


(* ITERATORS *)

  fun map f (RefVector ([], _, _)) = emptyVector
  |   map f (RefVector (l, n, _)) =
	RefVector (List.map (ref o f o !) l, n, ref ())

  fun apply f (RefVector (l, _, _)) =
	List.apply (f o !) l

  fun iterate f (RefVector ([], _, _)) = emptyVector
  |   iterate f (RefVector (l, n, _)) =
	RefVector (List.map ref (List.iterate f (List.map ! l)), n, ref ())

  fun iterateApply f (RefVector (l, _, _)) =
	List.iterateApply f (List.map ! l)


(* CONVERTERS *)

  fun list (RefVector (l, _, _)) = List.map ! l

  fun fromList [] = emptyVector
  |   fromList l = RefVector (List.map ref l, List.size l, ref ())

  fun stringSep start finish sep p (RefVector (l, _, _)) =
        List.stringSep start finish sep p (List.map ! l)
	
  fun string p (RefVector (l, _, _)) =
        List.stringSep "" "" " " p (List.map ! l)
	
  exception Sep of string * string * string * string

  (* The parse, parse' and read functions assume that entries
     are separated by formatting characters. *)

  fun parseSepN' start finish sep p n l =
	( case List.parseSepN' start finish sep p n l of
	    OK ([], s) => OK (emptyVector, s)
	  | OK (l', s) => OK (RefVector (List.map ref l', List.size l', ref ()), s)
	  | Fail (Some l', s) =>
	      Fail (Some (RefVector (List.map ref l', List.size l', ref ())), s)
	  | Fail (None, s) => Fail (None, s)
	)
	handle List.Sep x => raise Sep x

  fun parseSep' start finish sep p l =
        ( case List.parseSep' start finish sep p l of
            OK ([], s) => OK (emptyVector, s)
          | OK (l', s) => OK (RefVector (List.map ref l', List.size l', ref ()), s)
          | Fail (Some l', s) =>
	      Fail (Some (RefVector (List.map ref l', List.size l', ref ())), s)
          | Fail (None, s) => Fail (None, s)
        )
        handle List.Sep x => raise Sep x

  fun parseN' p n l =
        if n < 0 then raise General.Nat ("parseN'", n)
        else parseSepN' "" "" "" p n l

  fun parse' p l = parseSep' "" "" "" p l

  fun parseSepN start finish sep p n s =
        if n < 0 then raise General.Nat ("parseSepN", n)
        else
          case parseSepN' start finish sep p n (explode s) of
            OK (v, _) => OK v
          | Fail (x, _) => Fail x

  fun parseSep start finish sep p s =
        case parseSep' start finish sep p (explode s) of
          OK (v, _) => OK v
        | Fail (x, _) => Fail x

  fun parseN p n s =
        if n < 0 then raise General.Nat ("parseN", n)
        else parseSepN "" "" "" p n s

  fun parse p s = parseSep "" "" "" p s

  fun readSepN start finish sep p n i =
        ( case List.readSepN start finish sep p n i of
            OK [] => OK emptyVector
          | OK l  => OK (RefVector (List.map ref l, List.size l, ref ()))
          | Fail (Some l) =>
	      Fail (Some (RefVector (List.map ref l, List.size l, ref ())))
          | Fail None => Fail None
        )
        handle List.Sep x => raise Sep x

  fun readSep start finish sep p i =
        case List.readSep start finish sep p i of
          OK [] => OK emptyVector
        | OK l  => OK (RefVector (List.map ref l, List.size l, ref ()))
        | Fail (Some l) =>
	    Fail (Some (RefVector (List.map ref l, List.size l, ref ())))
        | Fail None => Fail None
        handle List.Sep x => raise Sep x

  fun readN p n i =
        if n < 0 then raise General.Nat ("readN", n)
        else readSepN "" "" "" p n i

  fun read p i = readSep "" "" "" p i

  fun fromFile p name =
        let fun readList i =
                  case p i
                  of Fail _ => (InStream.closeIn i; [])
                  |  OK x => x :: readList i
        in fromList (readList (InStream.openIn name))
        end

  fun file p (RefVector (l, _, _)) name =
        let val os = OutStream.openOut name
	    fun out s = OutStream.output' os s
        in List.apply (out o p o !) l;
           OutStream.closeOut os
        end


(* OBSERVERS *)

  fun size (RefVector (_, s, _)) = s

  fun empty v = (size v = 0)

  fun same (RefVector (_, _, r)) (RefVector (_, _, r')) =
        r = r'

  fun different (RefVector (_, _, r)) (RefVector (_, _, r')) =
        r <> r'

  fun eq p (RefVector (l, _, _)) (RefVector (l', _, _)) =
        List.eq p (List.map ! l) (List.map ! l')

  fun ne p (RefVector (l, _, _)) (RefVector (l', _, _)) =
        List.ne p (List.map ! l) (List.map ! l')

  fun lt p (RefVector (l, _, _)) (RefVector (l', _, _)) =
        List.lt p (List.map ! l) (List.map ! l')

  fun le p (RefVector (l, _, _)) (RefVector (l', _, _)) =
        List.le p (List.map ! l) (List.map ! l')

  fun gt p (RefVector (l, _, _)) (RefVector (l', _, _)) =
        List.gt p (List.map ! l) (List.map ! l')

  fun ge p (RefVector (l, _, _)) (RefVector (l', _, _)) =
        List.ge p (List.map ! l) (List.map ! l')


(* SELECTORS *)

  exception Sub of string * int
  fun sub (RefVector (l, _, _), n) =
        ! (List.sub (l, n))
        handle List.Sub _ => raise Sub ("sub", n)
  infix 9 sub;

  fun nth n v = v sub n
		handle Sub _ => raise Sub ("nth", n)

  exception Extract of int * int
  fun extract start finish (RefVector (l, _, _)) =
	if start = finish then emptyVector
        else RefVector (List.map (ref o !) (List.extract start finish l),
		        finish - start,
		        ref ())
             handle List.Extract r => raise Extract r


(* MANIPULATORS *)

  fun rev (RefVector (l, n, _)) =
	RefVector (List.rev (List.map (ref o !) l), n, ref ())

  infix 6 ^
  fun op ^ (RefVector (l, n, _), RefVector (l', n', _)) =
	RefVector (l @ l', n + n', ref ())

  exception Update of int
  fun update i v (RefVector (l, _, _)) =
        (List.sub (l, i) := v)
        handle List.Sub _ => raise Update i

  exception Copy of int * int * int
  local
    fun copy' start finish v start' v' =
	  if start = finish then ()
	  else (update start' (v sub start) v';
	        copy' (start + 1) finish v (start' + 1) v')
  in
    fun copy start finish v start' v' =
          if finish < start orelse start < 0 orelse finish > size v orelse
	     start' < 0 orelse start' + finish - start > size v'
	  then raise Copy (start, finish, start')
	  else copy' start finish v start' v'
  end
           
  exception UpdateRange of int * int
  local
    fun update' start finish i v =
	  if start = finish then ()
	  else (update start i v;
	        update' (start + 1) finish i v)
  in
    fun updateRange start finish i v =
	  if finish < start orelse start < 0 orelse finish > size v
	  then raise UpdateRange (start, finish)
	  else update' start finish i v
  end

  fun sort p (v as RefVector (l, n, r)) =
	let val l' = List.map ref (List.sort p (List.map ! l))
	    val v' = RefVector (l', n, r)
	in copy 0 n v' 0 v
	end


(* REDUCERS *)

  exception Empty of string

  fun foldL f base (RefVector (l, _, _)) =
	List.foldL f base (List.map ! l)

  fun foldL' f (RefVector ([], _, _)) =
	raise Empty "foldL'"
  |   foldL' f (RefVector (h :: t, _, _)) =
	List.foldL f (!h) (List.map ! t)

  fun foldR f base (RefVector (l, _, _)) =
	List.foldR f base (List.map ! l)

  fun foldR' f (RefVector ([], _, _)) =
	raise Empty "foldR'"
  |   foldR' f (RefVector (l, _, _)) =
	List.foldR' f (List.map ! l)

  fun pairwise f (RefVector (l, _, _)) =
	List.pairwise f (List.map ! l)
end
