(*$Vector' : GeneralTypes List' *)

loadSig "Vector'";

structure Vector' =

(* CONSTANT VECTORS

   Created by:	Dave Berry, LFCS, University of Edinburgh
		db@lfcs.ed.ac.uk
   Date:	4 Oct 1989

   Maintenance: Author


   DESCRIPTION

   A constant vector is a unique sequence of objects that can't be changed.
   Vector is an equality type.

   This is a reference implementation.  Most systems will provide most of
   the following as built-in functions.  Any sensible implementation will
   provide constant time access to each element.


   SEE ALSO

   RefVector.

*)

struct

  val version = 0.1


(* TYPES *)

  datatype 'a Vector = Vector of 'a list * int
   (* The first field contains the elements, and the second is the size. *)


(* CREATORS *)

  val emptyVector = Vector ([], 0);

  fun create size init =
	if size < 0 then raise General.Nat ("create", size)
	else if size = 0 then emptyVector
	else Vector (List'.create size init, size)

  fun generate size f =
	if size < 0 then raise General.Nat ("generate", size)
	else if size = 0 then emptyVector
	else Vector (List'.generate size f, size)

  fun generate' size f base =
	if size < 0 then raise General.Nat ("generate'", size)
	else if size = 0 then emptyVector
	else Vector (List'.generate' size f base, size)


(* ITERATORS *)

  fun map f (Vector (l, n)) =
	Vector (List'.map f l, n)

  fun iterate f (Vector (l, n)) =
	Vector (List'.iterate f l, n)

  fun apply f (Vector (l, _)) = List'.apply f l

  fun iterateApply f (Vector (l, _)) = List'.iterateApply f l


(* CONVERTORS *)

  fun list (Vector (l, _)) = l

  fun fromList [] = emptyVector
  |   fromList l  = Vector (l, List'.size l)


(* OBSERVERS *)

  fun size (Vector (_, s)) = s

  fun empty v = (size v = 0)

  fun eq p (Vector (l, _)) (Vector (l', _)) =
	List'.eq p l l'

  fun ne p (Vector (l, _)) (Vector (l', _)) =
	List'.ne p l l'

  fun le p (Vector (l, _)) (Vector (l', _)) =
	List'.le p l l'

  fun ge p (Vector (l, _)) (Vector (l', _)) =
	List'.ge p l l'

  fun lt p (Vector (l, _)) (Vector (l', _)) =
	List'.lt p l l'

  fun gt p (Vector (l, _)) (Vector (l', _)) =
	List'.gt p l l'


(* MANIPULATORS *)

  fun rev (Vector (l, n)) = Vector (List'.rev l, n)

  infix 6 ^
  fun op ^ (Vector (l, n), Vector (l', n')) =
	Vector (l @ l', n + n')


(* SELECTORS *)

  exception Sub of string * int

  infix 9 sub;
  fun op sub (Vector (l, _), i) =
	List'.sub (l, i)
	handle List'.Sub ("sub", n) => raise Sub ("sub", n)

  fun nth n v = v sub n
		handle Sub _ => raise Sub ("nth", n)

  exception Extract of int * int
  fun extract start finish (Vector (l, _)) =
	Vector (List'.extract start finish l, finish - start)
	handle List'.Extract x => raise Extract x


(* REDUCERS *)

  fun foldL f base (Vector (l, _)) = List'.foldL f base l

  fun foldR f base (Vector (l, _)) = List'.foldR f base l

  exception Empty of string

  fun foldR' f (Vector ([], _)) = raise Empty "foldR'"
  |   foldR' f (Vector (l, _)) = List'.foldR'  f l

  fun foldL' f (Vector ([], _)) = raise Empty "foldL'"
  |   foldL' f (Vector (l, _)) = List'.foldL' f l

  fun pairwise f (Vector (l, _)) = List'.pairwise f l
end
