(*$ByteRefVector : Byte List GeneralTypes InStreamType OutStreamType *)

structure ByteRefVector =

(* VARIABLE BYTE VECTORS

   Created by:	Dave Berry, LFCS, University of Edinburgh
		db@lfcs.ed.ac.uk
   Date:	12 Feb 1990

   Maintenance: Author


   DESCRIPTION

   A ByteRefVector is a unique sequence of bytes that can be updated in
   place.  There is only one empty ByteRefVector.

*)

struct

  val version = 0.1


(* TYPES *)

  type Object = Byte.Byte

  type ByteVector = ByteArray.bytearray



(* LIST CONVERTORS *)

  fun list v =
    let val n = ByteArray.length v
	fun list' i =
    	  if i = n then []
    	  else ByteArray.sub (v, i) :: list' (i+1)
    in list' 0
    end

  fun fromList l =
	let val v = ByteArray.array (List.size l, 0)
	    fun fromList' [] _ = ()
	    |   fromList' (h::t) i =
		  ( ByteArray.update (v, i, h);
		    fromList' t (i+1)
		  )
	    val _ = fromList' l 0
	in v
	end


(* CREATORS *)

  fun create size init =
	if size < 0 then raise General.Nat ("create", size)
	else ByteArray.array (size, init)


  fun generate size f =
	if size < 0 then raise General.Nat ("generate", size)
	else fromList (List.generate size f)

  fun generate' size f base =
	if size < 0 then raise General.Nat ("generate'", size)
        else fromList (List.generate' size f base)


(* ITERATORS *)

  fun map f v = fromList (List.map f (list v))

  fun apply f v = ByteArray.app f v

  fun iterate f v = fromList (List.iterate f (list v))

  fun iterateApply f v = List.iterateApply f (list v)


(* OBSERVERS *)

  val size = ByteArray.length

  fun empty v = (size v = 0)

  fun same v v' = (v = v')

  fun different v v' = (v <> v')

  fun eq v v' = List.eq Byte.eq (list v) (list v')

  fun ne v v' = List.ne Byte.ne (list v) (list v')

  fun lt v v' = List.lt Byte.lt (list v) (list v')

  fun le v v' = List.le Byte.le (list v) (list v')

  fun gt v v' = List.gt Byte.gt (list v) (list v')

  fun ge v v' = List.ge Byte.ge (list v) (list v')


(* CONVERTORS *)

  fun stringSep start finish sep v =
        List.stringSep start finish sep Byte.string (list v)

  fun string v = stringSep "" "" "" v

  exception Sep of string * string * string * string

  (* The parse, parse' and read functions assume that entries have a fixed
     width or are separated by formatting characters. *)

  fun parseSepN' start finish sep n l =
        ( case List.parseSepN' start finish sep Byte.parse' n l of
            OK (l', s) => OK (fromList l', s)
          | Fail (Some l', s) => Fail (Some (fromList l'), s)
          | Fail (None, s) => Fail (None, s)
        )
        handle List.Sep x => raise Sep x

  fun parseSep' start finish sep l =
        ( case List.parseSep' start finish sep Byte.parse' l of
            OK (l', s) => OK (fromList l', s)
          | Fail (Some l', s) => Fail (Some (fromList l'), s)
          | Fail (None, s) => Fail (None, s)
        )
        handle List.Sep x => raise Sep x

  fun parseN' n l =
        if n < 0 then raise General.Nat ("parseN'", n)
        else parseSepN' "" "" "" n l

  fun parse' l = parseSep' "" "" "" l

  fun parseSepN start finish sep n s =
        if n < 0 then raise General.Nat ("parseSepN", n)
        else
          case parseSepN' start finish sep n (explode s) of
            OK (v, _) => OK v
          | Fail (x, _) => Fail x

  fun parseSep start finish sep s =
        case parseSep' start finish sep (explode s) of
          OK (v, _) => OK v
        | Fail (x, _) => Fail x

  fun parseN n s =
        if n < 0 then raise General.Nat ("parseN", n)
        else parseSepN "" "" "" n s

  fun parse s = parseSep "" "" "" s

  fun readSepN start finish sep n i =
        ( case List.readSepN start finish sep Byte.read n i of
            OK l  => OK (fromList l)
          | Fail (Some l) => Fail (Some (fromList l))
          | Fail None => Fail None
        )
        handle List.Sep x => raise Sep x

  fun readSep start finish sep i =
        case List.readSep start finish sep Byte.read i of
          OK l  => OK (fromList l)
        | Fail (Some l) => Fail (Some (fromList l))
        | Fail None => Fail None
        handle List.Sep x => raise Sep x

  fun readN n i =
        if n < 0 then raise General.Nat ("readN", n)
        else readSepN "" "" "" n i

  fun read i = readSep "" "" "" i

  fun fromFile name =
        let fun readList i =
                  case Byte.read i
                  of Fail _ => (InStream.closeIn i; [])
                  |  OK x => x :: readList i
        in fromList (readList (InStream.openIn name))
        end

  fun file v name =
        let val os = OutStream.openOut name
            fun out s = OutStream.output' os s
        in apply (out o Byte.string) v;
           OutStream.closeOut os
        end


(* SELECTORS *)

  exception Sub of string * int
  fun sub (v, n) =
	ByteArray.sub (v, n)
	handle ByteArray.Subscript => raise Sub ("sub", n)
  infix 9 sub;

  fun nth n v = v sub n
		handle Sub _ => raise Sub ("nth", n)

  exception Extract of int * int
  fun extract start finish v =
	fromList (List.extract start finish (list v))
	handle List.Extract x => raise Extract x


(* MANIPULATORS *)

  fun rev v = fromList (List.rev (list v))

  infix 6 ^
  fun op ^ (v, v') = fromList (list v @ list v')

  exception Update of int
  fun update i x v =
	ByteArray.update (v, i, x)
	handle ByteArray.Subscript => raise Update i

  exception Copy of int * int * int
  local
    fun copy' start finish v start' v' =
          if start = finish then ()
          else (update start' (v sub start) v';
                copy' (start + 1) finish v (start' + 1) v')
  in
    fun copy start finish v start' v' =
          if finish < start orelse start < 0 orelse finish > size v orelse
             start' < 0 orelse start' + finish - start > size v'
          then raise Copy (start, finish, start')
          else copy' start finish v start' v'
  end

  exception UpdateRange of int * int
  local
    fun update' start finish i v =
          if start = finish then ()
          else (update start i v;
                update' (start + 1) finish i v)
  in
    fun updateRange start finish i v =
          if finish < start orelse start < 0 orelse finish > size v
          then raise UpdateRange (start, finish)
          else update' start finish i v
  end

  fun sort p v = fromList (List.sort p (list v))


(* REDUCERS *)

  exception Empty of string

  fun foldL f base v =
	ByteArray.fold f v base

  fun foldL' f v =
	if size v = 0 then raise Empty "foldL'"
	else List.foldL' f (list v)

  fun foldR f base v =
	ByteArray.revfold f v base

  fun foldR' f v =
	if size v = 0 then raise Empty "foldR'"
	else List.foldR' f (list v)

  fun pairwise f v =
	List.pairwise f (list v)
end
