/*

	Simple canvas widget

	@(#)SimpleCanvas.c	1.3 92/10/06 12:49:55
*/

#include <X11/copyright.h>
#include <stdio.h>

#include <X11/IntrinsicP.h>
#include <X11/Xaw/XawInit.h>
#include <X11/StringDefs.h>
#include "SimpleCanvasP.h"

static XtResource resources[] = {
#define offset(field) XtOffset(SimpleCanvasWidget, simplecanvas.field)
#define FONT(n)\
	{"font#n", XtCFont, XtRFontStruct, sizeof(XFontStruct *),\
		  offset(fontinfo[n]), XtRString, XtDefaultFont},
   /* {name, class, type, size, offset, default_type, default_addr}, */
   {XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
		   offset(foreground), XtRString, XtDefaultForeground},
   {XtNfont, XtCFont, XtRFontStruct, sizeof(XFontStruct *),
	     offset(fontinfo[0]), XtRString, XtDefaultFont},
   FONT(0)
   FONT(1)
   FONT(2)
   FONT(3)
   FONT(4)
   FONT(5)
   FONT(6)
   FONT(7)
   FONT(8)
   FONT(9)
   {XtNexposeCallback, XtCCallback, XtRCallback, sizeof(XtCallbackList),
		       offset(expose_callback), XtRCallback, NULL},
   {XtNoffsetCallback, XtCCallback, XtRCallback, sizeof(XtCallbackList),
		       offset(offset_callback), XtRCallback, NULL},
   {XtNextentCallback, XtCCallback, XtRCallback, sizeof(XtCallbackList),
		       offset(extent_callback), XtRCallback, NULL},
#undef offset
};

       void SimpleCanvasRedraw  ( /* Widget */ );
       void SimpleCanvasClear   ( /* Widget */ );
       void SimpleCanvasDestroy ( /* Widget */ );
       void SimpleCanvasInit    ( /* Widget , Widget */ );
       void SetSimpleCanvasOffset    ( /* Widget , Position, Position */ );

static
void    SimpleCanvasStartDrag(w, event, params, paramc)
	SimpleCanvasWidget       w;
	XButtonEvent *event;
	String       *params;
	Cardinal     *paramc;
{       if (!XtIsRealized(w)) return;
	{
	   w->simplecanvas.dragging = True;
	   w->simplecanvas.dragx = event->x;
	   w->simplecanvas.dragy = event->y;
	}
}

/*
	This (expensive) operation implements CONTINUOUS dragging of
	the canvas. On fast machines it works well; on slow machines it
	should be dispensed with -- StartDrag and EndDrag provide
	``jump'' dragging when used without this.
*/
static
void    SimpleCanvasDrag(w, event, params, paramc)
	SimpleCanvasWidget       w;
	XMotionEvent *event;
	String       *params;
	Cardinal     *paramc;
{       if (!XtIsRealized(w)) return;
	if (w->simplecanvas.dragging)
	{
	  SetSimpleCanvasOffset(w, w->simplecanvas.offsetx - event->x +
				   w->simplecanvas.dragx,
				   w->simplecanvas.offsety - event->y +
				   w->simplecanvas.dragy);
	}
	w->simplecanvas.dragging = True;
	w->simplecanvas.dragx    = event->x;
	w->simplecanvas.dragy    = event->y;
}

static
void    SimpleCanvasEndDrag(w, event, params, paramc)
	SimpleCanvasWidget       w;
	XButtonEvent *event;
	String       *params;
	Cardinal     *paramc;
{       if (!XtIsRealized(w)) return;
	if (w->simplecanvas.dragging)
	{
	  SetSimpleCanvasOffset(w, w->simplecanvas.offsetx - event->x +
				   w->simplecanvas.dragx,
				   w->simplecanvas.offsety - event->y +
				   w->simplecanvas.dragy);

	}
	w->simplecanvas.dragging = False;
}


static XtActionsRec actions[] =
{
   {"SimpleCanvasRedraw",       SimpleCanvasRedraw},
   {"SimpleCanvasClear",        SimpleCanvasClear},
   {"SimpleCanvasStartDrag",    SimpleCanvasStartDrag},
   {"SimpleCanvasDrag",         SimpleCanvasDrag},
   {"SimpleCanvasEndDrag",      SimpleCanvasEndDrag},
};

static char translations[] =
	"<BtnDown>:     SimpleCanvasStartDrag() \n\
	 <BtnMotion>:   SimpleCanvasDrag()      \n\
	 <BtnUp>:       SimpleCanvasEndDrag()   \n\
	";

static void
simplecanvasExpose(w, event, region)
   Widget  w;
   XEvent *event;
   Region  region;
{  if (!XtIsRealized(w)) return;
   XtCallCallbacks(w, XtNexposeCallback, (caddr_t) region);
   SimpleCanvasRedraw((SimpleCanvasWidget)w);
}

SimpleCanvasClassRec simplecanvasClassRec = {
   {   /* core fields            */
       /* superclass             */ (WidgetClass) & widgetClassRec,
       /* class_name             */ "SimpleCanvas",
       /* widget_size            */ sizeof(SimpleCanvasRec),
       /* class_initialize       */ XawInitializeWidgetSet,
       /* class_part_initialize  */ NULL,
       /* class_inited           */ FALSE,
       /* initialize             */ SimpleCanvasInit,
       /* initialize_hook        */ NULL,
       /* realize                */ XtInheritRealize,
       /* actions                */ actions,
       /* num_actions            */ XtNumber(actions),
       /* resources              */ resources,
       /* num_resources          */ XtNumber(resources),
       /* xrm_class              */ NULLQUARK,
       /* compress_motion        */ TRUE,
       /* compress_exposure      */ TRUE,
       /* compress_interleave    */ TRUE,
       /* visible_interest       */ FALSE,
       /* destroy                */ SimpleCanvasDestroy,
       /* resize                 */ XtInheritResize,
       /* expose                 */ simplecanvasExpose,
       /* set_values             */ NULL,
       /* set_values_hook        */ NULL,
       /* set_values_almost      */ XtInheritSetValuesAlmost,
       /* get_values_hook        */ NULL,
       /* accept_focus           */ NULL,
       /* version                */ XtVersion,
       /* callback_private       */ NULL,
       /* tm_table               */ translations,
       /* query_geometry         */ XtInheritQueryGeometry,
       /* display_accelerator    */ XtInheritDisplayAccelerator,
       /* extension              */ NULL
   },
   {   /* simplecanvas fields    */
       /* empty                  */ 0
   }
};


WidgetClass simplecanvasWidgetClass = (WidgetClass) & simplecanvasClassRec;


/*
	Drawing on the Canvas

*/

void BoundingBox(r, x,y, w,h) Position *x,*y; Dimension *w,*h; Thing* r;
{
	 switch (r->type)
	 {
	    case LineType:
		  *x=r->obj.line.startx;
		  *y=r->obj.line.starty;
		  *w=r->obj.line.endx-*x;
		  *h=r->obj.line.endy-*y;
	    break;
	    case BoxType:
		 *x=r->obj.box.startx;
		 *y=r->obj.box.starty;
		 *w=r->obj.box.width;
		 *h=r->obj.box.height;
	    break;
	    case StringType:
		 *x=r->obj.string.startx;
		 *y=r->obj.string.starty;
		 *w=r->obj.string.width;
		 *h=r->obj.string.ascent+r->obj.string.descent;
	    break;

	    default: *x = *y = *w = *h = 0; break;
	 }
}

static
Thing*  newLine(w, class, startx, starty, endx, endy)
	SimpleCanvasWidget  w;
	byte                class;
	Position            startx, starty, endx, endy;
{       Thing *r=XtNew(Thing);
	r->type = LineType;
	r->class= class;
	r->obj.line.startx=startx;
	r->obj.line.starty=starty;
	r->obj.line.endx=endx;
	r->obj.line.endy=endy;
	r->next = w->simplecanvas.everything;
	w->simplecanvas.everything = r;
	return r;
}

static
Thing   *newBox(w, class, startx, starty, width, height)
	SimpleCanvasWidget  w;
	byte                class;
	Position            startx, starty;
	Dimension           width,  height;
{       Thing *r=XtNew(Thing);
	r->type = BoxType;
	r->class= class;
	r->obj.box.startx=startx;
	r->obj.box.starty=starty;
	r->obj.box.width=width;
	r->obj.box.height=height;
	r->next = w->simplecanvas.everything;
	w->simplecanvas.everything = r;
	return r;
}

static
Thing*  newString(w, class, fontnumber, string, length, startx, starty,
		     width, ascent, descent)
	SimpleCanvasWidget  w;
	byte                class;
	int                 fontnumber, length;
	Position            startx, starty;
	Dimension           width, ascent, descent;
	String              string;
{       Thing *r=XtNew(Thing);
	r->type = StringType;
	r->class= class;
	r->obj.string.string=XtNewString(string);
	r->obj.string.fontnumber=fontnumber&0xFF;
	r->obj.string.startx=startx;
	r->obj.string.starty=starty;
	r->obj.string.width=width;
	r->obj.string.length=length;
	r->obj.string.ascent=ascent;
	r->obj.string.descent=descent;
	r->next = w->simplecanvas.everything;
	w->simplecanvas.everything = r;
	return r;
}

#define max(a,b) (a>b?a:b)
#define min(a,b) (a<b?a:b)

static
void    DrawThing(w, thing)
	SimpleCanvasWidget w;
	Thing              *thing;
{       Dimension offsetx = w->simplecanvas.offsetx,
		  offsety = w->simplecanvas.offsety;
	GC             gc = w->simplecanvas.gc[thing->class];
	Position  sx, sy, ex, ey;
#       define VISIBLE (sx>=0 || sy >= 0 || ex<=w->core.width || ey <=w->core.height )
	if (gc==NULL)
	{
	   fprintf(stderr, "[Warning: DrawThing(canvas=0x%x, type=%s) GC == NULL]\n",
			   w,
			   thing->type==LineType?"Line":
			   thing->type==BoxType?"Box"  :
			   thing->type==StringType?"String":"Unknown type");
	   return;
	}
	if (!XtIsRealized(w))
	{  fprintf(stderr, "[Warning: DrawThing(canvas=0x%x, type=%s) unrealized canvas]\n",
			   w,
			   thing->type==LineType?"Line":
			   thing->type==BoxType?"Box"  :
			   thing->type==StringType?"String":"Unknown type");
	   return;
	}
	switch (thing->type)
	{
	   case LineType:
	       sx=thing->obj.line.startx-offsetx;
	       sy=thing->obj.line.starty-offsety;
	       ex=thing->obj.line.endx-offsetx;
	       ey=thing->obj.line.endy-offsety;
	       if (VISIBLE)
	       XDrawLine(XtDisplay(w), XtWindow(w), gc, sx, sy, ex, ey);
	   break;

	   case BoxType:
	       sx=thing->obj.box.startx-offsetx;
	       sy=thing->obj.box.starty-offsety;
	       ex=sx+thing->obj.box.width-offsetx;
	       ey=sy+thing->obj.box.height-offsety;
	       if (VISIBLE)
	       XDrawRectangle(XtDisplay(w), XtWindow(w), gc,
			       sx,
			       sy,
			       thing->obj.box.width,
			       thing->obj.box.height);

	   break;
	   case StringType:
	   {    XTextItem text;
		text.chars =thing->obj.string.string;
		text.nchars=thing->obj.string.length;
		text.delta = 0;
		text.font  =
		 w->simplecanvas.fontinfo[thing->obj.string.fontnumber]->fid;

		sx=thing->obj.string.startx-offsetx;
		sy=thing->obj.string.starty-offsety;
		ex=sx+thing->obj.string.width;
		ey=sy+thing->obj.string.ascent+thing->obj.string.descent;
		if (VISIBLE)
		XDrawText
		(   XtDisplay(w), XtWindow(w), gc,
		    sx,
		    thing->obj.string.starty+thing->obj.string.ascent-offsety,
		    &text, 1
		);
	   }
	   break;
	 }
}

static
void   Extent(w, thing)
       SimpleCanvasWidget w;
       Thing              *thing;
/*     recalculate the extent of the diagram
       call the extent change callback if necessary
*/
{      Position  x,y, xx, yy;
       Position  tlx=w->simplecanvas.topleftx,
		 tly=w->simplecanvas.toplefty,
		 brx=w->simplecanvas.botrightx,
		 bry=w->simplecanvas.botrighty;
       Dimension ww,  hh;
       BoundingBox(thing, &x, &y, &ww, &hh);
       xx = x + ww;
       yy = y + hh;
       w->simplecanvas.topleftx   = min(tlx, x);
       w->simplecanvas.toplefty   = min(tly, y);
       w->simplecanvas.botrightx  = max(brx, xx);
       w->simplecanvas.botrighty  = max(bry, yy);
       if (!(  tlx==w->simplecanvas.topleftx
	   &&  tly==w->simplecanvas.toplefty
	   &&  brx==w->simplecanvas.botrightx
	   &&  bry==w->simplecanvas.botrighty))
       {  XtCallCallbacks(w, XtNextentCallback, NULL);
       }
}

static
void    DrawNewThing(w, thing)
	SimpleCanvasWidget w;
	Thing              *thing;
{       DrawThing(w, thing);
	Extent(w, thing);
}

void    DrawLine(c, class, x, y, w, h)
	SimpleCanvasWidget c;
	byte               class;
	Position           x,y;
	Dimension          w,h;
{
	DrawNewThing(c, newLine(c, class, x, y, x+w, y+h));
}

void    DrawBox(c, class, x, y, w, h)
	SimpleCanvasWidget c;
	byte               class;
	Position           x,y;
	Dimension          w,h;
{
	DrawNewThing(c, newBox(c, class, x, y, w, h));
}


void    DrawString(c, class, font, string, x, y, w)
	SimpleCanvasWidget c;
	byte               class;
	byte               font;
	String             string;
	Position           x,y;
	Dimension          *w;
{
	int             len = strlen(string);
	XCharStruct     box;
	int             dir, asc, desc;

	font &= 0xFF;
	XTextExtents(c->simplecanvas.fontinfo[font],
		     string, len, &dir, &asc, &desc, &box);
	DrawNewThing(c, newString(c, class, font, string, len, x, y,
				  box.width, asc, desc));
	if (w!=NULL) *w=box.width;
}

static
void    emboxthing(w, s) SimpleCanvasWidget w; Thing *s;
{  Position  x, y;
   Dimension wd, ht;
   GC gc = w->simplecanvas.gc[0];
   if (!XtIsRealized(w)) return;
   BoundingBox(s, &x, &y, &wd, &ht);
   XDrawRectangle(XtDisplay(w), XtWindow(w), gc,
		  x-w->simplecanvas.offsetx-1,
		  y-w->simplecanvas.offsety-1,
		  wd+1,
		  ht+1);
}

static
void    unboxthing(w, s) SimpleCanvasWidget w; Thing *s;
{  Position  x, y;
   Dimension wd, ht;
   GC gc = w->simplecanvas.gc[255];
   if (s==NULL) return;
   if (!XtIsRealized(w)) return;
   BoundingBox(s, &x, &y, &wd, &ht);
   XDrawRectangle(XtDisplay(w), XtWindow(w), gc,
		  x-w->simplecanvas.offsetx-1,
		  y-w->simplecanvas.offsety-1,
		  wd+1,
		  ht+1);
}

static
Bool    withinselectable(x, y, r)
	Position x, y;
	Thing *r;
{       if (r->class==0) return False;
	switch (r->type)
	{
		case StringType:
		{       Position xx, yy; Dimension ww, hh;
			BoundingBox(r, &xx, &yy, &ww, &hh);
			return xx <= x && x <= xx+ww &&
			       yy <= y && y <= yy+hh;
		}
		break;
	}
	return False;
}

static
Thing   *FindSelectable(w, x, y)
	SimpleCanvasWidget w;
	Position x, y;
{       Thing * r;
	for (r=w->simplecanvas.everything;
	     r!=NULL && !withinselectable(x, y, r);
	     r=r->next
	    ) {};
	return r;
}

void    SimpleCanvasFindAndSelect(w, x, y)
	SimpleCanvasWidget w;
	Position x,y;
{
	Thing *s = FindSelectable(w, x+w->simplecanvas.offsetx,
				     y+w->simplecanvas.offsety);
	if (s!=NULL)
	{  unboxthing(w, w->simplecanvas.selected[s->class]);
	   emboxthing(w, s);
	   w->simplecanvas.selected[s->class] = s;
	}
}

Thing   *SimpleCanvasGetSelection(w, class)
	SimpleCanvasWidget w;
	byte class;
{
	return w->simplecanvas.selected[class];
}

void    SimpleCanvasRedraw(w) SimpleCanvasWidget w;
{       Thing *thing;
	if (!XtIsRealized(w)) return;
	XClearWindow(XtDisplay(w), XtWindow(w));
	for (thing=w->simplecanvas.everything; thing!=NULL; thing=thing->next)
	    DrawThing(w, thing);
	/*
	    Embox the selected items
	*/
	{ int i;
	  for (i=0; i<256; i++)
	  {   Thing *s = w->simplecanvas.selected[i];
	      if (s!=NULL) emboxthing(w, s);
	  }
	}
}

void    SimpleCanvasClear(w)
	SimpleCanvasWidget w;
{       Thing   *next, *thing;
	for(thing=w->simplecanvas.everything; thing!=NULL; thing=next)
	{  next=thing->next;
	   switch (thing->type)
	   {
	      case StringType: XtFree(thing->obj.string.string);
	   }
	   XtFree(thing);
	}
	/*
	   Clear the display list
	*/
	w->simplecanvas.everything=NULL;
	w->simplecanvas.offsetx=0;
	w->simplecanvas.offsety=0;
	w->simplecanvas.topleftx   =
	w->simplecanvas.toplefty   =
	w->simplecanvas.botrightx  =
	w->simplecanvas.botrighty  = 0;
	XClearWindow(XtDisplay(w), XtWindow(w));
	/*
	   Clear the selections
	*/
	{ int i; for (i=0; i<256; i++) w->simplecanvas.selected[i]=NULL; }
}

static
void    SimpleCanvasInit(req, new) SimpleCanvasWidget req, new;
{       XtGCMask  mask =
		  GCForeground | GCBackground | GCLineStyle |
		  GCLineWidth  ;

	Screen    *ws     = XtScreen(new);

	Pixel     BLACK   = req->simplecanvas.foreground?
			    req->simplecanvas.foreground:
			    BlackPixelOfScreen(ws);
	Pixel     WHITE   = req->core.background_pixel?
			    req->core.background_pixel:
			    WhitePixelOfScreen(ws);

	Pixmap    STIPPLE = XmuCreateStippledPixmap(ws, BLACK, WHITE, 1);

	XGCValues values;

	Dimension width =req->core.width,
		  height=req->core.height;

	new->core.width  = width  == 0?20:width;
	new->core.height = height == 0?20:height;

	/*
		First zilch all the private properties
	*/


	{ int i;
	  for (i=0; i<256; i++)
	      { if (new->simplecanvas.fontinfo[i]==NULL)
		   new->simplecanvas.fontinfo[i]=
		   new->simplecanvas.fontinfo[0];
		   new->simplecanvas.gc[i]=0;
	      }
	  new->simplecanvas.everything = NULL;
	  new->simplecanvas.offsetx   =
	  new->simplecanvas.offsety   = 0;
	  new->simplecanvas.topleftx   =
	  new->simplecanvas.toplefty   =
	  new->simplecanvas.botrightx  =
	  new->simplecanvas.botrighty  = 0;
	}
	/* Selections */
	{ int i; for (i=0; i<256; i++) new->simplecanvas.selected[i]=NULL; }
	/*
		Make the initial set of drawing clases
		(this is really a bit arbitrary)

			0 -- positive unstippled -- used for boxing
			1 -- ditto
		      128 -- positive stippled   -- used for greying
		      255 -- inverse of 1        -- used for unboxing
	*/

	values.line_width = 0;
	values.line_style = LineSolid;
	values.background = WHITE;
	values.foreground = BLACK;
	new->simplecanvas.gc[1] =
	new->simplecanvas.gc[0] = XtGetGC(new, mask, values);

	values.background = BLACK;
	values.foreground = WHITE;
	new->simplecanvas.gc[255]  = XtGetGC(new, mask, values);

	values.background = WHITE;
	values.foreground = BLACK;
	values.stipple    = STIPPLE;
	values.fill_style = FillStippled;  /* For drawing greyed text */
	mask             |= GCStipple | GCFillStyle;
	new->simplecanvas.gc[128]  = XtGetGC(new, mask, values);
}

static
void    SimpleCanvasDestroy(w) SimpleCanvasWidget w;
{       int i;
	for (i=0; i<256; i++)
	    if (w->simplecanvas.gc[i]) XtReleaseGC(w, w->simplecanvas.gc[i]);
	SimpleCanvasClear(w);
}

void    SetSimpleCanvasFont(w, fn, fs)
	SimpleCanvasWidget w;
	byte  fn;
	XFontStruct *fs;
{
	w->simplecanvas.fontinfo[fn]=fs;
}

void    SetSimpleCanvasGC(w, gn, g)
	SimpleCanvasWidget w;
	byte gn;
	GC   g;
{
	w->simplecanvas.gc[gn]=g;
}

XFontStruct *GetSimpleCanvasFont(c, font)
	SimpleCanvasWidget c;
	byte               font;
{
	return(c->simplecanvas.fontinfo[font]);
}

GC      GetSimpleCanvasGC(c, gcid)
	SimpleCanvasWidget c;
	byte               gcid;
{
	return(c->simplecanvas.gc[gcid]);
}


void    SetSimpleCanvasX(w, x)
	SimpleCanvasWidget w;
	Position x;
{
	w->simplecanvas.offsetx=x;
	SimpleCanvasRedraw(w);
	XtCallCallbacks(w, XtNoffsetCallback, NULL);
}

void    SetSimpleCanvasY(w, y)
	SimpleCanvasWidget w;
	Position y;
{
	w->simplecanvas.offsety=y;
	SimpleCanvasRedraw(w);
	XtCallCallbacks(w, XtNoffsetCallback, NULL);
}

void    SetSimpleCanvasOffset(w, x, y)
	SimpleCanvasWidget w;
	Position x, y;
{       w->simplecanvas.offsetx=x;
	w->simplecanvas.offsety=y;
	SimpleCanvasRedraw(w);
	XtCallCallbacks(w, XtNoffsetCallback, NULL);

}

void    GetSimpleCanvasOffset(w, x, y)
	SimpleCanvasWidget w;
	Position *x, *y;
{
	*x=w->simplecanvas.offsetx;
	*y=w->simplecanvas.offsety;
}

void    GetSimpleCanvasExtent(c, x, y, w, h)
	SimpleCanvasWidget c;
	Position *x, *y;
	Dimension *w, *h;
{
	*x=c->simplecanvas.topleftx;
	*y=c->simplecanvas.toplefty;
	*w=c->simplecanvas.botrightx-c->simplecanvas.topleftx;
	*h=c->simplecanvas.botrighty-c->simplecanvas.toplefty;
}


