/*********************************************************************/
/*
**	Stuff for printing out GC statistics when 
**	the S runtime flag has been given.
*/

static double GCstart_time, GC_tot_time = 0;  /* For measuring total GC time    */
static double GCrstart_time, GC_rtot_time = 0;  /* For measuring total real GC time    */
static int tot_used_heap = 0; /* For counting total heap usage. */
static int no_of_GCs = 0;
static FILE *statf;	/* statistics file. */

int no_concreductions = 0;
int no_concassoc = 0;

static double
usertime()
{
#ifdef SYSV
    struct tms t;

    (void)times(&t);
    return (double)(t.tms_utime)/HZ;
#else
    struct rusage t;

    getrusage(0, &t);
    return(t.ru_utime.tv_sec + 1e-6*t.ru_utime.tv_usec);
#endif
}

static double now = 0.0;
static double
realtime()
{
#ifdef SYSV
    struct tms t;

    return (double)times(&t)/HZ - now;
#else
    struct timeb t;

    ftime(&t);
    return t.time + 1e-3*t.millitm - now;
#endif
}

/* Called at the beginning of execution of the program.
*/
GCstartup()
{
    if (sflag || Sflag)
	statf = fopen("STAT", "w");
    else
	statf = NULL;
    if(statf != NULL) {
	if(Sflag){
	    fprintf(statf,
/*####### ##### ##.## ###.# ###.## ####.# ####.## ####.# #######*/
 "   Heap   Stk    GC(real) GC acc (real)     tot (real) newheap\n");
	    fflush(statf);
	}
    }
    now = realtime();


}

/* Called at the beginning of each GC.
*/
/*ARGSUSED*/
GCstart(topofstack, bottomofstack, curheap, hp)
int **topofstack, **bottomofstack, **curheap, **hp;
{
    no_of_GCs++;
    tot_used_heap += hp - curheap;
    if(Bflag) {
	if (Bflag > 1)
		fprintf(stderr, " GC ");
	else
		fprintf(stderr, "\007");
    }
    GCstart_time = usertime();
    GCrstart_time = realtime();
}

/*	Called at the end of each GC.
*/
GCend(topofstack, bottomofstack, startheap, hp, newheapsize)
int **topofstack, **bottomofstack, **startheap, **hp;
{
    double time = usertime();
    double rtime = realtime();

    if (Bflag > 1)
	fprintf(stderr, "\b\b\b  \b\b\b");
    GC_tot_time += time-GCstart_time;
    GC_rtot_time += rtime-GCrstart_time;
    if(statf != NULL && Sflag){
/*####### ##### ##.## ###.# ###.## ####.# ####.## ####.# #######*/
	fprintf(statf,
	        "%7d %5d %5.2f %5.1f %6.2f %6.1f %7.2f %6.1f %7d\n", 
	        sizeof(int)*(hp-startheap), 
		bottomofstack-topofstack,
	    	(time-GCstart_time), 
	    	(rtime-GCrstart_time), 
		GC_tot_time, 
		GC_rtot_time, 
	    	time,
	    	rtime,
		newheapsize*sizeof(int)
		);
	fflush(statf);
    }
}

int nbigcmp, nbigcmpr, nsetind;

/*	Called a the end of execution of the program,
**	to print a summary of statistics.
*/
GCfinal(curheap, last_hp)
int **curheap, **last_hp;
{
    double time = usertime();
    double rtime = realtime();
    extern int no_chunks;

    tot_used_heap += last_hp - curheap;
    if(statf != NULL){
	fprintf(statf, "%10d GCs,\n", no_of_GCs);
	fprintf(statf, "%10.2f (%3.1f) seconds total time,\n", time, rtime);
	fprintf(statf, "%10.2f (%3.1f) seconds GC time", GC_tot_time, GC_rtot_time);
	fprintf(statf, " (%4.1f(%4.1f)%% of total time)\n", 
		GC_tot_time*100./time,
		GC_rtot_time*100./rtime);
	fprintf(statf, "%10d bytes allocated from the heap.\n", 
		sizeof(int)*tot_used_heap);
	fprintf(statf, "%d chunks allocated\n", no_chunks);
	fprintf(statf, "No of CONC reductions: %d, ", no_concreductions);
	fprintf(statf, "of which %d used associativity\n", no_concassoc);
	fprintf(statf, "No of bigcmp: %d, eq: %d, ind: %d\n", nbigcmp, nbigcmp-nbigcmpr, nsetind);
	fclose(statf);
    }
}
