(*************************************************************************)
(*                                                                       *)
(*                     Projet      Formel                                *)
(*                                                                       *)
(*                    CAML: users' library                               *)
(*                                                                       *)
(*************************************************************************)
(*                                                                       *)
(*                            LIENS                                      *)
(*                        45 rue d'Ulm                                   *)
(*                         75005 PARIS                                   *)
(*                            France                                     *)
(*                                                                       *)
(*************************************************************************)

(* trps.ml       translates graphics to postscript                       *)
(*              Emmanuel Chailloux & Guy Cousineau                       *)

module trps
using
type ps_int;
type ps_float;
type ps_bool;
type ps_array;
type ps_string;
type ps_matrix  = PS_MATRIX of float list;
type ps_font;
type ps_vm;
type ps_image;
type ps_channel =   PS_CHANNEL of in_channel&out_channel&out_channel&string
		  | PS_FILE    of out_channel&string;



value moveto_PS       : (float * float) -> unit;
value rmoveto_PS      : (float * float) -> unit;
value lineto_PS       : (float * float) -> unit;
value rlineto_PS      : (float * float) -> unit;
value arc_PS          : (float * float) -> float -> float -> float -> unit;
value arcn_PS         : (float * float) -> float -> float -> float -> unit;
value curveto_PS      : (float * float) -> (float * float) -> (float * float) -> unit;
value rcurveto_PS     : (float * float) -> (float * float) -> (float * float) -> unit;
value charpath_PS     : string -> bool -> unit;

value closepath_PS    : unit -> unit;
value setgray_PS      : float -> unit;
value setrgbcolor_PS  : float list -> unit;
value sethsbcolor_PS  : float list -> unit;
value newpath_PS      : unit -> unit;
value setlinewidth_PS : float -> unit;
value setlinecap_PS   : float -> unit;
value setlinejoin_PS  : float -> unit;
value setdash_PS      : float list -> float -> unit;

value stroke_PS       : unit -> unit;
value fill_PS         : unit -> unit;
value eofill_PS       : unit -> unit;
value clip_PS         : unit -> unit;
value eoclip_PS       : unit -> unit;
value image_PS        : int -> int -> int ->
                           ps_matrix -> ps_image -> int -> unit;
value show_PS         : string -> unit;
value F_PS            : string -> float -> unit;
value gsave_PS        : unit -> unit;
value grestore_PS     : unit -> unit;
value concat_PS       : ps_matrix -> unit;
value beginproc_PS    : string -> unit;
value endproc_PS      : unit -> unit;
value callproc_PS     : string -> unit;
value output_line_PS  : (string -> unit);
value translate_PS    : (float * float -> unit);
value newimage_PS     : (int -> int -> int -> ps_matrix -> int -> unit);
value name_PS         : unit -> string;
value open_PS         : string -> ps_channel;
value close_PS        : unit -> ps_channel;

type point = {xc:float;yc:float};
type geom_element =
  Seg of point list
| Arc of point * float * float * float   
        (* center,radius,start_angle,end_angle*)
| Curve of point * point * point * point;
        (* start,control1,control2,end *)
type transformation = {m11:float;m12:float;m13:float;
                       m21:float;m22:float;m23:float};
type frame = {*xmin:float;*xmax:float;*ymin:float;*ymax:float};
type path = Spath of  geom_element list  
          | Tpath of  transformation * path 
          | Cpath of path * path;
type sketch = {path: path; frame:frame; size:int};
type bitmap = {b_width:int;
               b_height:int;
               b_depth:int;
               b_bits:string vect};
type font = {Name:string; Size:float};
type font_description= {Name:string; Height:float;Width:float; 
			Descr:float vect;
                        Descr_bbox: ((float * float) * (float *	float)) vect};
type text = { t_string:string; t_font:font};
type color = Rgb of float * float * float
           | Hsb of float * float * float
           | Gra of float;
type linecap = Buttcap | Squarecap | Roundcap;
type linejoin = Beveljoin | Roundjoin | Miterjoin;
type linestyle = {linewidth:float;
                  linecap:linecap;
                  linejoin:linejoin;
                  dashpattern:int list};
type fillstyle = Nzfill | Eofill;
type clipstyle = Nzclip | Eoclip;
type interface = No_port | One_port of point * point
                    | Ports of (string * interface) list;
type pict =  Draw of path * linestyle * color * int   (* the int here is the *)
          |  Fill of path * fillstyle * color * int   (* path length which is*)
          |  Clip of clipstyle * path * pict * int    (* subject to display  *)
          |  Bitmap of bitmap                         (* limitations         *)
          |  Text of text * color
          |  Tpict of transformation * pict
          |  Cpict of pict * pict;
type picture = {pict : pict ; frame : frame;
               input_interface:interface; output_interface:interface};
value INV : transformation -> transformation;
value fit_picture_in_frame : picture -> frame -> picture;
value fit_sketch_in_frame : sketch -> frame -> sketch;
value default_display_frame : frame;
value default_linestyle : unit -> linestyle;
value default_color : unit -> color;
value make_draw_picture : linestyle * color -> sketch -> picture;
value find_font_description : font -> font_description
;;

#arith int;;
#arith float;;

let PATH_SIZE_LIMIT = ref 1500;;

let display_elt = 
    fun  (Seg [])  ->    ()
    | (Seg ({xc=x;yc=y}::ptl)) ->
             moveto_PS(x,y);do_list (fun {xc=x;yc=y} -> lineto_PS (x,y)) ptl
    | (Arc ({xc=x0;yc=y0},r,a1,a2))  ->
             arc_PS (x0,y0) r a1 a2
    | (Curve(pt1,pt2,pt3,pt4)) ->
             moveto_PS (pt1.xc,pt1.yc);
             curveto_PS (pt2.xc,pt2.yc)
                        (pt3.xc,pt3.yc)
                        (pt4.xc,pt4.yc);;

let PS_matrix {m11=a;m12=b;m13=c;m21=d;m22=e;m23=f} =
         PS_MATRIX [a;d;b;e;c;f];;

let gensym = 
  let c = ref 0 in
    (function () -> c:= !c+1; ("skel"^(string_of_int !c)));;

let skel_assoc_list = ref ([]: (geom_element list*string)list);;

let reset_trps () = skel_assoc_list:= [];();;

let declare_skel_proc_name skel =
    let n=gensym()
    in  skel_assoc_list := (skel,n)::!skel_assoc_list;
        beginproc_PS n;
        do_list display_elt skel;
        endproc_PS ();
        n;;

let find_skel_proc_name skel =
  try assq skel !skel_assoc_list
  with (failure _) 
           -> declare_skel_proc_name skel;;

let optimized_mode = ref true;;
let set_optimized_mode b = optimized_mode := b;;
(* Some postscript interpretors such as ghostscript do not allow this *)
(* optimization for obscure reasons                                   *)

let rec display_path =
  fun (Spath skel) -> if !optimized_mode 
                         then callproc_PS (find_skel_proc_name skel)
                         else do_list display_elt skel
    | (Tpath(t,p)) -> concat_PS (PS_matrix t);
                     display_path p;
                     concat_PS (PS_matrix (INV t))
    | (Cpath(p1,p2)) -> display_path p1;display_path p2
;;

let transform_linewidth {m11=a;m12=b;m21=c;m22=d;_} w =   
   2.0*w/(sqrt (a*a+c*c) + sqrt (b*b+d*d));;

let rec display_big_path =
  fun (Spath skel,w) -> newpath_PS();
                        callproc_PS (find_skel_proc_name skel);
                        setlinewidth_PS w;
                        stroke_PS();
                        closepath_PS()
    | (Tpath(t,p),w) -> let w' = transform_linewidth t w
                        in concat_PS (PS_matrix t);
                           display_big_path (p,w');
                           concat_PS (PS_matrix (INV t))
    | (Cpath(p1,p2),w) -> display_big_path (p1,w);display_big_path (p2,w)
;;


(* - To  bitmaps  ---------------------------  *)

let print_bitmap_PS b =
    for i=0 to b.b_height-1
     do
      output_line_PS b.b_bits.(i)
     done
;;

let display_bitmap_PS b =
 if b.b_depth > 16
 then failwith ("Bitmap: display_bitmap:"^"more than 16 planes ("^
                (string_of_int b.b_depth)^ ")")
 else
  begin
   gsave_PS();
  translate_PS(0.0,(float_of_int b.b_height)-1.0);
  newimage_PS b.b_width b.b_height b.b_depth
              (PS_MATRIX [1.0;0.0;0.0;-1.0;0.0;1.0]) 
              (((b.b_depth*b.b_width)+7) quo 8);

  print_bitmap_PS b;
  grestore_PS()
 end
;;


(* To display texts   *)

let actual_font = ref {Name="Bad_Name"; Size=0.0};;

let reset_actual_font() = actual_font:= {Name="Bad_Name"; Size=0.0};();;

let display_text_PS t c = 
begin
  let df = find_font_description t.t_font 
  in
   let size = t.t_font.Size 
   in
      F_PS df.Name size;
  begin
    match c with 
        Gra n       -> setgray_PS n
      | Rgb (x,y,z) -> setrgbcolor_PS [x;y;z]
      | Hsb (x,y,z) -> sethsbcolor_PS [x;y;z]
  end;
  moveto_PS (0.0,0.0);
  show_PS t.t_string
end
;;

let rec display_pict =   
 fun (Draw (p,lsty,c,n)) ->
    ((match c with Gra u -> setgray_PS u
               |  Rgb (x,y,z) -> setrgbcolor_PS [x;y;z]
               |  Hsb (x,y,z) -> sethsbcolor_PS [x;y;z]);
     (match lsty with {linewidth=w;
                       linecap=e;
                       linejoin=j;
                       dashpattern=d} ->
          (setlinewidth_PS w;
           (match e with Buttcap   -> setlinecap_PS 0.0
                      |  Squarecap -> setlinecap_PS 2.0
                      |  Roundcap  -> setlinecap_PS 1.0);
           (match j with Beveljoin  -> setlinejoin_PS 2.0
                      |  Roundjoin  -> setlinejoin_PS 1.0
                      |  Miterjoin  -> setlinejoin_PS 0.0);
           (if d<>[] then setdash_PS (map float_of_int d) 0.0);
             if n > !PATH_SIZE_LIMIT
                  then   display_big_path (p,w)
                  else                       
                         (newpath_PS();
                          display_path p;  
                          stroke_PS();
                          closepath_PS()))))


| (Fill (p,fsty,c,n)) ->
  if n > !PATH_SIZE_LIMIT
    then failwith ("Sketch size is limited to "
                   ^ (string_of_int !PATH_SIZE_LIMIT)
                   ^ " in a fill picture")
    else
    ((match c with Gra u -> setgray_PS u
               |  Rgb (x,y,z) -> setrgbcolor_PS [x;y;z]
               |  Hsb (x,y,z) -> sethsbcolor_PS [x;y;z]);
     newpath_PS ();
     display_path p;
     (match fsty with Nzfill -> fill_PS()
                |  Eofill -> eofill_PS()))
| (Clip (c,p,pict,n)) -> 
  if n > !PATH_SIZE_LIMIT
    then failwith ("Sketch size is limited to "
                   ^ (string_of_int !PATH_SIZE_LIMIT)
                   ^ " in a clip picture")
    else
                   gsave_PS();
                   newpath_PS();
                   display_path p; (* (flat_path p); *)
                   (match c with Nzclip -> clip_PS()
                              |  Eoclip -> eoclip_PS());
                   closepath_PS ();
                    display_pict pict;
                    grestore_PS()
| (Bitmap b) -> display_bitmap_PS b
| (Text (t,c))   -> display_text_PS t c
| (Tpict(t,pict))  -> gsave_PS();
                  concat_PS (PS_matrix t); display_pict pict;
                  grestore_PS()
| (Cpict(pict1,pict2))  ->  display_pict pict1;
                            display_pict pict2;;


let display_picture pict = display_pict (pict.pict);;

     
let open_PS str =
   let x = open_PS str 
    in match x with PS_FILE _    -> set_optimized_mode true; x
                  | PS_CHANNEL _ -> (if name_PS () = "(ghostscript)"
                                     then set_optimized_mode false) ; x
;;

let close_PS () =
      reset_trps();
      reset_actual_font();
      close_PS();;


end module
with

value display_picture
  and set_optimized_mode
  and reset_trps
  and open_PS
  and close_PS
;;

