(* $Id: c_scroll.ml,v 8.4 91/06/19 19:36:24 ddr Exp $
 *
 * Rogloglo Toolkit: scroll bar widget class
 *
 * $Log:	c_scroll.ml,v $
 * Revision 8.4  91/06/19  19:36:24  ddr
 * - merge avec zinc 1.8
 * 
 * Revision 8.3  91/06/15  15:49:28  ddr
 * - evolution
 * 
 * Revision 8.2  91/06/15  11:02:48  ddr
 * - merge avec zinc
 * 
 * Revision 8.1  91/06/15  09:56:57  ddr
 * - merge avec zinc
 * 
 * Revision 7.6  91/06/07  20:14:35  ddr
 * - redistrib
 *)

#standard arith false;;
#fast arith false;;

type scroll_global_info = {
  gray_pixm : Pixmap
}

and scroll_local_info = {
  orient        : orientation;
  vmin          : num;
  vmax          : num;
  bsize         : num;
  mutable vcur  : num;
  swin          : Window
}
;;

let scroll_global_info, get_scroll_global_info = dynamo_global_info
  "scroll_global_info" (ref (None: scroll_global_info option))
and scroll_local_info, get_scroll_local_info = dynamo_local_info
  "scroll_local_info" (ref (None: scroll_local_info option))
;;

let scroll_border = ref 1
and scroll_width = ref 14
and scroll_band = ref 1
;;

let nround x y =
  if x>0 & y> 0 or x<0 & y<0 then (2*x+y) quo (2*y)
  else (2*x-y) quo (2*y)
;;

let scr_callb wid args xev but f_x f_y =
  let (orient, vmin, vmax, _, callb) = args in
  let val = vmin + (
    if orient = C'Vertical then (
      nround ((num_of_C_Int(f_y xev))*(vmax-vmin)) wid.height
    ) else (
      nround ((num_of_C_Int(f_x xev))*(vmax-vmin)) wid.width
    )
  ) in
  callb(wid, but, max vmin (min val vmax))

and scroll_set(wid, val) =
  let li = get_scroll_local_info wid.info
  and xdm = wid.wid_xd.xdm in
  let gbsize = (if li.bsize = 0 then li.vmax-li.vmin else li.bsize) in
  let val = max li.vmin (min val (li.vmax+gbsize)) in
  li.vcur <- val;
  let x = (if li.orient = C'Vertical then !scroll_band
    else nround (wid.width*(val-li.vmin-gbsize)) (li.vmax-li.vmin)
  )
  and y = (if li.orient = C'Horizontal then !scroll_band
    else nround (wid.height*(val-li.vmin-gbsize)) (li.vmax-li.vmin)
  ) in
  XMoveWindow(xdm.dpy, li.swin, CINT x, CINT y);
  ()

and scroll_val wid =
  (get_scroll_local_info wid.info).vcur
;;

let ScrollA attr (orient, vmin, vmax, bsize, _ as args) =

  let szh = it_list (fun(w,h,b as szh) -> function
    C'WidthAtt v -> (Some v,h,b)
  | C'HeightAtt v -> (w,Some v,b)
  | C'BorderAtt v -> (w,h,Some v)
  | _ -> szh) (None,None,None) attr in

{
  wsize = (function xdm ->
    let make_global_info xdm =
      let pix = XCreatePixmapFromBitmapData(
        xdm.dpy, xdm.rootw, implode_ascii[85; 170], CINT 8, CINT 2,
        xdm.black, xdm.white, xdm.depth
      ) in
      xdm.end_func <- (function () ->
        let gi = get_scroll_global_info(ginfo xdm "scroll") in
        XFreePixmap(xdm.dpy, gi.gray_pixm);
        remove_ginfo xdm "scroll";
        ()
      ) :: xdm.end_func;
      add_ginfo xdm "scroll" scroll_global_info {
        gray_pixm = pix
      }
    in
    let gi =
      try get_scroll_global_info(ginfo xdm "scroll")
      with _ -> make_global_info xdm in
    let w = max (match szh with (Some v,_,_) -> v | _ -> 1) (
      (if orient = C'Vertical then !scroll_width+2*!scroll_band else 1))
    and h = max (match szh with (_,Some v,_) -> v | _ -> 1) (
      (if orient = C'Vertical then 1 else !scroll_width+2*!scroll_band))
    and b = match szh with (_,_,Some v) -> v | _ -> !scroll_border
    in (w,h,b)
  )
;
  wcreate = (function (xd, pwin, wdesc, x, y, width, height, border) ->
    let width = max 1 width
    and height = max 1 height in
    let xdm = xd.xdm in
    let win = create_window(
      xdm, pwin, x, y, width, height, border, attr,
      it_list (curry Long_OR) Zero_Long [
        ButtonPressMask; Button2MotionMask; StructureNotifyMask
      ]
    ) in
    let gbsize = (if bsize = 0 then vmax-vmin else bsize) in
    let ebsize = nround
      ((if orient = C'Vertical then height else width) * gbsize)
      (vmax - vmin)
    in
    let sx = if orient = C'Vertical then !scroll_band else 0
    and sy = if orient = C'Vertical then 0 else !scroll_band
    and sw = if orient = C'Vertical then width-2*!scroll_band else ebsize
    and sh = if orient = C'Vertical then ebsize else height-2*!scroll_band in
    let swin = XCreateSimpleWindow(
      xdm.dpy, win, CINT sx, CINT sy,
      CINT sw, CINT sh, Zero_Int, xdm.black, xdm.white
    )
    and ginfo = ginfo xdm "scroll" in
    let gi = get_scroll_global_info ginfo in
    XSetWindowBackgroundPixmap(xdm.dpy, swin, gi.gray_pixm);
    XMapSubwindows(xdm.dpy, win);
    let info = scroll_local_info {
      orient = orient; vmin = vmin; vmax = vmax; bsize = bsize;
      vcur = vmin+gbsize;
      swin = swin
    } in
    add_widget attr win {
      wid_xd = xd; win = win;
      x = x; y = y; width = width; height = height; border = border;
      wdesc = wdesc; is_mapped = false;
      info = info; user_info = no_info;
      children = []
    }
  )
;
  wdestroy = (function wid ->
    remove_widget attr wid.win wid
  )
;
  wdispatch = (function (wid, xev, param) ->
    let t = XEvent_type xev
    and xdm = wid.wid_xd.xdm in
    if t = ButtonPress then (
      let xev = XEvent_xbutton xev in
      scr_callb wid args xev (num_of_C_Int(XButtonEvent_button xev))
        XButtonEvent_x XButtonEvent_y
    ) else if t = MotionNotify then (
      let a = (xdm.dpy, ButtonMotionMask, xev) in
      while not is_null(XCheckMaskEvent a) do
        ()
      done;
      scr_callb wid args (XEvent_xmotion xev) 0 XMotionEvent_x XMotionEvent_y
    ) else if t = ConfigureNotify then (
      let xev = XEvent_xconfigure xev in
      let width = num_of_C_Int(XConfigureEvent_width xev)
      and height = num_of_C_Int(XConfigureEvent_height xev) in
      if width <> wid.width or height <> wid.height then (
        let li = get_scroll_local_info wid.info in
        wid.width <- width; wid.height <- height;
        let bsize = (if bsize = 0 then vmax-vmin else bsize) in
        let ebsize = nround
          ((if orient = C'Vertical then height else width) * bsize)
          (vmax - vmin)
        in
        let sw = if orient = C'Vertical then width-2*!scroll_band else ebsize
        and sh = if orient = C'Vertical then ebsize else height-2*!scroll_band
        in let sw = max 1 sw and sh = max 1 sh in
        XResizeWindow(xdm.dpy, li.swin, CINT sw, CINT sh);
        scroll_set(wid, li.vcur)
      );
      param
    ) else param
  )
;
  filler = mem C'FillerAtt attr
}
;;

let ScrollD = ScrollA []
;;
