-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: rmdemo.p
-- Author: Andy Lowry
-- SCCS Info: @(#)rmdemo.p	1.3 3/13/90

-- Filter around resource manager that creates and maintains a trace
-- window showing resource manager operations

rmDemo: using (rmDemo, sysRManager, rManager, terminalIO, safeWindow)
linking (rManager)

process (initQ: rmDemoInitQ)
  
declare
  initArgs: rmDemoInit;
  
  post: sysPostResourceQ;	-- post resource
  insert: sysInsertResourceQ;	-- insert a resource
  get: sysGetResourceQ;		-- get resource (copy)
  remove: sysGetResourceQ;	-- get resource (remove)
  delete: sysDeleteResourceQ;	-- delete resource

  postArgs: sysPostResource;
  insertArgs: sysInsertResource;
  getArgs: sysGetResource;
  removeArgs: sysGetResource;
  deleteArgs: sysDeleteResource;

  rm: sysRManager;		-- the real resource manager

  traceWindow: terminalCM;	-- the trace window
  trace: putStringFunc;		-- its putLine capability

begin
  receive initArgs from initQ;
  
  -- create service ports for client
  new post; new insert; new get; new remove; new delete;
  -- create connections to service ports
  new initArgs.rManager;
  connect initArgs.rManager.post to post;
  connect initArgs.rManager.insert to insert;
  connect initArgs.rManager.get to get;
  connect initArgs.rManager.remove to remove;
  connect initArgs.rManager.delete to delete;

  -- create trace window and get a shorthand for its putLine capability
  traceWindow <- initArgs.openTraceWindow("trace rm", "-Wh 5 -Ww 40 -inverse");
  trace := traceWindow.terminal.putLine;
  
  return initArgs;
  
  -- create the real resource manager and get service ports
  rm <- rManagerInitFn#(create of process rManager)();
  
  -- now loop taking requests... log each one and forward real
  -- resource manager, and log the return status
  while ('true') repeat
    select
    event post
      receive postArgs from post;
      call trace("Received Post request from User " | postArgs.userName
	    | " for resource " | postArgs.resourceName);
      block begin
	call rm.post(postArgs.userName, postArgs.resourceName,
	  postArgs.resource, postArgs.accessFunction);
	call trace("Normal Return");
	return postArgs;
      on (sysPostResource.duplicateName)
	call trace("Duplicate Name Return");
	return postArgs exception duplicateName;
      on (others)
	call trace("Discarded Return");
	discard postArgs;
      end block;
      
    event insert
      receive insertArgs from insert;
      call trace("Received Insert request from User " | insertArgs.userName
	    | " for resource " | insertArgs.resourceName);
      block begin
	call rm.insert(insertArgs.userName, insertArgs.resourceName,
	  insertArgs.resource, insertArgs.accessFunction);
	call trace("Normal Return");
	return insertArgs;
      on (sysInsertResource.duplicateName)
	call trace("Duplicate Name Return");
	return insertArgs exception duplicateName;
      on (others)
	call trace("Discarded Return");
	discard insertArgs;
      end block;
      
    event get
      receive getArgs from get;
      block begin
	call trace("Received Get request from User " | getArgs.userName
	  | " for resource " | getArgs.resourceName);
	getArgs.resource <- rm.get(getArgs.userName, 
	  getArgs.resourceName, getArgs.parameter);
	call trace("Normal Return");
	return getArgs;
      on (sysGetResource.accessDenied)
	call trace("Access Denied Return");
	return getArgs exception accessDenied;
      on (others)
	call trace("Discarded Return");
	discard getArgs;
      end block;
      
    event remove
      receive removeArgs from remove;
      block begin
	call trace("Received Remove request from User " | removeArgs.userName
	  | " for resource " | removeArgs.resourceName);
	removeArgs.resource <- rm.remove(removeArgs.userName, 
	  removeArgs.resourceName, removeArgs.parameter);
	call trace("Normal Return");
	return removeArgs;
      on (sysGetResource.accessDenied)
	call trace("Access Denied Return");
	return removeArgs exception accessDenied;
      on (others)
	call trace("Discarded Return");
	discard removeArgs;
      end block;
      
    event delete
      receive deleteArgs from delete;
      block begin
	call trace("Received Delete request from User " | deleteArgs.userName
	      | " for resource " | deleteArgs.resourceName);
	call rm.delete(deleteArgs.userName, deleteArgs.resourceName);
	call trace("Normal Return");
	return deleteArgs;
      on (sysDeleteResource.notOwner)
	call trace("Not Found/Not Owner Return");
	return deleteArgs exception notOwner;
      end block;
      
    otherwise
    end select;
    
  end while;
  
end process
