/* 
   Copyright (C) 1990 C van Reewijk, email: dutentb.uucp!reeuwijk

This file is part of GLASS.

GLASS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

GLASS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GLASS; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* File tmvar.c
 *
 * Variable management.
 */

/* standard UNIX libraries */
#include <stdio.h>

/* tm library definitions */
#include <tmc.h>

/* local definitions */
#include "tmdefs.h"

#include "tmds.h"
#include "tmstring.h"
#include "tmmisc.h"
#include "debug.h"
#include "tmerror.h"
#include "tmvar.h"
#include "tmglobal.h"

static long newcnt_var = 0;
static long frecnt_var = 0;

/******************************************************
 *    hashing parameters                              *    
 ******************************************************/

/* NOTE: The remainder of the code assumes HASHWIDTH is a power of 2. */
#define HASHBITS 6
#define HASHWIDTH (1<<HASHBITS)
#define HASHMASK (HASHWIDTH-1)

/******************************************************
 *    Datastructures                                  *    
 ******************************************************/

struct var {
    struct var *next;
    unsigned int lvl;
    char *varnm;
    char *varval;
};

#define varNIL (struct var *)0

static struct var *varlist[HASHWIDTH];

static unsigned int varlvl;

/******************************************************
 *    hash function                                   *
 ******************************************************/

/* Hash function: map a 'normal' distribution of strings on a
   evenly distributed number between 0..HASHWIDTH-1.
 */
static unsigned int hashval( s )
 register char *s;
{
    register unsigned int v = 0;

    while( *s!= '\0' ){
	 v = (v ^ *s);
	 v<<=1;
	 if( v & HASHWIDTH ) v++;
	 v &= HASHMASK;
	 s++;
    }
    return( v );
}

/******************************************************
 *    variable management                             *
 ******************************************************/

/* Search in current context level 'lvl' for variable with name 'nm'.
   Return pointer to struct var of variable, or varNIL if not found.
 */
static struct var *findlocvar( nm )
 char *nm;
{
    struct var *l;
    unsigned int hv;

    hv = hashval( nm );
    l = varlist[hv];
    while( l!=varNIL && l->lvl == varlvl ){
	if( strcmp( nm, l->varnm ) == 0 ) return( l );
	l=l->next;
    }
    return( varNIL );
}

/* Search in all contexts for variable with name 'nm'. Return pointer to
   struct var of variable, or varNIL if not found.
 */
static struct var *findvar( nm )
 char *nm;
{
    struct var *l;
    unsigned int hv;

    hv = hashval( nm );
    l = varlist[hv];
    while( l!=varNIL ){
	if( strcmp( nm, l->varnm ) == 0 ) return( l );
	l=l->next;
    }
    return( varNIL );
}

/* Add a variable 'nm' with value 'v' to the known variables.
   If variable 'nm' exists, overwrite old value, else create a
   new variable with the given value.
 */
void setvar( nm, v )
 char *nm;
 char *v;
{
    struct var *vr;
    unsigned int hv;

    if( vartr ) fprintf( tracestream, "set: %s = '%s' (lvl=%u)\n", nm, v, varlvl );
    vr = findlocvar( nm );
    if( vr != varNIL ){
	fre_string( vr->varval );
	vr->varval = new_string( v );
	return;
    }
    hv = hashval( nm );
    vr = (struct var*) ckmalloc( sizeof(*vr));
    newcnt_var++;
    vr->lvl = varlvl;
    vr->varnm = new_string( nm );
    vr->varval = new_string( v );
    vr->next = varlist[hv];
    varlist[hv] = vr;
}

/* Start a new variable context. */
void newvarctx()
{
    varlvl++;
    if( vartr ) fprintf( tracestream, "New variable context (lvl=%u)\n", varlvl );
}

/* Drop a variable context. */
void flushvar()
{
    struct var *l;
    struct var *n;
    unsigned int hv;

    for( hv=0; hv<HASHWIDTH; hv++ ){
	l = varlist[hv];
	while( l!=varNIL && l->lvl==varlvl ){
	    n=l->next;
	    fre_string( l->varval );
	    fre_string( l->varnm );
	    free( (char *) l );
	    frecnt_var++;
	    l=n;
	}
	varlist[hv] = l;
    }
    varlvl--;
}

/* Search for variable with name 'nm'. Return pointer to the value of
   'nm', or NULL of not found.
 */
char *getvar( nm )
 char *nm;
{
    struct var *v;

    v = findvar( nm );
    if( v != varNIL ){
	if( vartr )
	    fprintf( tracestream, "found: %s = '%s' (lvl=%u)\n", nm, v->varval, v->lvl );
	return( v->varval );
    }
    if( vartr ) fprintf( tracestream, "not found: %s\n", nm );
    return( CHARNIL );
}

/******************************************************
 *                                                    *
 *    statistics                                      *
 *                                                    *
 ******************************************************/

/* Initialize 'var' routines. */
void init_var()
{
    unsigned int i;

    for( i=0; i<HASHWIDTH; i++ ){
	varlist[i] = varNIL;
    }
    varlvl = 0;
    newcnt_var = 0;
    frecnt_var = 0;
}

/* Print allocation statistics of 'var' routines to file 'f'. */
void stat_var( f )
 FILE *f;
{
    PRSTAT( f, "var", newcnt_var, frecnt_var );
    newcnt_var = 0;
    frecnt_var = 0;
}
