/* 
   Copyright (C) 1990 C van Reewijk, email: dutentb.uucp!reeuwijk

This file is part of GLASS.

GLASS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

GLASS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GLASS; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* File: tmexpr.c
 *
 * Handle expressions.
 */

/* Standard UNIX libraries and expressions */
#include <stdio.h>
#include <ctype.h>

/* tm library */
#include <tmc.h>

/* local definitions */
#include "tmdefs.h"

#include "tmds.h"
#include "tmstring.h"
#include "debug.h"
#include "tmerror.h"
#include "tmexpr.h"
#include "tmglobal.h"
#include "tmmisc.h"

/* integer expression evaluation.
 * All these functions evaluate a set of operators at one
 * priority level, set a value pointed to by a passed pointer to
 * the evaluated value, and return a pointer to the remaining string.
 *
 * Priorities (in order of increasing binding power:
 *
 * & |
 * < <= > >= == !=
 * + -
 * * / %
 * + - ~ ! (unary)
 */

static char *evalbool();

/* Evaluate integer constants and '()'. */
static char *evalconst( x, vp )
 register char *x;
 long *vp;
{
    string buf;
    char *bufp;
    char *s;

    while( isspace( *x ) ) x++;
    if( *x == ORBRAC ){
	s = evalbool( x+1, vp );
	while( isspace( *s ) ) s++;
	if( *s != CRBRAC ){
	    return( x );
	}
	else{
	    return( s+1 );
	}
    }
    buf = new_string( x );
    bufp = buf;
    while( isdigit( *x ) ){
	*bufp++ = *x++;
    }
    *bufp = '\0';
    if( buf[0] == '\0' ){
	line_error( BADEXPR );
	*vp = 0;
    }
    else{
	*vp = atoi( buf );
    }
    fre_string( buf );
    return( x );
}

/* Evaluate unary operators. */
static char *evalun( x, vp )
 register char *x;
 long *vp;
{
    char *s;
    long v;

    while( isspace( *x ) ) x++;
    if( *x == '-' ){
	s = evalun( x+1, &v );
	*vp = -v;
	return( s );
    }
    if( *x == '!' ){
	s = evalun( x+1, &v );
	*vp = !v;
	return( s );
    }
    if( *x == '+' ){
	return( evalun( x+1, vp ) );
    }
    return( evalconst( x, vp) );
}

/* Evaluate product operators. */
static char *evalprod( x, vp )
 register char *x;
 long *vp;
{
    char *s;
    long v1;
    long v2;

    while( isspace( *x ) ) x++;
    s = evalun( x, &v1 );
    if( s == x ){
	*vp = 0;
	return( x );
    }
    while( isspace( *s ) ) s++;
    if( *s == '*' ){
	s = evalprod( s+1, &v2 );
	*vp = v1*v2;
	return( s );
    }
    if( *s == '/' ){
	s = evalprod( s+1, &v2 );
	if( v2 == 0 ){
	    *vp = ~(-1);
	}
	else {
	    *vp = v1/v2;
	}
	return( s );
    }
    if( *s == '%' ){
	s = evalprod( s+1, &v2 );
	if( v2 == 0 ){
	    *vp = ~(-1);
	}
	else {
	    *vp = v1%v2;
	}
	return( s );
    }
    *vp = v1;
    return( s );
}

/* Evaluate sum operators. */
static char *evalsum( x, vp )
 register char *x;
 long *vp;
{
    char *s;
    long v1;
    long v2;

    while( isspace( *x ) ) x++;
    s = evalprod( x, &v1 );
    if( s == x ){
	*vp = 0;
	return( x );
    }
    while( isspace( *s ) ) s++;
    if( *s == '+' ){
	s = evalsum( s+1, &v2 );
	*vp = v1+v2;
	return( s );
    }
    if( *s == '-' ){
	s = evalsum( s+1, &v2 );
	*vp = v1-v2;
	return( s );
    }
    *vp = v1;
    return( s );
}

/* Evaluate compare operators. */
static char *evalcmp( x, vp )
 register char *x;
 long *vp;
{
    char *s;
    long v1;
    long v2;

    while( isspace( *x ) ) x++;
    s = evalsum( x, &v1 );
    if( s == x ){
	*vp = 0;
	return( x );
    }
    while( isspace( *s ) ) s++;
    if( s[0] == '!' && s[1] == '=' ){
	s = evalsum( s+2, &v2 );
	*vp = (v1!=v2);
	return( s );
    }
    if( s[0] == '=' && s[1] == '=' ){
	s = evalsum( s+2, &v2 );
	*vp = (v1==v2);
	return( s );
    }
    if( s[0] == '<' && s[1] == '=' ){
	s = evalsum( s+2, &v2 );
	*vp = (v1<=v2);
	return( s );
    }
    if( s[0] == '<' ){
	s = evalsum( s+1, &v2 );
	*vp = (v1<v2);
	return( s );
    }
    if( s[0] == '>' && s[1] == '=' ){
	s = evalsum( s+2, &v2 );
	*vp = (v1>=v2);
	return( s );
    }
    if( s[0] == '>' ){
	s = evalsum( s+1, &v2 );
	*vp = (v1>v2);
	return( s );
    }
    *vp = v1;
    return( s );
}

/* Evaluate boolean binary operators. */
static char *evalbool( x, vp )
 register char *x;
 long *vp;
{
    char *s;
    long v1;
    long v2;

    s = evalcmp( x, &v1 );
    while( isspace( *s ) ) s++;
    if( s[0] == '&' ){
	s = evalbool( s+1, &v2 );
	*vp = v1 && v2;
	return( s );
    }
    if( s[0] == '|' ){
	s = evalbool( s+1, &v2 );
	*vp = v1 || v2;
	return( s );
    }
    *vp = v1;
    return( s );
}

/* Given a string 'x' containing a numerical expression,
 * evaluate it, and construct a string from the resulting integer.
 */
char *evalexpr( x )
 string x;
{
    char buf[NUMBUFSIZE];
    char *s;
    long v;

    if( fntr ){
	fprintf( tracestream, "evaluating expression $[%s]\n", x );
    }
    s =  evalbool( x, &v );
    while( isspace( *s ) ) s++;
    if( *s != '\0' ){
	(void) strcpy( errarg, s );
	line_error( BADEXPR );
    }
    (void) sprintf( buf, "%ld", v );
    if( fntr ){
	fprintf( tracestream, "expression value: '%s'\n", buf );
    }
    return( new_string( buf ) );
}
