/* 
   Copyright (C) 1990 C van Reewijk, email: dutentb.uucp!reeuwijk

This file is part of GLASS.

GLASS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

GLASS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GLASS; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* file: tmerror.c
   Error handling.
 */

#include "tmdefs.h"
#include <tmc.h>
#include "tmds.h"
#include "tmstring.h"
#include "tmglobal.h"
#include "debug.h"
#include "tmerror.h"

extern char *strcpy();
extern char *sys_errlist[];
extern int sys_nerr;

char errpos[ERRARGLEN];
char errarg[ERRARGLEN];

static bool goterror;      /* true if a general error has occurred */

/******************************************************
 *                                                    *
 *            ERROR MESSAGE TABLES                    *
 *                                                    *
 ******************************************************/

#define ENDTAB -1        /* end of table mark */

/* Structure for the storage of error message strings.
   Arrays of these structures are used to translate
   error codes to error strings.
 */
struct errm {
   tmerrcode errmcode;	/* code of the error */
   char *errmstr;	/* associated string */
};

struct errm tmerrtab[] =
{
   { BADCONSNM  , "bad constructor name" },
   { BADDOTCOM  , "unknown dot command" },
   { BADEXPR    , "bad expression" },
   { BADNUMBER  , "malformed number" },
   { BADPADDING , "empty padding string" },
   { BADPAR     , "bad parameter" },
   { BADPARNO   , "bad number of parameters" },
   { BADRE      , "bad regular expression" },
   { BADTAG,      "bad tag" },
   { BADTOK     , "bad token" },
   { BADTYPENM  , "bad type name" },
   { BUFOVER,     "buffer overflow" },
   { CRASH,       "internal error" },
   { DOUBLECONS,  "double use of constructor name" },
   { DOUBLEFIELD, "double use of field name" },
   { ERRDET,      "errors detected, program aborted" },
   { EXTRATERM  , "unexpected termination command" },
   { MISSINGPAR , "missing parameter" },
   { NOCLOSEBRAC, "missing close bracket" },
   { NOEXPR     , "missing expression" },
   { NONAME     , "no name specified" },
   { NOROOM,      "no room" },
   { NOSEP      , "no separator" },
   { NOSUCHCONS , "no such constructor" },
   { NOSUCHELM  , "no such constructor element" },
   { NOSUCHFN   , "no such function" },
   { NOSUCHTYPE , "no such type" },
   { NOTACONS   , "not a constructor type" },
   { NOTATUPLE  , "not a tuple type" },
   { NOUNDERSCORE, "No '_' allowed in name" },
   { SYNTAXERR,   "syntax error" },
   { TOOMANYARG , "excess function parameters" },
   { UNEXPECTDOT, "unexpected dot command" },
   { UNEXPECTEOF, "unexpected end of file" },
   { UNEXPECTEOL, "unexpected end of line" },
   { VARNOTFOUND, "variable not found" },
   { ERREND, "" }
};

/* Translate an error code 'no' into an error string using a translation
   table 'tab'.  If the code is not known, return NULL.

   Each table is assumed to end with an entry with the special
   code 'ERREND'. Tables without such an end mark will cause havoc.
 */
static char *finderrm( no, tab )
 tmerrcode no;
 struct errm *tab;
{
   while( tab->errmcode != ERREND ){
      if( tab->errmcode == no ) return( tab->errmstr );
      tab++;
   }
   return( NULL );
}

/******************************************************
 *                                                    *
 *            ERROR HANDLERS                          *
 *                                                    *
 ******************************************************/

/* Initialize error handling routines. */
void init_error()
{
   errarg[0] = '\0';
   errpos[0] = '\0';
   goterror = FALSE;
}

/* General error printing routine: print error message possibly preceded
   by string in 'errpos', and possibly followed by string in 'errarg'.
   Set flag 'goterror' to indicate an error has occurred. At the next call to
   errcheck() this flag will be checked, and the program will be stopped if
   it was set.
 */
static void printerror( msg )
 char *msg;
{
    if( errpos[0] != '\0' ) fprintf( stderr, "%s: ", errpos );
    fputs( msg, stderr );
    if( errarg[0] != '\0' ) fprintf( stderr, ": %s", errarg);
    fputs( "\n", stderr );
    errarg[0] = '\0';
    errpos[0] = '\0';
    goterror = TRUE;
}

/* Given an error code 'no', search the table of error message ('tmerrtab')
 * for that code, and print the string associated with the code. Complain
 * if the error code is not known.
 */
void error( no )
 tmerrcode no;
{
    char *errm;

    errm = finderrm( no, tmerrtab );
    if( errm == NULL ){
	(void) sprintf( errarg, "%d", no );
	errm = "unknown error code";
    }
    printerror( errm );
}

/* System error handler. */
void sys_error( no )
 int no;
{
    if( no>sys_nerr ){
	(void) sprintf( errarg, "%d", no );
	printerror( "unknown system error" );
    }
    else {
	printerror( sys_errlist[no] );
    }
}

/* Error handler that supplies line number and file name from the
   global variables 'tpllineno' and 'tplfilename'.
 */
void line_error( no )
 int no;
{
    if( tplfilename == CHARNIL ){
	(void) sprintf( errpos, "(%d)", tpllineno );
    }
    else {
	(void) sprintf( errpos, "%s(%d)", tplfilename, tpllineno );
    }
    error( no );
}

/* Internal error handler. Print the given message with number 'no'
   with given position in the source: file name and line number.
 */
void docrash( file, line, no )
 char *file;
 int line;
 int no;
{
   (void) sprintf( errpos, "internal error at %s(%d)", file, line );
   error( no );
}

/* Check if 'goterror' flag is set, and do exit(1) if it is. */
void errcheck()
{
   if( !goterror ) return;
   error( ERRDET );
   exit( 1 );
}
