static char RCSid[] = "$Id: op_misc.c,v 1.20 1993/01/22 23:38:47 kadhim Exp $";
/* Copyright, 1989, The Regents of the University of Colorado */

/* Routines:
 *      1) output_grammar_listing - Outputs an existing grammar
 *                                  structure in a listing format.
 *      2) pause_cagt - Spawn subprocess to allow user to execute command
 *                      interpreter commands.
 *	3) rename_arbitrary_identifiers - Allows the user to
 *			rename any identifiers in the current
 *			symbol table.
 *	4) change_chainrule_settings - Allows the user to change the
 *			chainrule status of rules in the grammar.
 *	5) view_symbol_eq - Outputs the equivalences between symbols
 *			in the current symbol table.
 *	6) write_cull_file() - Writes all the current chainrules in
 *			a grammar to the cull file.
 */




#include <stdio.h>
#include <stdlib.h>
#include <curses.h>
#include <sys/file.h>
#include <string.h>

#include "cagt.h"
#include "cagt_config.h"
#include "cagt_usr_err.h"
#include "support.h"
#include "io_curses.h"
#include "gram.h"
#include "queue.h"

#ifdef VAX11C
#include <descrip.h> 
#include <ssdef.h>
#else
#include <sys/wait.h>                               /* Needed by pause_cagt */
#endif



char *mktemp();







public void output_grammar_listing(start)
   struct left_node *start;
/*
 * output_ebnf_grammar places the internal EBNF grammar structure in
 * file d in a listing type format.
 *
*/
   {
   WINDOW *win;
   char name[132];
   FILE  *d;
   int cur_line,
   cur_col;
   int error;
   int rule_num = 0;
   struct left_node *left;
   int extra;

   /* Get a window to cover up the invocation window */
   win = newwin(0,0,0,0);
   box(win,'|','-');
   output_centered(win,1,TRUE,"Write Grammar Listing");
   mvwaddstr(win,2,1,LINE);

   /* Get a valid file name */
   mvwaddstr(win,5,START_COL,"Enter a file name for the listing > ");
   getyx(win,cur_line,cur_col);
   do {
      (void) wmove(win,cur_line,cur_col);
      (void) wclrtoeol(win);
      (void) wmove(win,cur_line,cur_col);
      box(win,'|','-');
      (void) wrefresh(win);
      GET_STRING(win,name)
      if (error = !(d = fopen(name,"w")))
         {
         (void) wmove(win,6,START_COL);
         (void) wprintw(win,"cagt: Unable to open \"%s\".",name);
         mvwaddstr(win,7,START_COL,"Do you wish to try again? (y or n)> ");
	 box(win,'|','-');
         (void) wrefresh(win);
         if (!get_boolean_response(win,TRUE)) break;
         (void) wmove(win,6,START_COL);
         (void) wclrtobot(win);
         }
      } while (error);


   /* If no error, Output the structure */
   if (!error)
      {
      mvwaddstr(win,6,START_COL, "Mark Chain Rules, Include Rule Numbers,");
      mvwaddstr(win,7,START_COL, "    And Indicate Deleted Rules (y or n)> ");
      box(win,'|','-');
      (void) wrefresh(win);
      extra = get_boolean_response(win,TRUE);
      (void) wmove(win,11,START_COL);
      (void) wprintw(win,"Writing Listing File \"%s\"...",name);
      box(win,'|','-');
      (void) wrefresh(win);
      for (left = start; left != (LEFT_NODE_PTR)0; left = left->next_rule)
         {
         if (extra) (void) fprintf(d,"%s %d) ",(left->chain_rule) ? "C" :" ",
						++rule_num);
         if (left->code != NULT)                      /* Not a deleted rule */
               {
               output_left_node(d,left);
               output_right_side(d,left->right_side,FALSE);
               (void) fprintf(d,"\n\n");
               }
            else
               if (extra)
                  {
                  (void) fprintf(d,"   - Deleted -");
                  (void) fprintf(d,"\n\n");
                  }
         }
      if (fclose(d) == EOF)
            message_and_pause(win,12,"Unable to close file. ");
         else
            message_and_pause(win,12,"Done.");
      }

   (void) wclear(win);
   (void) wrefresh(win);
   delwin(win);

   }







/* pause_cagt - Create subprocess to allow user to execute command 
 *      entry:	some curses generated screen is currently being displayed.
 *	exit:   the user has terminated the subprocess.
 *      	the screen has been restored by curses to it's
 *             	pre-invocation state.
*/
public void pause_cagt()
{
    WINDOW *win;
    char *shell_path;				/* Path of the shell to use */
    char *shell_name;				/* Name of the shell to use */
    int	stat;					/* status from fork(2) */

    /* Get a window to cover up the invocation window */
    win = newwin(0,0,0,0);
    (void) wclear(win);
    (void) wrefresh(win);

#ifndef lint    /* lint complains about stty, but I can't change the macros */
    (void) nocrmode();
    (void) echo();
#endif

    if (stat = fork())
	    {
	    if (stat == -1)
		    cagt_msg(0, MSG_RET, (cagt_msg_text, msg_arr[-(CAGT_FORKERR)], 0))
		else
		    (void) wait((int *)0);
	    }
	else
	    {
	    /* exec the shell specified by SHELL in the environment */
	    if (!(shell_path = getenv("SHELL"))) shell_path = "/bin/sh";
	    shell_name = shell_path + strlen(shell_path);
	    while (shell_name > shell_path)
		if (*--shell_name == '/')
		    {
        	    ++shell_name;
		    break;
		    }
	    execl(shell_path, shell_name, 0);	/* Ignore the status */
	    }

#ifndef lint    /* lint complains about stty, but I can't change the macros */
    (void) noecho();
    (void) nl();
    (void) crmode();
#endif

    (void) wclear(win);
    (void) wrefresh(win);
    delwin(win);

}			/* pause_cagt */







public void rename_arbitrary_identifiers(grammar)
   LEFT_NODE_PTR grammar;
/*
 * rename_arbitrary_identifiers allows the user to rename any identifier
 * in the current symbol table.
 *
 * On Entry:
 *	A valid symbol table for grammar is enabled.
 *	grammar points at the left side chain of a grammar structure.
 *
 * On Exit:
 *	Desired identifiers have been renamed.
 *
*/
   {
   WINDOW *win;
   int cur_line;
   QUEUE_PTR queue;
   SYMBOL cur_sym;
   SYMBOL def_sym;
   char first_pass = TRUE;
   char non_term = TRUE;

#ifdef lint
def_sym = (SYMBOL)0;   /* So lint won't give may be used before set msg */
#endif

   /* Get a window to cover up the invocation window */
   win = newwin(0,0,0,0);
   box(win,'|','-');
   output_centered(win,1,TRUE,"Arbitrary Identifier Renaming");
   mvwaddstr(win,2,1,LINE);

   queue = init_queue();
   c_select_identifiers(queue);
   while (!empty(queue))
	{
	cur_sym = (SYMBOL) dequeue(queue);
	cur_line = 6;
	(void) wmove(win,cur_line,START_COL);
	(void) wclrtobot(win);
	(void) wmove(win,cur_line,START_COL);
	if (!first_pass)
	   {
	   mvwaddstr(win,cur_line,START_COL,"The default identifier is ");
	   (void) c_prtsym(win,def_sym);
	   (void) wprintw(win,".");
	   cur_line += 2;
	   }
	mvwaddstr(win, cur_line, START_COL,
	        "Enter a name to be used in place of ");
	(void) c_prtsym(win,cur_sym);
        (void) wprintw(win,"> ");
	box(win,'|','-');
	(void) wrefresh(win);
	def_sym = get_identifier(win,TRUE,(!first_pass), def_sym, &non_term);
	rename_identifier(cur_sym,def_sym);
	if (first_pass) first_pass = FALSE;
	}


   (void) NULT_redundant_rules(grammar);

   delete_queue(queue);
   (void) wclear(win);
   (void) wrefresh(win);
   delwin(win);

   }







private void change_chainrule_settings_help()
/*
 * change_chainrule_settings_help outputs a help screen for use
 * by change_chainrule_settings.
 *
*/
   {
   WINDOW *win;

   /* Get a window to cover up the invocation window */
   win = newwin(0,0,0,0);
   output_centered(win,1,TRUE,"HELP: Change Chain Rule Settings");
   mvwaddstr(win,2,1,LINE);
   (void) wrefresh(win);

   mvwaddstr(win, 4,10,"The Following Commands Are Availible:");
   mvwaddstr(win, 6,10,"Return - Causes the next page of rules to be displayed.");
   mvwaddstr(win, 7,10,"         If on the last page, returns to previous menu.");
   mvwaddstr(win, 8,10,"Rule # - Entering a rule number causes that rule to");
   mvwaddstr(win, 9,10,"         be displayed at the top of the current page.");
   mvwaddstr(win,10,10,"I - Initial: Restores the chain rule status to the state that");
   mvwaddstr(win,11,10,"    existed at the invocation of chain rule setting.");
   mvwaddstr(win,12,10,"O - Original: Restores the chain rule status to the");
   mvwaddstr(win,13,10,"    CAGT startup defaults.");
   mvwaddstr(win,14,10,"N - None: The grammar contains no chain rules.");
   mvwaddstr(win,15,10,"S # - Select: Marks the specified rule as a chain rule. ");
   mvwaddstr(win,16,10,"R # - Remove: Marks the specified rule as a non-chain rule.");
   mvwaddstr(win,17,10,"D - Done: Returns to the previous menu.");
   mvwaddstr(win,18,10,"H - Help: Displays this menu.");

   box(win,'|','-');
   (void) wrefresh(win);
   message_and_pause(win,18,"");

   (void) wclear(win);
   (void) wrefresh(win);
   delwin(win);

   }







public void change_chainrule_settings(start)
   LEFT_NODE_PTR start;
/*
 * change_chainrule_settings outputs the internal EBNF grammar structure to
 * the screen in a style similar to the UNIX more utility, allowing the
 * user to set the chainrule status for rules in the grammar.
 *
 * On Entry:
 *    start points at the left hand chain of the grammar structure.
 *
 * On Exit:
 *    The chainrule status for rules in the grammar have been modified
 *	 under user control.
 *
*/
   {
   WINDOW *win;
   int cur_line,
       cur_col;
   int top_rule;		   /* Remembers rule above at top of screen */
   int not_clipped;
   struct indexed_access_block access_blk;         /* Allows Indexed Access */
   LEFT_NODE_PTR left;
   char bottom[80];           /* Used to output message at bottom of screen */
   int rule_num = 0;			       /* Current rule number index */
   char *initial;		    /* Remembers invocation chainrule state */
   int i;
   LEFT_NODE_PTR temp;			/* Used to weed out 'deleted' rules */
   char quit = FALSE;				  /* Indicates user is done */
   int menubar_num;				 /* Number typed at menubar */
   char menubar_char;			      /* Character typed at menubar */
   SYMBOL_PTR_NODE_PTR dummy_sym;	       /* Used to call is_chainrule */

/* menubar character options for rule and identifier selection */
static MENUBAR_OPT_ARR char_opts =
			      { FALSE, FALSE,		/* A */
				FALSE, FALSE,		/* B */
				FALSE, FALSE,		/* C */
				TRUE , FALSE,		/* Done */
				FALSE, FALSE,		/* E */
				FALSE, FALSE,		/* F */
				FALSE, FALSE,		/* G */
				TRUE , FALSE,		/* Help */
				TRUE , FALSE,		/* Initial */
				FALSE, FALSE,		/* J */
				FALSE, FALSE,		/* K */
				FALSE, FALSE,		/* L */
				FALSE, FALSE,		/* M */
				TRUE , FALSE,		/* None*/
				TRUE , FALSE,		/* Original */
				FALSE, FALSE,		/* P */
				FALSE, FALSE,		/* Q */
				TRUE , TRUE ,		/* Remove # */
				TRUE , TRUE ,		/* Select # */
				FALSE, FALSE,		/* T */
				FALSE, FALSE,		/* U */
				FALSE, FALSE,		/* V */
				FALSE, FALSE,		/* W */
				FALSE, FALSE,		/* X */
				FALSE, FALSE,		/* Y */
				FALSE, FALSE };		/* Z */


#ifdef lint                             /* Makes lint think cur_col is used */
   cur_col = 1;
   cur_line = cur_col++;
#endif

   /* Get a window to cover up the invocation window */
   win = newwin(0,0,0,0);
   output_centered(win,1,TRUE,"Change Chain Rule Settings");
   mvwaddstr(win,2,1,LINE);

   /* Get indexed access to the grammar */
   init_indexed_access(start,&access_blk);

   /* Get an array to keep track of initial rules */
   GET_MEMORY(initial, char *, access_blk.rule_cnt, char,
	      "change_chainrule_settings", 1)

   /* Remember initial chainrule status */
   temp =  start;
   for (i=0; i<access_blk.rule_cnt; i++)
	{
	initial[i] = temp->chain_rule;
	temp = temp->next_rule;
	}


   while ( (rule_num < access_blk.rule_cnt) && (!quit) )
      {
      top_rule = rule_num;
      cur_line = 4;
      (void) wmove(win,cur_line,START_COL);
      (void) wclrtobot(win);
      not_clipped = TRUE;
      cur_col = START_COL;
      (void) wmove(win,cur_line,START_COL);
      while (not_clipped && (rule_num < access_blk.rule_cnt) &&
             (cur_line < CLIP_THRESHOLD) )
         {
         rule_num++;
         left = rule_by_index(&access_blk,rule_num);
         if (left->code != NULT)                      /* Not a deleted rule */
               {
               (void) wmove(win,cur_line,START_COL);
	       if (left->chain_rule) (void) wstandout(win);
               (void) wprintw(win," %d)",rule_num);
	       if (left->chain_rule) (void) wstandend(win);
               (void) wprintw(win,"  ");
               c_output_left_node(win,left);
               not_clipped = c_output_right_side(win,left->right_side,
                                          FALSE,((RIGHT_NODE_PTR)0));
	       if (not_clipped)
		     {
                     (void) wprintw(win,"\n\n");
                     getyx(win,cur_line,cur_col);
		     }
		  else
		     {
		     rule_num--;
		     (void) wmove(win,cur_line,START_COL);
		     (void) wclrtobot(win);
		     }
               }
         }
      (void) sprintf(bottom,
        "[ %d/%d   Ret,Rule #,(I)ni,(O)rg,(N)one,(S)el,(R)em,(D)one,(H)lp ] ",
	             rule_num,access_blk.rule_cnt);
      output_centered(win, LINES -2, TRUE, bottom);
      box(win,'|','-');
      (void) wrefresh(win);
      switch (get_menubar_response(win, TRUE, char_opts, TRUE, TRUE,
					      TRUE, 1, access_blk.rule_cnt,
					      &menubar_char, &menubar_num))
	  {
	  case MENUBAR_CHAR_ONLY    :
				{
				switch (menubar_char) {
				    case 'H' :			    /* Help */
					{
					change_chainrule_settings_help();
					break;
					}
				    case 'D' :			    /* Done */
					{
					quit=TRUE;
					break;
					}
				    case 'I' :			 /* Initial */
					{
					temp =  start;
					for (i=0; i<access_blk.rule_cnt; i++)
					    {
					    temp->chain_rule = initial[i];
					    temp = temp->next_rule;
					    }
					break;
					}
				    case 'O' :			/* Original */
					{
					temp = start;
					while(temp)
					    {
					    temp->chain_rule =
						is_chainrule(temp->right_side,
						    FALSE, &dummy_sym, TRUE);
					    temp = temp->next_rule;
					    }
					break;
					}
				    case 'N' :			/* None */
					{
					temp = start;
					while(temp)
					    {
					    temp->chain_rule = FALSE;
					    temp = temp->next_rule;
					    }
					break;
					}
				    }
				rule_num = top_rule;
				break;
				}
	  case MENUBAR_NUM_ONLY     :
				{
					rule_num = menubar_num -1;
					break;
				}
	  case MENUBAR_CHAR_AND_NUM :
				{		    /* (S)elect or (R)emove */
				temp = rule_by_index(&access_blk,
							     menubar_num);
				if (is_chainrule(temp->right_side,
						 FALSE, &dummy_sym, FALSE))
				        temp->chain_rule =
					       (menubar_char=='S')?TRUE:FALSE;
				    else
					DING;	    /* It's not a chainrule */
				rule_num = top_rule;
				break;
					}
	  }
      }


   /* Put initial rules in queue, and return initial array to O.S. */
   for (i=0; i < access_blk.rule_cnt; i++)
      if (initial[i])
	  {
	  temp = rule_by_index(&access_blk,i+1);
/*	  if (temp->code == IDNT) enqueue(queue,((char *) temp)); /*No NULTS*/
	  }
   FREE_MEMORY(initial, "change_chainrule_settings", 1)



   /* Delete indexed access front end to grammar structure */
   delete_indexed_access(&access_blk);

   /* Get rid of curses window */
   (void) wclear(win);
   (void) wrefresh(win);
   delwin(win);

   }







public void view_symbol_eq()
/*
 * view_symbol_eq shows the identifiers in the current symbol table
 * that are currently equivalent.
 *
 * On Entry:
 *	A valid symbol table for grammar exists.
 *
 * On Exit:
 *	The equivalent symbols have been viewed.
*/
{
    SYMTBL old_symtbl;
    LEFT_NODE_PTR eq_gram;
    LEFT_NODE_PTR del_rule;
    char *scr_template = "/tmp/cagt_temp.XXXXXX";
    char scr_name[25];
    FILE *scr;
    int scr_fd;

    if (!num_assoc_sym())	/* Nothings been renamed yet... */
	{
	WINDOW *win;

	/* Get a window to cover up the invocation window */
	win = newwin(0,0,0,0);
	(void) wclear(win);
	(void) wrefresh(win);
	message_and_pause(win, 10, "No Renamed Identifiers Found.");
	(void) wclear(win);
	(void) wrefresh(win);
	delwin(win);
	return;
	}

    /*
     * Abstract:
     *
     * In order to reuse as much existing software as possible
     * (and thus minimize the work involved), we do the following:
     *
     *	    - Write the grammar equivalences to a scratch file
     *	      using a valid EBNF syntax (output_symbol_assoc())
     *	    - Establish a new symbol table, being careful to preserve
     *	      the current one.
     *	    - Rewind the scratch file, and read it into a grammar
     *	      structure.
     *	    - Close the scratch file, and delete it.
     *	    - Use c_output_grammar_listing to show the user the
     *	      equivalent symbols in the form of EBNF rules.
     *	    - Delete the grammar.
     *	    - Delete the new symbol table and re-establish the old one.
    */

    /* Write the scratch file */
    (void) strcpy(scr_name, scr_template);
    (void) mktemp(scr_name);
    if ((scr = fopen(scr_name, "w")) == 0)
	{
        cagt_msg(errno, MSG_RET, (cagt_msg_text, msg_arr[-(CAGT_OPNERR)],
           "Scratch", scr_name, "symbol equivalence grammar "))
	return;
	}
    output_symbol_assoc(scr);	/* Write the symbol equivalence grammar */
    if (fclose(scr) == EOF)
	cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_CLOSERR)],
           "Scratch", scr_name))

    /* Get new Symbol Table */
    old_symtbl = set_cur_symtbl(init_sym());

    /* Open the file at the Unix level for use by the source module. */
    if ((scr_fd = open(scr_name, O_RDONLY)) <= SYS_ERR)
	cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_OPNERR)],
           "Scratch", scr_name, "symbol equivalence grammar "))
		 
    /* Read the symbol equivalence "grammar" */
    eq_gram = get_ebnf_grammar(scr_name, scr_fd, FALSE);

    /* unlink the file and close it */
    if (unlink(scr_name) == -1)
        cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_RMERR)],
           scr_name))
    if (close(scr_fd) == -1)
	cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_CLOSERR)],
           "Scratch", scr_name))

    /* Display the symbol equivalences */
    c_output_grammar_listing(eq_gram, "Equivalent Symbols");

    /* Delete the grammar and the new symbol table. Restore the old table */
    while (eq_gram)
	{
	del_rule = eq_gram;
	eq_gram = eq_gram->next_rule;
	delete_rule(del_rule);
	}
    delete_symtbl();
    (void) set_cur_symtbl(old_symtbl);

}







public void output_cull_file(start)
   struct left_node *start;
/*
 *  
 * output_cull_file() places the chainrules in the internal EBNF grammar
 * structure into the cull file.
 *
*/
{
  WINDOW *win;
  FILE  *d;
  struct left_node *left;
  int chain_rules;
  
  /* Get a window to cover up the invocation window */
  win = newwin(0,0,0,0);
  box(win,'|','-');
  output_centered(win,1,TRUE,"Write Cull File");
  mvwaddstr(win,2,1,LINE);
  output_centered(win,10,FALSE,"Writing...");
  (void) wrefresh(win);
  
  /* Are there any chainrules? */
  for (left = start; left != (LEFT_NODE_PTR)0; left = left->next_rule)
    if (chain_rules = ((left->code != NULT) && (left->chain_rule))) break;

  if (chain_rules)
    {
      /* open output file */
      if ((d = fopen(cull_fname, "w")) == 0)
	cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_OPNERR)],
				   "Cull File", cull_fname, "output"));
	     
      /* Output any chainrules */
      for (left = start; left != (LEFT_NODE_PTR)0; left = left->next_rule)
	{
	  if ((left->code != NULT) && (left->chain_rule))   /* Chain rule? */
	    {
	      output_left_node(d,left);
	      output_right_side(d,left->right_side,FALSE);
	      (void) fprintf(d,"\n\n");
	    }
	}
      if (fclose(d) == EOF)
	message_and_pause(win,12,"Unable to close file. ");
      else
	message_and_pause(win,12,"Done.");
    }
  else
    {
      output_centered(win,12,FALSE,"No chainrules found.");
      message_and_pause(win,13,"Cull file not written.");
    }

  (void) wclear(win);
  (void) wrefresh(win);
  delwin(win);

}
