static char RCSid[] = "$Id: dbgen.c,v 1.11 90/04/04 17:38:22 waite Exp $";
/* Copyright, 1989, The Regents of the University of Colorado */

/* dbgen.c: generic database access functions */

/*LINTLIBRARY*/


#include <varargs.h>	/* for db{Add,Delete}Tuple */
#include "err.h"	/* Eli message functions */
#include "dbtypes.h"	/* database implementation types */

#define MSGBUFSIZ	(512)	/* size of buffers for error messages */

extern char *malloc(), *sprintf();


/* external declarations for things that depend on the DDL specification;
   these are defined in the tool-generated file ddldefs.c
*/

extern short	    dbMaxOtId;		/* number of object types - 1 */
extern short	    dbMaxRelId;		/* number of relations - 1 */
extern short	    dbMaxRoleId;	/* number of roles - 1 */
extern char	   *dbOtNames[];	/* names of object types */
extern char	   *dbAtNames[];	/* names of attributes */
extern char  	   *dbRelNames[];	/* names of relations */
extern char  	   *dbRoleNames[];	/* names of roles */
extern short  	   *dbRoles[];		/* assoc. between relations and roles */
extern short	   *dbRoleTypes[];	/* object types of roles */
extern short	   *dbSpecializers[];	/* specialization trees */
extern short	   *dbAttributes[];	/* attributes of object types */
extern dbRelation  *dbRelHdrs[];	/* relation headers */
extern short	    dbNumRoles[];	/* number of roles per relation */


/* error messages */

static char *db_E_alloc    = "allocation failure";
static char *db_E_arity    = "relation %s was declared %d-ary, not %d-ary";
static char *db_E_badot    = "undeclared object type: otype = %d";
static char *db_E_badrel   = "undeclared relation: relId = %d";
static char *db_E_badroleN = "undeclared role: roleId%d = %d";
static char *db_E_duproles = "duplicate roles: roleId%d = roleId%d = %s";
static char *db_E_init	   = "database module not initialized!";
static char *db_E_nopred   = "tuple's predecessor not found, can't happen!";
static char *db_E_noroleN  = "no roleId%d component found, can't happen!";
static char *db_E_null     = "null object not allowed here";
static char *db_E_otat     = "attribute %s not declared for object type %s";
static char *db_E_relrole  = "role %s not declared for relation %s";
static char *db_E_roletp   = "object type is %s, role type is %s for role %s";
static char *db_E_special  = "object type %s does not specialize %s";
static char *db_E_undef    = "attribute %s of object type %s is not defined";



/* initialization status: set to 1 by dbInit */

static int dbIstat = 0;



/* value representing "no object" */
#define db_nullobj	((tKey *) 0)



/* dbError: report an error message msg from function fname */
static void
dbError (severity, fname, msg)

	int   severity;
	char *fname, *msg;
{
	char buf[MSGBUFSIZ];

	(void) sprintf(buf, "database error in %s: %s", fname, msg);
	message(severity, buf, 0, &curpos);
	if (severity == DEADLY)
		exit(1);

} /* dbError */




/* isroleof: returns nonzero if roleId is a role of relation relId
   on entry -
      relId  is a valid relation name
      roleId is a valid role name
*/
static int
isroleof (relId, roleId)

	int 	     relId;
	register int roleId;
{
	register short *role;

	/* search roles of relation relId */
	for (role = dbRoles[relId]; *role >= 0; role++)
		if (*role == roleId)
			return(1);	/* found */
	
	return(0);			/* not found */

} /* isroleof */




/* issubtype: test for specialization type -
   returns nonzero if object type "cand" is a (direct or indirect)
   specialization of object type "ot";
   on entry -
      ot, cand are valid declared object types
*/
static int
issubtype (ot, cand)

	int ot, cand;
{
	register short *sub;

	/* search direct specializations of ot */
	for (sub = dbSpecializers[ot]; *sub >= 0; sub++)
		if (*sub == cand)
			return(1);	/* found cand */
	
	/* cand not found among direct specializations of ot;
	   now try indirect specializations recursively
	*/
	for (sub = dbSpecializers[ot]; *sub >= 0; sub++)
		if (issubtype(*sub, cand))
			return(1);	/* cand is specialization of sub */
	
	/* cand is neither direct nor indirect specialization of ot */
	return(0);

} /* issubtype */




/* isatof: test if attribute is declared for object type:
   returns nonzero if the attribute with id "atid" is declared
   as an attribute of object type "otype";
   otherwise, 0 is returned
   on entry -
      atid  is a declared attribute
      otype is a declared object type
*/
static int
isatof (otype, atid)

	int otype, atid;
{
	register short *at;

	/* search attribute list of object type otype */
	for (at = dbAttributes[otype]; *at >= 0; at++)
		if (*at == atid)
			return(1);	/* found */
	
	/* not found among attributes of otype */
	return(0);

} /* isatof */




/* at_defined: test whether an attribute is defined for an object --
   returns nonzero iff the attribute with id given by atid is defined
   for the object obj;
   on entry -
      obj is a non-null object
*/
static int
at_defined (obj, atid)

	tKey         *obj;
	register int  atid;
{
	register db_atrep *at;

	/* search attribute list of obj */
	for (at = obj->atlist; at; at = at->nxt)
		if (at->id == atid)
			return(1);	/* found */
	
	/* attribute not found in obj's list */
	return(0);

} /* at_defined */




/* getroletype: get the object type id of a role --
   on entry -
      relId  is the id of a declared relation
      roleId is the id of a declared role of RelId
   on exit -
      getroletype returns object type id of the role given by roleId
      in the relation given by relId
*/
static int
getroletype (relId, roleId)

	int relId, roleId;
{
	char *fname = "getroletype", buf[MSGBUFSIZ];
	register short *role;
	int		offset;

	/* find offset of roleId among relId's roles */
	for (role = dbRoles[relId]; *role >= 0; role++)
		if (*role == roleId)
			break;
	
	/* paranoid: check that we found the role */
	if (*role != roleId) {
		(void) sprintf(buf, db_E_relrole,
			dbRoleNames[roleId], dbRelNames[relId]);
		return(dbError(FATAL, fname, buf), -1);
	}

	/* return type of role */
	offset = role - dbRoles[relId];
	return(*(dbRoleTypes[relId] + offset));

} /* getroletype */




/* dbInit: initialize database module --
   returns nonzero iff everything was initialized successfully
*/
int
dbInit ()
{
	register int i;
	register dbRelation *hdr;
	char *fname = "dbInit";

	/* if already initialized, do nothing */
	if (dbIstat == 1)
		return(1);

	/* initialize relation headers */
	for (i=0; i <= dbMaxRelId; i++) {
		hdr = (dbRelation *) malloc(sizeof(dbRelation));
		if (!hdr)
			return(dbError(DEADLY, fname, db_E_alloc), 0);
		hdr->id      = i;
		hdr->arity   = dbNumRoles[i];
		hdr->firstt  = (db_tuple *) 0;
		dbRelHdrs[i] = hdr;
	}

	/* initialization successful */
	dbIstat = 1;
	return(1);

} /* dbInit */





/* dbItest: check if the database module was initialized --
   for the moment it just calls dbInit and writes an error message;
*/
static void
dbItest (fname)

	char *fname;
{
	if (!dbIstat) {
		dbError(FATAL, fname, db_E_init);
		(void) dbInit();
	}

} /* dbItest */




/* dbfNull: return "no object" */
tKey *
dbfNull ()
{
	char *fname = "dbfNull";

	/* check that database module is initialized */
	dbItest(fname);

	return(db_nullobj);

} /* dbfNull */





/* dbGetAttribute: retrieve an attribute of an object;
   on entry -
      obj  : object whose attribute value is to retrieved
      at   : id of attribute to be retrieved
      deflt: default value for attribute
   on exit -
      if the attribute at is defined for object obj,
      the attribute value is returned;
      otherwise, the default value given by deflt is returned
*/
char *
dbGetAttribute (obj, at, deflt)

	tKey        *obj;
	register int at;
	dbAtVal      deflt;
{
	char *fname = "dbGetAttribute";
	register db_atrep *ap;

	/* check that database module is initialized */
	dbItest(fname);

	/* test for null object which has no attributes */
	if (obj == db_nullobj)
		return(deflt.pval);
	
	/* search for attribute at in obj's attribute list */
	for (ap = obj->atlist; ap; ap = ap->nxt)
		if (ap->id == at)
			return(ap->val.pval);	/* found */
	
	/* attribute at not defined for obj */
	return(deflt.pval);

} /* dbGetAttribute */




/* dbSetAttribute: add or change an attribute of an object;
   on entry -
      obj    : object whose attribute value is to be modified
      at     : id of attribute to be modified
      added  : new value of attribute
      changed: changed value of attribute
   on exit -
      if the attribute at is defined for object obj,
      the attribute value is changed to the value given by "changed";
      otherwise, the attribute at is defined for obj with the value
      given by "added";
      the function returns the new value of the attribute at;
*/
char *
dbSetAttribute (obj, at, added, changed)

	tKey         *obj;
	register int  at;
	dbAtVal       added, changed;
{
	char		  *fname = "set_attribute";
	register db_atrep *ap;

	/* check that database module is initialized */
	dbItest(fname);

	/* test for null object which has no attributes */
	if (obj == db_nullobj)
		return(added.pval);
	
	/* search for attribute at in obj's attribute list */
	for (ap = obj->atlist; ap; ap = ap->nxt)
		if (ap->id == at)
			return(ap->val = changed, changed.pval); /* found */
	
	/* attribute at not yet defined for obj */
	ap = (db_atrep *) malloc(sizeof(db_atrep));
	if (ap == (db_atrep *) 0)			/* allocation failure */
		return(dbError(DEADLY, fname, db_E_alloc), added.pval);
	ap->id         = at;
	ap->nxt        = obj->atlist;			/* prepend */
	obj->atlist    = ap;
	return(ap->val = added, added.pval);

} /* dbSetAttribute */




/* dbFind: find certain objects participating in a relation
   on entry -
      relId   : a valid relationship id
      roleId1 : id of a role that belongs to relId
      obj     : an object in the database (or db_nullobj)
      roleId2 : id of a role that belongs to relId
   on exit -
      dbFind returns an object list consisting of all objects
      that appear in the roleId2 component of a tuple in the
      relation relId with the object obj in the roleId1 component
*/
tKeyList *
dbFind (relId, roleId1, obj, roleId2)

	int   relId, roleId1, roleId2;
	tKey *obj;
{
	char *fname = "dbFind";
	char  buf[MSGBUFSIZ];
	dbRelation 	  *rel;		/* header of relation relId */
	register db_tuple *tuple;
	db_olistmem	  *new;
	tKeyList 	  *list = (tKeyList *) 0;


	/* check that database module is initialized */
	dbItest(fname);

	/* check validity of relation id */
	if (relId < 0  ||  relId > dbMaxRelId) {
		(void) sprintf(buf, db_E_badrel,  relId);
		dbError(FATAL, fname, buf);
		return((tKeyList *) 0);
	}
	rel = dbRelHdrs[relId];

	/* check validity of roleId1 */
	if (roleId1 < 0  ||  roleId1 > dbMaxRoleId) {
		(void) sprintf(buf, db_E_badroleN, 1, roleId1);
		dbError(FATAL, fname, buf);
		return((tKeyList *) 0);
	}

	/* check validity of roleId2 */
	if (roleId2 < 0  ||  roleId2 > dbMaxRoleId) {
		(void) sprintf(buf, db_E_badroleN, 2, roleId2);
		dbError(FATAL, fname, buf);
		return((tKeyList *) 0);
	}

	/* check if roleId1 is a role of relation relId */
	if (!isroleof(relId, roleId1)) {
		(void) sprintf(buf, db_E_relrole,
			dbRoleNames[roleId1], dbRelNames[relId]);
		dbError(FATAL, fname, buf);
		return((tKeyList *) 0);
	}

	/* check if roleId2 is a role of relation relId */
	if (!isroleof(relId, roleId2)) {
		(void) sprintf(buf, db_E_relrole,
			dbRoleNames[roleId2], dbRelNames[relId]);
		dbError(FATAL, fname, buf);
		return((tKeyList *) 0);
	}

	/* check each tuple in relation relId for a match in roleId1 */
	for (tuple = rel->firstt; tuple; tuple = tuple->nxt) {

		register db_relobj *cp;	/* searches for components */

		/* find roleId1 component in this tuple */
		for (cp = tuple->firsto; cp; cp = cp->nxt)
			if (cp->role == roleId1)
				break;

		/* make sure we found the roleId1 component */
		if (!cp) {
			(void) sprintf(buf, db_E_noroleN, 1);
			dbError(DEADLY, fname, buf);
			return((tKeyList *) 0);
		}

		/* check if roleId1 component is object obj */
		if (cp->object != obj)
			continue;	/* no: next tuple */
		
		/* yes; now find roleId2 component in this tuple */
		for (cp = tuple->firsto; cp; cp = cp->nxt)
			if (cp->role == roleId2)
				break;
		
		/* make sure we found the roleId2 component */
		if (!cp) {
			(void) sprintf(buf, db_E_noroleN, 2);
			dbError(DEADLY, fname, buf);
			return((tKeyList *) 0);
		}

		/* create result list header (if not done yet) */
		if (!list) {
			list = (tKeyList *) malloc(sizeof(tKeyList));
			if (!list)
				return(dbError(DEADLY, fname, db_E_alloc),
					(tKeyList *) 0);
			list->first = list->pos = (db_olistmem *) 0;
		}

		/* allocate new list member for roleId2 component */
		new = (db_olistmem *) malloc(sizeof(db_olistmem));
		if (!new)
			return(dbError(DEADLY, fname, db_E_alloc), list);
		
		/* record roleId2 component and prepend to result list */
		new->object = cp->object;
		new->nxt    = list->first;
		list->first = list->pos = new;
	
	} /* for tuple */

	return(list);

} /* dbFind */



/* dbAddTuple (n, relId, roleId1,...,roleIdn, obj1,...,objn)
   adds the n-tuple of objects (obj1,...,objn) to the n-ary relation
   with id given by relId;  the correspondence of objects and roles
   is established by the roleId1,...,roleIdn arguments (i.e., obj1
   corresponds to the role with id roleId1 and so on);
   dbAddTuple returns nonzero if the n-tuple was successfully added
   to the relation and zero if the n-tuple was already a member of
   the relation;
   if any error occurs, an error message is issued and zero is returned
*/
/*VARARGS*/
int
dbAddTuple (va_alist)

	va_dcl
{
	va_list       	    ap;		/* argument pointer */
	int           	    relId;
	int		   *roles;
	tKey	          **objects;
	register int  	    n, i;
	register db_tuple  *tuple;
	register db_relobj *cp;

	char  buf[MSGBUFSIZ], *fname = "dbAddTuple";

	/* check that database module is initialized */
	dbItest(fname);

	va_start(ap);			/* initialize argument retrieval */
	n     = va_arg(ap, int);	/* arity of relation */
	relId = va_arg(ap, int);	/* relation id */

	/* check if relId is a valid relation */
	if (relId < 0  ||  relId > dbMaxRelId) {
		(void) sprintf(buf, db_E_badrel, relId);
		return(dbError(FATAL, fname, buf), 0);
	}

	/* check that relation was declared n-ary */
	if ((i = dbRelHdrs[relId]->arity) != n) {
		(void) sprintf(buf, db_E_arity, dbRelNames[relId], i, n);
		return(dbError(FATAL, fname, buf), 0);
	}

	/* create an array to hold the role ids */
	roles = (int *) malloc((unsigned) n * sizeof(int));
	if (roles == (int *) 0)
		return(dbError(DEADLY, fname, db_E_alloc), 0);
	
	/* retrieve and check the role id arguments */
	for (i=0; i < n; i++) {

		register int j;
		register int role = va_arg(ap, int);

		/* check that role id is valid */
		if (role < 0  ||  role > dbMaxRoleId) {
			(void) sprintf(buf, db_E_badroleN, i+1, role);
			free((char *) roles);
			return(dbError(FATAL, fname, buf), 0);
		}

		/* check that role id belongs to relation relId */
		if (!isroleof(relId, role)) {
			(void) sprintf(buf, db_E_relrole,
				dbRoleNames[role], dbRelNames[relId]);
			free((char *) roles);
			return(dbError(FATAL, fname, buf), 0);
		}

		/* check that role ids are unique:
		   this is O(n**2) -- we should do something better...
		*/
		for (j=0; j < i; j++)
			if (role == *(roles+j)) {
				(void) sprintf(buf, db_E_duproles,
					j+1, i+1, dbRoleNames[role]);
				free((char *) roles);
				return(dbError(FATAL, fname, buf), 0);
			}
		
		/* ok, we can accept this role */
		*(roles + i) = role;
	}

	/* allocate an array for the object arguments */
	objects = (tKey **) malloc((unsigned) n * sizeof(tKey *));
	if (objects == (tKey **) 0) {
		free((char *) roles);
		return(dbError(DEADLY, fname, db_E_alloc), 0);
	}

	/* retrieve and check the object arguments */
	for (i=0; i < n; i++) {

		register tKey *obj = va_arg(ap, tKey *);
		register int   ot;

		/* always accept the null object */
		if (obj == db_nullobj) {
			*(objects + i) = obj;
			continue;
		}

		/* non-null object: check its type */
		if ((ot = getroletype(relId, *(roles+i))) != obj->otid) {
			(void) sprintf(buf, db_E_roletp, dbOtNames[obj->otid],
				dbOtNames[ot], dbRoleNames[*(roles+i)]);
			free((char *) roles);
			free((char *) objects);
			return(dbError(FATAL, fname, buf), 0);
		}

		/* ok, accept this object */
		*(objects + i) = obj;
	}

	/* check if argument tuple is already in the relation */
	for (tuple = dbRelHdrs[relId]->firstt; tuple; tuple = tuple->nxt) {

		/* compare this tuple with argument tuple */
		for (cp = tuple->firsto; cp; cp = cp->nxt) {

			register int role = cp->role;

			/* find arg. index i corresponding to this component */
			for (i=0; i < n; i++)
				if (*(roles+i) == role)
					break;
			
			/* if objects are not equal, go to the next tuple */
			if (*(objects+i) != cp->object)
				break;
		}

		/* if tuple matches argument n-tuple, stop the search */
		if (cp == (db_relobj *) 0)
			break;
	}

	/* if tuple is non-null here, the argument tuple is already in the
	   relation
	*/
	if (tuple) {
		free((char *) roles);
		free((char *) objects);
		return(0);
	}

	/* argument tuple not found in the relation: add it */

	/* create the list of components forming the new tuple */
	cp = (db_relobj *) 0;
	for (i = n-1; i >= 0; i--) {

		register db_relobj *new;
		
		new = (db_relobj *) malloc(sizeof(db_relobj));
		if (!new)
			return(dbError(DEADLY, fname, db_E_alloc), 0);
		
		new->role   = *(roles+i);
		new->object = *(objects+i);
		new->nxt    = cp;
		cp	    = new;
	}

	/* create the tuple header and prepend new tuple to relation's list */
	tuple = (db_tuple *) malloc(sizeof(db_tuple));
	if (!tuple)
		return(dbError(DEADLY, fname, db_E_alloc), 0);
	tuple->firsto = cp;
	tuple->nxt    = dbRelHdrs[relId]->firstt;
	dbRelHdrs[relId]->firstt = tuple;
	free((char *) roles);
	free((char *) objects);
	return(1);

} /* dbAddTuple */




/* dbDeleteTuple (n, relId, roleId1, ..., roleIdn, obj1, ..., objn)
   deletes the n-tuple (obj1, ..., objn) from the n-ary relation
   with id given by relId; the correspondence of objects and roles
   is established by the roleId1,...,roleIdn arguments (i.e., obj1
   corresponds to the role with id roleId1 and so on);
   dbDeleteTuple returns nonzero if the n-tuple was successfully
   deleted from the relation and zero if the n-tuple was not a member
   of the relation prior to the call;
   if any error occurs, an error message is issued and zero is returned
*/
/*VARARGS*/
int
dbDeleteTuple (va_alist)

	va_dcl
{
	va_list       	    ap;		/* argument pointer */
	int           	    relId;
	dbRelation	   *rel;	/* relation header */
	int		   *roles;
	tKey	          **objects;
	register int  	    n, i;
	register db_tuple  *tuple;
	register db_relobj *cp;

	char  buf[MSGBUFSIZ], *fname = "dbDeleteTuple";

	/* check that database module is initialized */
	dbItest(fname);

	va_start(ap);			/* initialize argument retrieval */
	n     = va_arg(ap, int);	/* arity of relation */
	relId = va_arg(ap, int);	/* relation id */

	/* check if relId is a valid relation */
	if (relId < 0  ||  relId > dbMaxRelId) {
		(void) sprintf(buf, db_E_badrel, relId);
		return(dbError(FATAL, fname, buf), 0);
	}
	rel = dbRelHdrs[relId];

	/* check that relation was declared n-ary */
	if ((i = dbRelHdrs[relId]->arity) != n) {
		(void) sprintf(buf, db_E_arity, dbRelNames[relId], i, n);
		return(dbError(FATAL, fname, buf), 0);
	}

	/* create an array to hold the role ids */
	roles = (int *) malloc((unsigned) n * sizeof(int));
	if (roles == (int *) 0)
		return(dbError(DEADLY, fname, db_E_alloc), 0);
	
	/* retrieve and check the role id arguments */
	for (i=0; i < n; i++) {

		register int j;
		register int role = va_arg(ap, int);

		/* check that role id is valid */
		if (role < 0  ||  role > dbMaxRoleId) {
			(void) sprintf(buf, db_E_badroleN, i+1, role);
			free((char *) roles);
			return(dbError(FATAL, fname, buf), 0);
		}

		/* check that role id belongs to relation relId */
		if (!isroleof(relId, role)) {
			(void) sprintf(buf, db_E_relrole,
				dbRoleNames[role], dbRelNames[relId]);
			free((char *) roles);
			return(dbError(FATAL, fname, buf), 0);
		}

		/* check that role ids are unique:
		   this is O(n**2) -- we should do something better...
		*/
		for (j=0; j < i; j++)
			if (role == *(roles+j)) {
				(void) sprintf(buf, db_E_duproles,
					j+1, i+1, dbRoleNames[role]);
				free((char *) roles);
				return(dbError(FATAL, fname, buf), 0);
			}
		
		/* ok, we can accept this role */
		*(roles + i) = role;
	}

	/* allocate an array for the object arguments */
	objects = (tKey **) malloc((unsigned) n * sizeof(tKey *));
	if (objects == (tKey **) 0) {
		free((char *) roles);
		return(dbError(DEADLY, fname, db_E_alloc), 0);
	}

	/* retrieve and check the object arguments */
	for (i=0; i < n; i++) {

		register tKey *obj = va_arg(ap, tKey *);
		register int   ot;

		/* always accept the null object */
		if (obj == db_nullobj) {
			*(objects + i) = obj;
			continue;
		}

		/* non-null object: check its type */
		if ((ot = getroletype(relId, *(roles+i))) != obj->otid) {
			(void) sprintf(buf, db_E_roletp, dbOtNames[obj->otid],
				dbOtNames[ot], dbRoleNames[*(roles+i)]);
			free((char *) roles);
			free((char *) objects);
			return(dbError(FATAL, fname, buf), 0);
		}

		/* ok, accept this object */
		*(objects + i) = obj;
	}

	/* find the argument tuple */
	for (tuple = rel->firstt; tuple; tuple = tuple->nxt) {

		/* compare this tuple with argument tuple */
		for (cp = tuple->firsto; cp; cp = cp->nxt) {

			register int role = cp->role;

			/* find arg. index i corresponding to this component */
			for (i=0; i < n; i++)
				if (*(roles+i) == role)
					break;
			
			/* if objects are not equal, go to the next tuple */
			if (*(objects+i) != cp->object)
				break;
		}

		/* if tuple matches argument tuple, stop searching */
		if (cp == (db_relobj *) 0)
			break;
	
	}

	/* if tuple is null here, the argument tuple is not in the relation */
	if (!tuple) {
		free((char *) roles);
		free((char *) objects);
		return(0);
	}

	/* argument tuple found: delete it */

	/* unlink tuple from relation's tuple list */
	if (rel->firstt == tuple)
		rel->firstt = tuple->nxt;	/* it's the first tuple */
	else {
		/* not the first tuple - find predecessor */
		register db_tuple *t;
		for (t = rel->firstt; t; t = t->nxt)
			if (t->nxt == tuple)
				break;
		
		/* make sure we found tuple's predecessor */
		if (!t)
			dbError(DEADLY, fname, db_E_nopred);

		/* unlink tuple */
		t->nxt = tuple->nxt;
	}

	/* deallocate tuple's components */
	for (cp = tuple->firsto; cp; ) {
		register db_relobj *next = cp->nxt;
		free((char *) cp);
		cp = next;
	}

	/* all done, return success */
	free((char *) roles);
	free((char *) objects);
	return(1);

} /* dbDeleteTuple */




/* dbOlReset: rewind an object list --
   the object list "ol" is rewound and returned,
   the next call to dbOlNext will return the first member of the list
*/
tKeyList *
dbOlReset (ol)

	tKeyList *ol;
{
	char *fname = "dbOlReset";

	/* check that database module is initialized */
	dbItest(fname);

	/* check for null list */
	if (ol == (tKeyList *) 0)
		return(ol);
	
	/* reset current position pointer of list */
	ol->pos = ol->first;
	return(ol);

} /* dbOlReset */




/* dbOlNext: return next member of an object list --
   the list member at the current position of the list "ol" is returned
   and the current position pointer of the object list "ol" is advanced;
   if the current position in "ol" is the end of the list or
   if the list is empty, the object dbNull is returned
*/
tKey *
dbOlNext (ol)

	tKeyList *ol;
{
	char *fname = "dbOlNext";
	tKey *cur;

	/* check that database module is initialized */
	dbItest(fname);

	/* check for null list */
	if (ol == (tKeyList *) 0)
		return(db_nullobj);
	
	/* check for end of list */
	if (ol->pos == (db_olistmem *) 0)
		return(db_nullobj);
	
	/* return object at current position, then advance */
	cur = ol->pos->object;
	ol->pos = ol->pos->nxt;
	return(cur);

} /* dbOlNext */



/* dbNewObject: creates a new object of the object type given by "otype" */
tKey *
dbNewObject (otype)

	int otype;
{
	char  buf[MSGBUFSIZ], *fname = "dbNewObject";
	tKey *new;

	/* check that database module is initialized */
	dbItest(fname);

	/* check that otype is a valid object type id */
	if (otype < 0  ||  otype > dbMaxOtId) {
		(void) sprintf(buf, db_E_badot, otype);
		return(dbError(FATAL, fname, buf), db_nullobj);
	}

	/* allocate new object */
	new = (tKey *) malloc(sizeof(tKey));
	if (new == db_nullobj)
		return(dbError(DEADLY, fname, db_E_alloc), db_nullobj);

	/* initialize and return new object */
	new->otid = otype;
	new->atlist = (db_atrep *) 0;
	return(new);

} /* dbNewObject */



/* dbGetObjectType: returns the object type id of the object "obj" */
int
dbGetObjectType (obj)

	tKey *obj;
{
	char *fname = "dbGetObjectType";

	/* check that database module is initialized */
	dbItest(fname);

	/* check that obj is a non-null object */
	if (obj == db_nullobj)
		return(dbError(FATAL, fname, db_E_null), -1);

	return(obj->otid);

} /* dbGetObjectType */




/* dbSetObjectType: move an object into a specialization object type -
   returns 1 if obj was successfully moved into type otype,
   0 otherwise;
   error messages are given if
   - obj is dbNull
   - otype is not a declared object type
   - otype is not a specialization of (or equal to) obj's current type
   - obj has attributes defined that are not declared (or inherited) for otype
*/
int
dbSetObjectType (obj, otype)

	tKey *obj;
	int   otype;
{
	char *fname = "dbSetObjectType";
	char  buf[MSGBUFSIZ];
	register db_atrep *at;

	/* check that database module is initialized */
	dbItest(fname);

	/* check that obj is not null */
	if (obj == db_nullobj)
		return(dbError(FATAL, fname, db_E_null), 0);
	
	/* check that otype is a valid object type */
	if (otype < 0  ||  otype > dbMaxOtId) {
		(void) sprintf(buf, db_E_badot, otype);
		return(dbError(FATAL, fname, buf), 0);
	}

	/* check that otype is a specialization of obj's current type */
	if (otype != obj->otid  &&  !issubtype(obj->otid, otype)) {
		(void) sprintf(buf, db_E_special,
			dbOtNames[otype], dbOtNames[obj->otid]);
		return(dbError(FATAL, fname, buf), 0);
	}

	/* check that attributes of obj are consistent with otype */
	for (at = obj->atlist; at; at = at->nxt)
		if (!isatof(otype, at->id)) {
			(void) sprintf(buf, db_E_otat,
				dbAtNames[at->id], dbOtNames[otype]);
			return(dbError(FATAL, fname, buf), 0);
		}

	/* change object type and return success */
	return(obj->otid = otype, 1);

} /* dbSetObjectType */



/* dbTestObjectType: test if object's attributes are consistent with object type --
   returns nonzero iff all attributes defined for object "obj" are declared
   attributes of object type "otype" and vice versa;
   if "msg" is nonzero, any inconsistencies found will be reported
*/
int
dbTestObjectType (obj, otype, msg)

	tKey *obj;
	int   otype;
	int   msg;
{
	char *fname = "dbTestObjectType";
	char buf[MSGBUFSIZ];

	int 		   okflag = 1;
	register db_atrep *at;
	register short    *atid;

	/* check that database module is initialized */
	dbItest(fname);

	/* check that obj is not null */
	if (obj == db_nullobj) {
		if (msg) dbError(FATAL, fname, db_E_null);
		return(0);
	}
	
	/* check that otype is a valid object type id */
	if (otype < 0  ||  otype > dbMaxOtId) {
		if (msg) {
			(void) sprintf(buf, db_E_badot, otype);
			dbError(FATAL, fname, buf);
		}
		return(0);
	}

	/* check that obj's attributes are declared for otype */
	for (at = obj->atlist; at; at = at->nxt)
		if (!isatof(otype, at->id)) {
			okflag = 0;
			if (msg) {
				(void) sprintf(buf, db_E_otat,
					dbAtNames[at->id], dbOtNames[otype]);
				dbError(FATAL, fname, buf);
			}
		}
	
	/* check that otype's attributes are defined for obj */
	for (atid = dbAttributes[otype]; *atid >= 0; atid++)
		if (!at_defined(obj, *atid)) {
			okflag = 0;
			if (msg) {
				(void) sprintf(buf, db_E_undef,
					dbAtNames[*atid], dbOtNames[otype]);
				dbError(FATAL, fname, buf);
			}
		}
	
	return(okflag);

} /* dbTestObjectType */

