/*
** $Id: ccomarl.h,v 1.2 90/10/23 11:37:08 cogito Exp $
*/

/* ccomarl.h - interface to COMAR library routines */
/* Last Change: 16.02.90	From: Kalle */

#ifndef ccomarl_DEF
#define ccomarl_DEF


/* TABLE of CONTENTS:
 *
 * 	1. Higher-level access routines than provided by ccomar module.
 * 	2. Add objects while implicitly performing symbol table insertion.
 * 	3. Access delimited lists (preceded by begin-list delimiter).
 *	4. Cycle through productions of a COMAR specification. 
 *	5. Subtree and structure copy, equality, pattern match. 
 *
 */



/* SECTION 1. Higher-level access routines than provided by ccomar module. */

/* cmrl_didtosymb - return pointer to string for symbol identified by did 
 *	exit:	cmrl_didtosymb == NULL ==> did represents no definition in c
 *		else *cmrl_didtosymb == string representation of symbol
 *		    represented by did in c.
 */
extern char *cmrl_didtosymb( /* p_comar c; DID did; */ );


/* cmrl_sidtosymb - return pointer to string for symbol identified by sid
 *      exit:   cmrl_sidtosymb == NULL ==> sid represents no symbol in c
 *              else *cmrl_sidtosymb == string representation of symbol
 *                  represented by sid in c.
 */
/* extern char *cmrl_sidtosymb(p_comar c, SID sid); */
#define	cmrl_sidtosymb(c,sid)	cmr_get_symb_entry_sym(c,sid)


/* cmrl_strtosid - return sid from COMAR symbol part, corresponding to string
 *	exit:	cmrl_strtosid == CMR_UNKERR ==> 
 *		    string unknown in c or else not a P_STRING.
 *		else cmrl_strtosid == sid for string.
 */
extern SID cmrl_strtosid( /* p_comar c; char *str; */ );


/* cmrl_namtosid - return sid from COMAR symbol part, corresponding to name
 *	exit:	cmrl_namtosid == CMR_UNKERR ==> 
 *		    name unknown in c or else not a P_NAME.
 *		else cmrl_namtosid == sid for name.
 */
extern SID cmrl_namtosid( /* p_comar c; char *nam; */ );



/* SECTION 2. Add objects while implicitly performing symbol table insertion */


/* cmrl_add_string - add string to symb_part of c, return SID.
 *	entry:	str == a COMAR string.
 *	exit:	if str is already in symb_part then
 *		    sid == the sid of the previously entered string,
 *		else sid == newly generated sid.
 */
/* extern SID cmrl_add_string(p_comar c; char *str); */
#define	cmrl_add_string(c,str)	cmr_new_symb_entry(c, P_STRING, str)


/* cmrl_add_name - add name to symb_part of c; return SID.
 *	entry:	nam == a COMAR name.
 *	exit:	if nam is already in symb_part then
 *		    sid == the sid of the previously entered name,
 *		else sid == newly generated sid.
 */
/* extern SID cmrl_add_name(p_comar c; char *nam); */
#define	cmrl_add_name(c,nam)	cmr_new_symb_entry(c, P_NAME, nam)



/*  SECTION 3. Access delimited lists (preceded by begin-list delimiter) */

/* cmrl_dlist_firstis - return TRUE iff tag of first item in dlist matches ptag
 *	entry:	ptag == COMAR node tag
 *	exit:	TRUE iff tag of first item in dlist matches ptag
 *		else cmrl_dlist_firstis == FALSE.
 */
extern Boolean cmrl_dlist_firstis( /* pGenList dlist; int ptag */ );



/* The following four macros should not be used in future !
 * In the IDL-implementation exist no delimited lists. Therefore
 * they are no longer needed.
 */

#define cmrl_dlist_len(dlist)  (cmr_list_len(dlist))


/* cmrl_splice_list_in_dlist - returns dlist after list spliced in 
 *                             after item in dlist addressed by p
 * on entry : list == undelimited list
 *            p == pointer to item in dlist or (Gen_class)NULL
 * on exit  : dlist with list spliced in after item in dlist addressed by p. 
 *
 * extern pGenList cmrl_splice_list_in_dlist( * pGenList list, dlist; Gen_class p * );
 * So used in earlier implementations.
 */
#define cmrl_splice_list_in_dlist(list,dlist,p)  (dlist = cmr_splice_lists(list,dlist,p))


/* cmrl_splice_dlists - returns dlist2 after dlist1 spliced in
 *                     after item in dlist2 addressed by p
 * on entry : dlist1, dlist2 == delimited lists
 *            p == pointer to item in dlist2 or (pGenList)NULL
 * on exit  : dlist2 with dlist1 spliced in after item in dlist2 addressed by p.
 *
 *extern pGenList cmrl_splice_dlists(* pGenList dlist1, dlist2, Gen_class p *);
 * So used in earlier implementations.
 */
#define cmrl_splice_dlists(dlist1,dlist2,p) (dlist2 = cmr_splice_lists(dlist1,dlist2,p))


/* cmrl_dlist_delitem - returns dlist with item deleted addressed by p
 * on entry : dlist == delimited list
 *            p == pointer to item in dlist
 * on exit  : dlist with item pointed to by p removed from dlist
 *
 *extern pGenList cmrl_dlist_delitem(* pGenList dlist; Gen_class p *);
 * So used in earlier implementations.
 */
#define cmrl_dlist_delitem(dlist,p)  (dlist = cmr_list_delitem(dlist,p))


/* SECTION 4. Cycle through productions of a COMAR specification. */

/* cmrl_first_prod - return did of first production in c.
 *	exit:	cmrl_first_prod == -1 ==> there are no productions in c
 *		else cmrl_first_prod == did of first production.
 */
/* extern DID cmrl_first_prod(p_comar c); */
#define	cmrl_first_prod(c)	cmrl_next_prod(c, 0)


/* cmrl_next_prod - return did of production after that identified by p in c.
 *	exit:	cmrl_first_prod == -1 ==> there are no productions after p in c
 *		else cmrl_first_prod == did of first production after p in c.
 */
extern DID cmrl_next_prod( /* p_comar c; DID p; */ );


/************* Routines added by PA ********************************/

/* cmrl_next_nterm - return did of nonterminal after that identified by p in c.
 * on exit : cmrl_next_nterm == -1 ==> there are no nonterminals after p in c
 *           else did of next nonterminal after that identified by p
 */
extern DID cmrl_next_nterm( /* p_comar c; DID p; */ );



/* cmrl_first_nterm - return did of first nonterminal in c
 * on exit : cmrl_first_nterm == -1 ==> there is no nonterminal in c
 *           else 
 *                did of first nonterminal in c
 */
/* extern DID cmrl_first_nterm(p_comar c); */
#define	cmrl_first_nterm(c)	cmrl_next_nterm(c, 0)


/* cmrl_next_term - return did of terminal after that identified by p in c
 * on exit : cmrl_next_term == -1 ==> there are no terminals after p in c
 *           else 
 *               did of next terminal after that identified by p
 */
extern DID cmrl_next_term( /* p_comar c; DID p; */ ); 


/* cmrl_first_term - return did of first terminal in c
 * on exit : cmrl_first_term == -1 ==> there is no terminal in c
 *           else
 *               did of first terminal in c 
 */
/* extern DID cmrl_first_term(p_comar c); */
#define	cmrl_first_term(c)	cmrl_next_term(c, 0)


/* cmrl_next_other - return did of other after that identified by p in c 
 * on exit : cmrl_next_other == -1 ==> there are no other after p in c
 *           else
 *               did of next other after that identified by p
 */
extern DID cmrl_next_other( /* p_comar c; DID p; */ );  


/* cmrl_first_other - return did of first other in c 
 * on exit : cmrl_first_other == -1 ==> there is no other in c
 *           else
 *               did of first other in c
 */
/* extern DID cmrl_first_other(p_comar c);  */
#define	cmrl_first_other(c)	cmrl_next_other(c, 0)



/* cmrl_next_prod_with_lhs - return did of production with lhs 
 *                                  after that identified by p in c 
 * on exit : cmrl_next_prod_with_lhs == -1 ==> there are no productions with
 *                         lhs after that identified by p in c
 *           else 
 *               did of next production with lhs after that identified by p
 */
extern DID cmrl_next_prod_with_lhs(/* p_comar c; DID p; DID lhs */);


/* cmrl_first_prod_with_lhs - return did of first production with lhs in c
 * on exit : cmrl_first_prod_with_lhs == -1 ==> there is no production with
 *                          lhs in c
 *           else
 *               did of first production with lhs in c
 */
/* extern DID cmrl_first_prod_with_lhs(p_comar c; DID lhs); */
#define	cmrl_first_prod_with_lhs(c,lhs)	cmrl_next_prod_with_lhs(c, 0, lhs)


#define CMRL_BUFSIZE 80
#define CMRL_NTPREFIX "CMRHELP"

/* cmrl_gen_nterm - return did of a new generated nonterminal
 * on exit : a new generated nonterminal is added to SIDTABLE
 *           and it is added to DIDTABLE
 *           it returns the did of the new nonterminal
 */
extern DID cmrl_gen_nterm( /* p_comar c; */ );  


/* SECTION 5. Subtree and structure copy, equality, pattern match. */



/************ ROUTINES for copying subtrees ************/


/* cmrl_copy_list - returns a copy of an undelimited list
 * on entry : t points to list element
 * on exit  : returns a copy of the undelimited list
 *            do not use this function on subtree constructs
 */
extern pGenList cmrl_copy_list( /* pGenList t */ );

/* cmrl_copy_subtr - returns a copy of the given subtree
 * on entry: t points to a COMAR node
 * on exit : returns a copy of subtree t if t is not tagged with P_ITEM
 *           (tCOMAR *)NULL otherwise
 */
extern Gen_class cmrl_copy_subtr( /* Gen_class t */ );


/* cmrl_copy_list_with_subst - returns a modified copy of an undelimited list
 * on entry : t points to a list element
 *            old points to a node in list t or adjacent subtrees
 *            new points to a list or a subtree
 * on exit  : modified copy of list t
 *            id old is not a node in list t or adjacent subtrees
 *               then t will only be copied
 *            do not use this function on subtree constructs
 */
extern pGenList cmrl_copy_list_with_subst( /* pGenList t, old, new; */ );


/* cmrl_copy_subtr_with_subst - returns a modified copy of subtree t
 * on entry: t points to a subtree
 *           old points to a node in subtree t
 *           new points to a subtree
 * on exit : modified copy of subtree t
 *           if old is not a node in t then t will only be copied
 *           if t is tagged with P_ITEM it returns ()NULL
 */
extern Gen_class cmrl_copy_subtr_with_subst( /* Gen_class  t, old, new; */ );


/************ other usefull routines ****************************/


/* cmrl_cmbine_rules_with_lhs - combines all rules with given lhs
 * on entry: lhs is a nonterminal
 * on exit : all rules with given lhs are copied and will be 
 *           combined to one production tree
 *           (tCOMAR *)NULL on error
 */
extern pGenList cmrl_cmbine_rules_with_lhs( /* p_comar c, DID lhs */ );


/* cmrl_get_eps_did - returns did of S_EMPTY
 * on exit : if the empty word is in c then did of S_EMPTY
 *           else the empty word will be added to DIDTABLE
 *                and the new generated did will be returned
 */
extern DID cmrl_get_eps_did( /* p_comar c */ );


/* cmrl_get_number_of_nterms - returns the number of nonterminals
 * on entry : the grammar defined in COMAR databse c
 * on exit  : number of nonterminals in the grammar c
 */
extern int cmrl_get_number_of_nterms( /* p_comar c */ );


/* cmrl_sid_to_firstdid - returns the first did for given sid
 * on entry : the grammar defined in COMAR databse c
 *            sid index for symboltable entry
 * on exit  : CMR_UNKERR if there is no did in DEFTABLE with given sid
 *            else first did in DEFTABLE with given sid
 */
extern DID cmrl_sid_to_firstdid( /* p_comar c, SID sid */ );

/* cmrl_sid_to_nextdid - returns the next did for given sid and did
 * on entry : the grammar defined in COMAR databse c
 *            sid index for symboltable entry
 *            did index in deftable from which the search goes on
 * on exit  : CMR_UNKERR if there is no further did in DEFTABLE with given sid
 *            else next did in DEFTABLE with given sid
 */
extern DID cmrl_sid_to_nextdid( /* p_comar c, SID sid, DID did */ );

#endif !ccomarl_DEF
