static char RCSid[] = "$Id: io_curses.c,v 1.18 1992/08/17 13:36:56 waite Exp $";
/* Copyright, 1989, The Regents of the University of Colorado */

/* 							July 13, 1986
 * Created module as part of reorganization to isolate all curses
 * related routines,
 *
 *    It is desirable to present a common style in the user interface
 * to cagt. This means that certain operations involving curses
 * occur repeatedly. To reduce the size of the code, improve
 * readability and increase dependability, such operations are
 * implemented as routines in this module.
 *
 * Routines:
 *			- P R O M P T I N G -
 *      1) getline(*) - UNIX curses input routine
 *      2) tflush(*) - UNIX terminal typeahead buffer flushing routine.
 *      3) get_boolean_response - Gets Yes/No answer from user.
 *      4) get_unsigned_int_response - Gets unsigned integer from user.
 *	5) get_menubar_response - Reads character, number pairs from the
 *				  user in a standard way for use in
 *				  interpreting user menu bar responses
 *      6) get_identifier - Gets an identifer (which doesn't have to already
 *			    exist in the symbol table) from user and returns
 *                          a SYMBOL for it.
 *	7) output_centered - Centers text on a given line.
 *	8) message_and_pause - Centers a message on a given line and
 *			       waits for the user to hit Return.
 *
 *			- G R A M M A R   O U T P U T -
 *	1) c_prtsym - Print a symbol table entry in a curses window.
 *      2) c_output_left_node - Outputs the contents of an existing left node
 *                              using curses.
 *      3) c_output_right_node - Outputs an existing right node using curses.
 *      4) c_output_right_side - Outputs the contents of an existing right
 *                               side using curses.
 *      5) c_output_grammar_listing - Outputs an existing grammar to the
 *                                    screen in a style similar to UNIX more
 *                                    using curses.
 *	6) c_select_grammar_rules -  Outputs an existing grammar to the
 *				     screen in a style similar to the UNIX
 *				     more utility, allowing the user to
 *				     select rules from the grammar.
 *	7) c_select_identifiers -  Outputs the identifiers currently in
 *				     use in an existing grammar to the
 *				     screen allowing the user to
 *				     select identifiers from the grammar.
 *      8) c_output_rule_pair - Outputs a pair of rules with the ability
 *                              to highlight a specific node.
 *
 *      (*) getline and tflush courtesy of Andy Rudoff and Sylvan Rudd
 *
*/




#include <stdio.h>
#include <stdlib.h>
#include <curses.h>
#include <math.h>
#include <ctype.h>
#include "cagt_config.h"
#include "cagt_usr_err.h"
#include "support.h"
#include "gram.h"
#include "queue.h"
#include "io_curses.h"

#ifndef VAX11C
#include <sys/ioctl.h>                      /* Needed by getline and tflush */
#endif





#define MAX_ANS 80
public char ans_str[MAX_ANS];          	       /* Holds ASCII user response */
public char bool_str[MAX_ANS];	       /* Holds ASCII user boolean response */


/* The following is to make drawing lines easier */
public char LINE[] = 
"--------------------------------------------------------------------------------------------------------------------------------------------------------------";



/* menubar character options for simple listing routine */
private MENUBAR_OPT_ARR listing_char_opts =
		     	      { FALSE, FALSE,		/* A */
				FALSE, FALSE,		/* B */
				FALSE, FALSE,		/* C */
				TRUE , FALSE,		/* Done */
				FALSE, FALSE,		/* E */
				FALSE, FALSE,		/* F */
				FALSE, FALSE,		/* G */
				TRUE , FALSE,		/* Help */
				FALSE, FALSE,		/* I */
				FALSE, FALSE,		/* J */
				FALSE, FALSE,		/* K */
				FALSE, FALSE,		/* L */
				FALSE, FALSE,		/* M */
				FALSE, FALSE,		/* N */
				FALSE, FALSE,		/* O */
				FALSE, FALSE,		/* P */
				FALSE, FALSE,		/* Q */
				FALSE, FALSE,		/* R */
				FALSE, FALSE,		/* S */
				FALSE, FALSE,		/* T */
				FALSE, FALSE,		/* U */
				FALSE, FALSE,		/* V */
				FALSE, FALSE,		/* W */
				FALSE, FALSE,		/* X */
				FALSE, FALSE,		/* Y */
				FALSE, FALSE };		/* Z */



/* menubar character options for rule and identifier selection */
private MENUBAR_OPT_ARR selection_char_opts =
			      { TRUE , FALSE,		/* All */
				FALSE, FALSE,		/* B */
				FALSE, FALSE,		/* C */
				TRUE , FALSE,		/* Done */
				FALSE, FALSE,		/* E */
				FALSE, FALSE,		/* F */
				FALSE, FALSE,		/* G */
				TRUE , FALSE,		/* Help */
				FALSE, FALSE,		/* I */
				FALSE, FALSE,		/* J */
				FALSE, FALSE,		/* K */
				FALSE, FALSE,		/* L */
				FALSE, FALSE,		/* M */
				TRUE , FALSE,		/* None*/
				FALSE, FALSE,		/* O */
				FALSE, FALSE,		/* P */
				FALSE, FALSE,		/* Q */
				TRUE , TRUE ,		/* Remove # */
				TRUE , TRUE ,		/* Select # */
				FALSE, FALSE,		/* T */
				FALSE, FALSE,		/* U */
				FALSE, FALSE,		/* V */
				FALSE, FALSE,		/* W */
				FALSE, FALSE,		/* X */
				FALSE, FALSE,		/* Y */
				FALSE, FALSE };		/* Z */








/* --------------- P R O M P T I N G ---------------------------------------*/

#ifndef VAX11C                                /* UNIX curses input routines */
/*
**      getline.c -- a line input routine smarter that curses
**
**      15 august 1983
**
**      home directory: HOME/lib
**
**      andrew m. rudoff
**      sylvan o.l. ruud
**      university of colorado
**
**      gets a line of input into string, ignores input past len-1 characters
*/


private tflush();

public void getline(win, string, len)
WINDOW *win;
char    *string;
int     len;            /* maximum length of string (including '\0') */

{
int     nchars = 0;             /* index into string (next unused) */
int     c;
int     ch;


        /* flush the input buffer! */

        tflush(stdin);

        while ((c = getc(stdin)) != '\n')
        {
                if ((nchars >= (len - 1)) && (c != '\b'))
                {
                        (void) putchar('\007');
                        (void) fflush(stdout);
                        continue;
                }

                if (c == '\033')
                {
                        if ((c = getc(stdin)) != 'O' && (c != '['))
                                (void) ungetc(c, stdin);

                        else
                        {
                                ch = getc(stdin);
                                c = '\b';
                        }
                }

                else if (c == '\177')
                        c = '\b';




                if ((c == '\b') && nchars)
                {
                        nchars--;
                        (void) waddch(win, '\b');
                        (void) waddch(win, ' ');  /* destructive backspace */
                        (void) waddch(win, '\b');
                        (void) (void) wrefresh(win);
                }

                else if (c == '\014')   /* ^L -- redraw the screen */
                {
                        (void) wrefresh(curscr);
                }

                else if (c < '\040')
                {
                        (void) putchar('\007');
                        (void) fflush(stdout);
                }

                else
                {
                        string[nchars++] = c;
                        (void) waddch(win, c);
                        (void) wrefresh(win);
                }
        }

        string[nchars] = '\0';

        (void) waddch(win, '\n');
        (void) wrefresh(win);
#ifdef  lint
        ch = ch; /* to shut up lint */
#endif
}







private tflush(stream)
FILE *stream;
{
        int     typahead;
        int     ch;


        (void) ioctl(fileno(stream), (int) FIONREAD, (char *) &typahead);

        while (typahead--)
                ch = getc(stream);
#ifdef lint
/*shut it up!*/
        ch = ch;
#endif
}


#endif                                        /* UNIX curses input routines */






public int get_boolean_response(win,bell)
   WINDOW *win;
   char bell;
/*
 * On Entry:
 *      win points at a valid curses window.
 *      bell is TRUE if bad entries should ring the bell.
 *      The calling routine has just asked the user a Yes-No type question.
 *
 * On Exit:
 *      If the user answers 'No', get_boolean_response returns FALSE,
 *         otherwise TRUE.
*/
   {
   int line;
   int col;
   short ptr;                                      /* Points at char to use */
   int result;


   getyx(win,line,col);
   do {
      GET_STRING(win,bool_str)

      ptr = 0;
      while (ptr < MAX_ANS-1)
         if (bool_str[ptr] == ' ')
               ptr++;
            else
               break;

      if ( result = ((bool_str[ptr] != 'Y') && (bool_str[ptr] != 'N') &&
                     (bool_str[ptr] != 'y') && (bool_str[ptr] != 'n')) )
                {
                if (bell) DING;                                /* Ring bell */
                (void) wmove(win,line,col);
                (void) wclrtoeol(win);
                box(win,'|','-');
                (void) wrefresh(win);
                }
      } while ( result);

   return((bool_str[ptr] == 'Y') || (bool_str[ptr] == 'y'));

   }








public int get_unsigned_int_response(win, bell, allow_empty, check_range,
				     lower, upper)
   WINDOW *win;
   char bell;
   char allow_empty;
   char check_range;
   int lower,
       upper;
/*
 * On Entry:
 *      win points at a valid curses window.
 *      bell is TRUE if bad entries should ring the bell.
 *      The calling routine has just asked the user a question requiring
 *         an unsigned integer in response..
 *      allow_empty is TRUE if the user is allowed to type a string
 *         containing nothing, a space or a tab.
 *      check_range is TRUE if the response must fall between lower
 *         and upper (lower <= response <= upper).
 *
 * On Exit:
 *      If the user enters a valid unsigned integer,
 *         get_unsigned_int_response returns an integer meeting the
 *         above requirements.
 *      If the user enters an empty string or a string containing only
 *         spaces and tabs and allow_empty is TRUE, get_unsigned_int_response
 *	   returns -1.
*/
   {
   int line;						   /* Line on entry */
   int col;						    /* Col on entry */
   int ans;                                       /* Holds potential result */
   char *i;
   char *cur;				      /* Current position in string */
   char valid = FALSE;				   /* TRUE if input is good */



   getyx(win,line,col);			/* Remember initial screen position */

   do {
      GET_STRING(win,ans_str)				    /* Get response */
      cur = ans_str;

      /* Strip leading spaces and tabs - if empty handle here */
      while ( (*cur != '\0') &&
	      ((*cur == ' ') || (*cur == '\t')) ) cur++;

      if (*cur == '\0')
	  {
	  if (allow_empty)
	      {
	      ans = -1;
	      valid = TRUE;
	      }
	  goto loop_bottom;
	  }


      /* Check for numeric characters */
      i = cur;
      while ((*i != '\0') && (*i >= '0') && (*i <= '9')) i++;
      /* Skip trailing spaces and tabs */
      while ( (*i != '\0') && ((*i == ' ') || (*i == '\t')) ) i++;
	  if (*i != '\0') goto loop_bottom;
      (void) sscanf(cur,"%d",&ans);		   /* Convert to int */
      if (check_range)
	  if ( (ans < lower) || (ans > upper) ) goto loop_bottom;
      valid = TRUE;

	
   loop_bottom:
      if (!valid)
         {
         (void) wmove(win,1,1);		       /* Cursor not on bottom line */
         (void) wrefresh(win);
         if (bell) DING;                                       /* Ring Bell */
         (void) wmove(win,line,col);
         (void) wclrtoeol(win);			     /* Wipe out bad answer */
         (void) wmove(win,line,col);
         box(win,'|','-');
         (void) wrefresh(win);
         }

      } while (!valid);

   return(ans);
   }







public int get_menubar_response(win,bell,char_opts,allow_empty,allow_num_only,
       	                        check_range,lower,upper,leading_char,num)
   WINDOW *win;
   char bell;
   MENUBAR_OPT_ARR char_opts;
   char allow_empty;
   int allow_num_only;
   char check_range;
   int lower,
       upper;
   char *leading_char;
   int *num;
/*
 * Description:
 *	This routine is called directly after the calling routine has
 *	   asked the user a question requiring a response with the
 *	   general form:
 *
 *			char num
 *
 *	   where char is a single letter A-Z (case insensitive) and
 *	   num is an unsigned integer value. There need not be any
 *	   space between char and num. It is possible to specify that
 *	   the character may be omitted (allow_num_only). Char_opts
 *	   specifies for each letter of the alphabet whether that
 *	   letter is allowed and whether it has a numeric field.
 *	   Empty responses can be allowed (allow_empty). In the cases
 *	   where the number is allowed, it is posssible to specify range
 *	   checking on it's value.
 *
 * On Entry:
 *      win points at a valid curses window.
 *      bell is TRUE if bad entries should ring the bell.
 *	char_opts is an array of 26 elements - one per each letter
 *	   of the alphabet (A-Z). Each element has a
 *	   char_allowed field that dictates if that character
 *	   is allowed, and a num field that dictates if
 *	   a number follows the character.
 *      allow_empty is TRUE if the user is allowed to type a string
 *         containing nothing, a space or a tab.
 *	allow_num_only is TRUE if the user is allowed to leave out the
 *	   character part and type only a number.
 *      check_range is TRUE if the response must fall between lower
 *         and upper (lower <= response <= upper).
 *	if check_range is TRUE, lower contains the lower bound for
 *	   the allowed range and upper contains the upper bound
 *	   (inclusive).
 *
 * On Exit:
 *	If the user enters an empty string or a string containing
 *	   only spaces and tabs and allow_empty is TRUE,
 *	   get_menubar_response returns MENUBAR_EMPTY.
 *
 *	If the user enters a character that is allowed by
 *	   char_opts:
 *
 *		If char_opts did not allow a following number:
 *			- get_menubar_response returns
 *			  MENUBAR_CHAR_ONLY
 *			- *leading_char contains it's ASCII code.
 *			- *num is 0.
 *
 *		If char_opts does allow a following number:
 *			- get_menubar_response returns MENUBAR_CHAR_AND_NUM.
 *			- *leading_char contains it's ASCII code.
 *			- *num contains the number (if present).
 *
 *	If the user enters only the unsigned integer part and
 *	   allow_num_only is TRUE:
 *		- get_menubar_response returns MENUBAR_NUM_ONLY
 *		- *leading_char contains '\0'
 *		- *num contains the number.
*/
   {
   int status;						    /* Return value */
   int line;						   /* Line on entry */
   int col;						    /* Col on entry */
   char *i;
   char *cur;

   char num_found;				/* TRUE if number part seen */
   char char_found;			     /* TRUE if character part seen */
   char valid = FALSE;				   /* TRUE if input is good */



   getyx(win,line,col);			/* Remember initial screen position */

   do {
      GET_STRING(win,ans_str)				    /* Get response */

      char_found = FALSE;				  /* Preassumptions */
      num_found = FALSE;
      cur = ans_str;

      /* Strip leading spaces and tabs - if empty handle here*/
      while ( (*cur != '\0') &&
	      ((*cur == ' ') || (*cur == '\t')) ) cur++;

      if (*cur == '\0')
	  {
	  if (allow_empty)
	      {
	      status = MENUBAR_EMPTY;
	      *leading_char = '\0';
	      *num = 0;
	      valid = TRUE;
	      }
	  goto loop_bottom;
	  }


      /* Is there a leading character */
      *cur = _toupper(*cur);
      if ((*cur >= 'A') && (*cur <= 'Z'))
	      {
	      *leading_char = *cur;
	      char_found = TRUE;
	      cur++;
	      };


      if (!char_found && !allow_num_only) goto loop_bottom;

      /* Skip spaces and tabs */
      while ( (*cur != '\0') &&
	      ((*cur == ' ') || (*cur == '\t')) ) cur++;

      /* Is there an integer part? */
      if (*cur == '\0')
		num_found = FALSE;
	   else
		{
		/* Make sure chars left are only numeric */
		i = cur;
		/* Check for numeric characters */
		while ((*i != '\0') && (*i >= '0') && (*i <= '9')) i++;
	        /* Skip trailing spaces and tabs */
	        while ( (*i != '\0') && ((*i == ' ') || (*i == '\t')) ) i++;
		if (*i != '\0') goto loop_bottom;
	        (void) sscanf(cur,"%d",num);	/* Convert to int */
	        if (check_range)
		    if ( (*num < lower) || (*num > upper) ) goto loop_bottom;
		num_found = TRUE;
		}

      /* Decide if valid response */
      if (char_found)
	   if (char_opts[(*leading_char) - 'A'].char_allowed)
		if (char_opts[(*leading_char) - 'A'].num)
			if (num_found)
				{
				status = MENUBAR_CHAR_AND_NUM;
				valid = TRUE;
				}
			    else
				goto loop_bottom;
		    else
			if (num_found)
				goto loop_bottom;
			    else
				{
				status = MENUBAR_CHAR_ONLY;
				*num = 0;
				valid = TRUE;
				}
	        else;
	else
	   if (allow_num_only)
		{
		status = MENUBAR_NUM_ONLY;
		*leading_char = '\0';
		valid = TRUE;
		}
	  
	
   loop_bottom:
      if (!valid)
         {
         (void) wmove(win,1,1);		       /* Cursor not on bottom line */
         (void) wrefresh(win);
         if (bell) DING;                                       /* Ring Bell */
         (void) wmove(win,line,col);
         (void) wclrtoeol(win);			     /* Wipe out bad answer */
         (void) wmove(win,line,col);
         box(win,'|','-');
         (void) wrefresh(win);
         }

      } while (!valid);

   return(status);
   }








public SYMBOL get_identifier(win,bell,allow_def,def,nt)
   WINDOW *win;
   char bell;
   char allow_def;
   SYMBOL def;
   char *nt;
/*
 * On Entry:
 *      win points at a valid curses window.
 *      bell is TRUE if bad entries should ring the bell.
 *	A valid symbol table must be active.
 *	allow_def is TRUE if the default symbol can be selected by
 *	   pressing Return.
 *	def contains the default SYMBOL iff allow_def is TRUE.
 *	   The calling routine must ensure that this SYMBOL is a
 *	   valid non-terminal identifier contained in table.
 *      The calling routine has just asked the user a question requiring
 *         an identifier name (possibly new) in response.
 *	nt is TRUE if the identifier should be forced to be a nonterminal,
 *	   FALSE otherwise.
 *
 * On Exit:
 *	if *nt was TRUE, then *nt returns TRUE indicating that the
 *	   identifier was forced to be a non-terminal.
 *	if *nt was FALSE, then *nt returns the current nonterminal/terminal
 *	   status in the symbol table.
 *      get_identifier returns a pointer to a SYMBOL
 *         for the user entered identifier.
*/
   {
   int line;
   int col;
   int ans_len;                                  /* Length of user response */
   int valid;                                   /* TRUE if response is good */
   int t = IDNT;                                    /* Used to call getsymb */
   SYMBOL sym;                                              /* Final result */


   getyx(win,line,col);
   do {
      GET_STRING(win,ans_str);
      ans_len = strlen(ans_str);

      if (!ans_len && allow_def)
            {
	    mvwaddstr(win,line + 1, START_COL,
	     "The default identifier already exists. Is that OK? (y or n)> ");
	    (void) wrefresh(win);
	    if (valid = get_boolean_response(win,bell))
	          return(def);			      /* Return the default */
	    }
	 else
            if (!(valid = is_idnt(ans_str)))
               { if(bell) DING; }                              /* Ring Bell */
             else
                if (idnt_exists(ans_str, ans_len))
                   {
                   mvwaddstr(win,line + 1,START_COL,
                    "That identifier already exists. Is that OK? (y or n)> ");
                   (void) wrefresh(win);
                   valid = get_boolean_response(win,bell);
                   }

      (void) wmove(win,line + 1,START_COL);              /* Clean up screen */
      (void) wclrtobot(win);
      (void) wmove(win,line,col);
      (void) wclrtoeol(win);
      (void) wmove(win,line,col);
      box(win,'|','-');
      (void) wrefresh(win);

      } while (!valid);

   getsymb(ans_str,ans_len,&t,&sym,nt);
   return(sym);
   }







public void output_centered(win,line,hl,text)
   WINDOW *win;
   int line;
   char hl;
   char *text;
/*
 * On Entry:
 *      win points at a valid curses window.
 *      line contains the line number on the screen.
 *      hl is TRUE if the text should be highlighted, FALSE otherwise.
 *      text points at a null terminated string to be displayed.
 *
 * On Exit:
 *      text has been output in the center of line. Any desired
 *         highlighting has been performed. No refresh
 *         of the screen is done.
*/
   {
   if (hl) (void) wstandout(win);
   mvwaddstr(win,line, (COLS/2 - strlen(text)/2 - 2), text);
   if (hl) (void) wstandend(win);
   }







public void message_and_pause(win,line,text)
   WINDOW *win;
   int line;
   char *text;
/*
 * On Entry:
 *      win points at a valid curses window.
 *      line contains the line number where the message should be placed.
 *      text points at a null terminated string to be displayed.
 *
 * On Exit:
 *	The message has been displayed on the given line. On the
 *	   following line, the message 'Press Return to Continue' has
 *	   been displayed, and the user has pressed Return. The screen
 *	   is not cleaned up prior to exit.
*/
   {
   mvwaddstr(win,line, (COLS/2 - strlen(text)/2 - 2), text);
   mvwaddstr(win,line+2,(COLS/2 - 14),"Press Return To Continue.");
   (void) wrefresh(win);
   (void) wgetch(win);
   }







/* --------------- G R A M M A R   O U T P U T------------------------------*/

public int c_prtsym(win, s)
   WINDOW *win;
   SYMBOL s;
/*
 * On Entry:
 *	win points at a valid curses window.
 *	s contains a valid SYMBOL
 *
 * On exit:
 *      s has been output to the curses window at *cur_col.
 *	c_prtsym returns the number of characters output.
*/
   {
   char c[BUFSIZ];

   (void)strncpy(c,string[s->str],s->l); c[s->l] = '\0';
   if (s->is_IDNT) (void) wprintw(win,"%s",c);
   else {
      char *p;
      wprintw(win,"'");
      for (p = c; *p; p++)
         { wprintw(win,"%c",*p); if (*p == '\'') wprintw(win,"'"); }
      wprintw(win,"'");
   }
   return(s->l + (s->is_IDNT ? 0 : 2));
   }







public void c_output_left_node(win,node)
   WINDOW *win;
   LEFT_NODE_PTR node;
/*
 * On Entry:
 *      win points at a valid curses window.
 *      node points at the left side of a rule in the EBNF internal structure
 * On Exit:
 *      If the left side is an IDNT:
 *         The identifier is output on win at the current cursor position
 *      Otherwise:
 *         An error is signalled and execution of CAGT is terminated.
*/
   {
   if (node->code != IDNT)
	cagt_msg(0, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_INVLDLFTSD)], "c_output_left_side"))
   (void) c_prtsym(win,node->text->symbol_ptr);
   }







public void c_output_right_node(win,node)
   WINDOW *win;
   RIGHT_NODE_PTR node;
/*
 * On Entry:
 *      win points at a valid curses window.
 *      node contains a pointer to a node on the right side of a
 *         rule in the EBNF internal structure.
 * On Exit:
 *      The node has been output to the curses window.
*/
   {
   if (node == (RIGHT_NODE_PTR) 0) return;

   switch (node->code) {
        case NULT : break;                                  /* Deleted node */
        case LITT :
        case IDNT:
                {
                (void) c_prtsym(win,node->x.text->symbol_ptr);
		break;
		}
	default :
                {
                (void) wprintw(win,"%s",token_symb[node->code]);
                break;
                };
        }

   }







public int c_output_right_side(win,node,space,hl)
   WINDOW *win;
   RIGHT_NODE_PTR node;
   char space;
   RIGHT_NODE_PTR hl;
/*
 * On Entry:
 *      win points at a valid curses window.
 *      node contains a pointer to a node on the right side of a
 *         rule in the EBNF internal structure.
 *      space is true if last output node needs a space following it
 *         (Used for formatting)
 *      if hl points at a IDNT or LITT node in the right side,
 *         that node will be output in curses highlight mode.
 * On Exit:
 *      The right side of the rule, starting from the node pointed
 *         at by node has been output to the curses window.
 *      c_output_right_side returns TRUE if the entire rule was output
 *         to the screen without any clipping on the bottom margin. Any
 *         part of the rule that would move below the bottom line
 *         of the screen is not output.
*/
   {
   int cur_line,
       cur_col;
   int not_clipped = TRUE;

   if (node == (RIGHT_NODE_PTR) 0) return(TRUE);

   getyx(win,cur_line,cur_col);

   switch (node->code) {
        case NULT :                                         /* Deleted node */
                {
                not_clipped = c_output_right_side(win,node->next,space,hl);
                break;
                }
        case DOTT :
                {
                (void) wprintw(win,"%s",token_symb[node->code]);
                break;
                };
        case  IST :
                {
                (void) wprintw(win," ");
                (void) wprintw(win,"%s",token_symb[node->code]);
                if (++cur_line >= (LINES - 3) ) return(FALSE);
                (void) wmove(win,cur_line,R_INDENT);
                not_clipped = c_output_right_side(win,node->next,FALSE,hl);
                break;
                }
        case GTRT : 
        case LSST :
                {
                if (space) (void) wprintw(win," ");
                (void) wprintw(win,"%s",token_symb[node->code]);
                not_clipped = c_output_right_side(win,node->next,TRUE,hl);
                break;
                }
        case AMPT : 
	case  ATT :
	case DOLT :
                {
                if ((cur_col + 2 + node->x.nest->x.text->symbol_ptr->l) >
                                                                  COLS)
                        {
                        if (++cur_line >= (LINES - 3) ) return(FALSE);
                        (void) wmove(win,cur_line,R_INDENT);
                        }                       
                   else
                        if (space) (void) wprintw(win," ");
                (void) wprintw(win,"%s",token_symb[node->code]);
                not_clipped = ( c_output_right_side(win,node->x.nest,FALSE,hl)
                              && c_output_right_side(win,node->next,TRUE,hl));
                break;
                }
        case ASTT : 
        case PLST : 
                {
                not_clipped = c_output_right_side(win,node->x.nest,space,hl);
                (void) wprintw(win,"%s",token_symb[node->code]);
                not_clipped = (not_clipped && c_output_right_side(win,
                                           node->next,TRUE,hl));
                break;
                }
        case BART : 
        case SEPT : 
                {
                not_clipped = c_output_right_side(win,node->x.infix->left,
                                                  space,hl);
                (void) wprintw(win," %s ",token_symb[node->code]);
                not_clipped = (not_clipped &&
                               c_output_right_side(win,node->x.infix->right,
                                                   FALSE,hl) &&
                               c_output_right_side(win,node->next,TRUE,hl) );
                break;
                }
        case LITT :
        case IDNT:
                {
                if ((cur_col + node->x.text->symbol_ptr->l + 5) > COLS)
                        {
                        if (++cur_line >= CLIP_THRESHOLD ) return(FALSE);
                        (void) wmove(win,cur_line,R_INDENT);
                        }                       
                   else
                        if (space) (void) wprintw(win," ");
                if (node == hl) (void) wstandout(win);
                (void) c_prtsym(win,node->x.text->symbol_ptr);
                if (node == hl) (void) wstandend(win);
                not_clipped = c_output_right_side(win,node->next,TRUE,hl);
                break;
                }
        case LPNT :
        case LBKT :
                {
                if (space) (void) wprintw(win," ");
                (void) wprintw(win,"%s",token_symb[node->code]);
                cur_col += token_symb_len[node->code];
                not_clipped = (
                              c_output_right_side(win,node->x.nest,FALSE,hl)&&
                              c_output_right_side(win,node->next,TRUE,hl) );
                break;
                }
        case RPNT:
        case RBKT:
                {
                not_clipped = TRUE;
                (void) wprintw(win,"%s",token_symb[node->code]);
                break;
                }
        }

   return(not_clipped);
   }







private c_output_grammar_listing_help(title)
char *title;
/*
 * c_output_grammar_listing_help outputs a help screen for use
 * by c_output_grammar_listing.
 *
 * entry:
 *	title - ^ to string to use as title at top of screen.
*/
   {
   WINDOW *win;
   char buf[128];

   /* Get a window to cover up the invocation window */
   win = newwin(0,0,0,0);
   strcpy(buf, "HELP: ");
   strncat(buf, title, 121);
   buf[127] = '\0';
   output_centered(win,1,TRUE, buf);
   mvwaddstr(win,2,1,LINE);
   (void) wrefresh(win);

   mvwaddstr(win, 5,10,"The Following Commands Are Availible:");
   mvwaddstr(win, 7,10,"Return - Causes the next page of rules to be displayed.");
   mvwaddstr(win, 8,10,"         If on the last page, returns to previous menu.");
   mvwaddstr(win, 9,10,"Rule # - Entering a rule number causes that rule to");
   mvwaddstr(win,10,10,"         be displayed at the top of the current page.");
   mvwaddstr(win,11,10,"D - Done: Returns to the previous menu.");
   mvwaddstr(win,12,10,"H - Help: Displays this menu.");

   box(win,'|','-');
   (void) wrefresh(win);
   message_and_pause(win,18,"");

   (void) wclear(win);
   (void) wrefresh(win);
   delwin(win);

   }







public void c_output_grammar_listing(start, title)
   LEFT_NODE_PTR start;
   char *title;
/*
 * c_output_grammar_listing outputs the internal EBNF grammar structure to
 * the screen in a style similar to the UNIX more utility, using curses.
 *
 * entry:
 *	start - ^ to start of grammar left hand side.
 *	title - ^ to string to use as title at top of screen.
 *
*/
   {
   WINDOW *win;			/* Curses window */
   int cur_line,
       cur_col;
   int top_rule;		/* Remembers rule above at top of screen */
   char not_clipped;		/* Tells when screen full */
   struct indexed_access_block access_blk;	/* Allows Indexed Access */
   LEFT_NODE_PTR left;		/* Left side of current output rule */
   char bottom[80];	/* Used to output message at bottom of screen */
   int rule_num = 0;		/* Current output rule index */
   int menubar_num;		/* Number typed at menubar */
   char menubar_char;		/* Character typed at menubar */
   int menubar_response = MENUBAR_NEVER;	/* To get main loop started */
   char quit = FALSE;



#ifdef lint                             /* Makes lint think cur_col is used */
   cur_col = 1;
   cur_line = cur_col++;
#endif



   /* Get a window to cover up the invocation window */
   win = newwin(0,0,0,0);
   output_centered(win,1,TRUE,title);
   mvwaddstr(win,2,1,LINE);

   init_indexed_access(start,&access_blk);

   
   while ( (rule_num < access_blk.rule_cnt) && !quit)
      {
      top_rule = rule_num;
      cur_line = 4;
      (void) wmove(win,cur_line,START_COL);
      (void) wclrtobot(win);
      not_clipped = TRUE;
      cur_col = START_COL;
      (void) wmove(win,cur_line,START_COL);
      while (not_clipped && (rule_num < access_blk.rule_cnt) &&
             (cur_line < CLIP_THRESHOLD) )
         {
         rule_num++;
         left = rule_by_index(&access_blk,rule_num);
         if (left->code != NULT)                      /* Not a deleted rule */
               {
               (void) wmove(win,cur_line,START_COL);
               (void) wprintw(win,"%s %d)  ",(left->chain_rule) ? "C" : " ",
					     rule_num);
               c_output_left_node(win,left);
               not_clipped = c_output_right_side(win,left->right_side,
                                          FALSE,((RIGHT_NODE_PTR) 0));
	       if (not_clipped)
		     {
                     (void) wprintw(win,"\n\n");
                     getyx(win,cur_line,cur_col);
		     }
		  else
		     {
		     rule_num--;
		     (void) wmove(win,cur_line,START_COL);
		     (void) wclrtobot(win);
		     }
               }
         }
      (void) sprintf(bottom,
           "[ %d/%d     Return for next page, Rule #, (D)one, (H)elp ] ",
           rule_num,access_blk.rule_cnt);
      output_centered(win, LINES -2, TRUE, bottom);
      box(win,'|','-');
      (void) wrefresh(win);
      menubar_response = get_menubar_response(win, TRUE, listing_char_opts,
					      TRUE, TRUE, TRUE, 1,
					      access_blk.rule_cnt,
					      &menubar_char, &menubar_num);
      if (menubar_response == MENUBAR_NUM_ONLY)
	      rule_num = menubar_num - 1;
	  else			   /* menubar_response == MENUBAR_CHAR_ONLY */
	      if (menubar_char == 'H')
		      {
		      c_output_grammar_listing_help(title);
		      rule_num = top_rule;
		      }
		  else
		      if (menubar_char == 'D') quit = TRUE;
      }

   delete_indexed_access(&access_blk);
   (void) wclear(win);
   (void) wrefresh(win);
   delwin(win);

   }







private void c_select_grammar_rules_help()
/*
 * c_select_grammar_rules_help outputs a help screen for use
 * by c_select_grammar_rules_help.
 *
*/
   {
   WINDOW *win;

   /* Get a window to cover up the invocation window */
   win = newwin(0,0,0,0);
   output_centered(win,1,TRUE,"HELP: Similar Rule Combination");
   mvwaddstr(win,2,1,LINE);
   (void) wrefresh(win);

   mvwaddstr(win, 5,10,"The Following Commands Are Availible:");
   mvwaddstr(win, 7,10,"Return - Causes the next page of rules to be displayed.");
   mvwaddstr(win, 8,10,"         If on the last page, returns to previous menu.");
   mvwaddstr(win, 9,10,"Rule # - Entering a rule number causes that rule to");
   mvwaddstr(win,10,10,"         be displayed at the top of the current page.");
   mvwaddstr(win,11,10,"A - All: Selects every rule.");
   mvwaddstr(win,12,10,"N - None: All selected rules are un-selected.");
   mvwaddstr(win,13,10,"S # - Select: Selects the specified rule.");
   mvwaddstr(win,14,10,"R # - Remove: Un-selects the the specified rule.");
   mvwaddstr(win,15,10,"D - Done: Returns to the previous menu.");
   mvwaddstr(win,16,10,"H - Help: Displays this menu.");

   box(win,'|','-');
   (void) wrefresh(win);
   message_and_pause(win,18,"");

   (void) wclear(win);
   (void) wrefresh(win);
   delwin(win);

   }







public void c_select_grammar_rules(start,queue,title)
   LEFT_NODE_PTR start;
   QUEUE_PTR queue;
   char *title;
/*
 * c_select_grammar_rules outputs the internal EBNF grammar structure to
 * the screen in a style similar to the UNIX more utility, allowing the
 * user to select rules from the grammar.
 *
 * On Entry:
 *    start points at the left hand chain of the grammar structure.
 *    queue represents a valid queue. If queue is not empty, it is assummed
 *	to contain pointers to rules in the grammar structure that should
 *	be initialized as selected. It is further assummed that any such
 *	rules are in order from top to bottom.
 *    title points at a string to be placed at the top of the screen.
 *
 * On Exit:
 *    Pointers to the selected rules have been placed in the queue. The
 *      rules are ordered according to their relative position in the
 *      grammar, starting from the top.
 *
*/
   {
   WINDOW *win;
   int cur_line,
       cur_col;
   int top_rule;		   /* Remembers rule above at top of screen */
   int not_clipped;
   struct indexed_access_block access_blk;         /* Allows Indexed Access */
   LEFT_NODE_PTR left;
   char bottom[80];           /* Used to output message at bottom of screen */
   int rule_num = 0;			       /* Current rule number index */
   char *selected;		           /* Keeps track of selected rules */
   int i;
   LEFT_NODE_PTR temp;			/* Used to weed out 'deleted' rules */
   char quit = FALSE;				  /* Indicates user is done */
   int menubar_num;				 /* Number typed at menubar */
   char menubar_char;			      /* Character typed at menubar */


#ifdef lint                             /* Makes lint think cur_col is used */
   cur_col = 1;
   cur_line = cur_col++;
#endif

   /* Get a window to cover up the invocation window */
   win = newwin(0,0,0,0);
   output_centered(win,1,TRUE,title);
   mvwaddstr(win,2,1,LINE);

   /* Get indexed access to the grammar */
   init_indexed_access(start,&access_blk);

   /* Get an array to keep track of selected rules */
   GET_MEMORY(selected, char *, access_blk.rule_cnt, char,
	      "c_select_grammar_rules", 1)
   for (i=0; i<access_blk.rule_cnt; i++) selected[i] = FALSE;

   /* Set initially selected rules */
   if (!empty(queue))
	{
	left = (LEFT_NODE_PTR) dequeue(queue);
	for (i=0; i<access_blk.rule_cnt; i++)
	    if (left == rule_by_index(&access_blk,i+1))
		{
		selected[i] = TRUE;
		if (!empty(queue))
			left = (LEFT_NODE_PTR) dequeue(queue);
		    else
			break;
		}
	}

   empty_queue(queue);			     /* This shouldn't be necessary */


   
   while ( (rule_num < access_blk.rule_cnt) && (!quit) )
      {
      top_rule = rule_num;
      cur_line = 4;
      (void) wmove(win,cur_line,START_COL);
      (void) wclrtobot(win);
      not_clipped = TRUE;
      cur_col = START_COL;
      (void) wmove(win,cur_line,START_COL);
      while (not_clipped && (rule_num < access_blk.rule_cnt) &&
             (cur_line < CLIP_THRESHOLD) )
         {
         rule_num++;
         left = rule_by_index(&access_blk,rule_num);
         if (left->code != NULT)                      /* Not a deleted rule */
               {
               (void) wmove(win,cur_line,START_COL);
	       if (selected[rule_num -1]) (void) wstandout(win);
               (void) wprintw(win,"%s %d)",(left->chain_rule) ? "C" : " ",
					   rule_num);
	       if (selected[rule_num -1]) (void) wstandend(win);
               (void) wprintw(win,"  ",rule_num);
               c_output_left_node(win,left);
               not_clipped = c_output_right_side(win,left->right_side,
                                          FALSE,((RIGHT_NODE_PTR) 0));
	       if (not_clipped)
		     {
                     (void) wprintw(win,"\n\n");
                     getyx(win,cur_line,cur_col);
		     }
		  else
		     {
		     rule_num--;
		     (void) wmove(win,cur_line,START_COL);
		     (void) wclrtobot(win);
		     }
               }
         }
      (void) sprintf(bottom,
 "[ %d/%d    Return,Rule #,(A)ll,(N)one,(S)el,(R)em,(D)one,(H)lp ] ",
	             rule_num,access_blk.rule_cnt);
      output_centered(win, LINES -2, TRUE, bottom);
      box(win,'|','-');
      (void) wrefresh(win);
      switch (get_menubar_response(win, TRUE, selection_char_opts, TRUE, TRUE,
					      TRUE, 1, access_blk.rule_cnt,
					      &menubar_char, &menubar_num))
	  {
	  case MENUBAR_CHAR_ONLY    :   {
					if (menubar_char == 'H')
					    {
					    c_select_grammar_rules_help();
					    break;
					    }
					if (menubar_char == 'D')
					    {
					    quit=TRUE;
					    break;
					    }
				        {	     /* It's (A)ll or (N)one */
					char b = (menubar_char == 'A') ?
								TRUE : FALSE;
					int t = 0;

					for (t=0; t < access_blk.rule_cnt; t++)
					    selected[t] = b;
					}
					rule_num = top_rule;
					break;
					}
	  case MENUBAR_NUM_ONLY     :
					{
					rule_num = menubar_num -1;
					break;
					}
	  case MENUBAR_CHAR_AND_NUM :
					{	     /* (S)elect or (R)emove */
					selected[menubar_num-1] =
					    (menubar_char=='S') ? TRUE : FALSE;
					rule_num = top_rule;
					break;
					}
	  }
      }


   /* Put selected rules in queue, and return selected array to O.S. */
   for (i=0; i < access_blk.rule_cnt; i++)
      if (selected[i])
	  {
	  temp = rule_by_index(&access_blk,i+1);
	  if (temp->code == IDNT) enqueue(queue,((void *) temp)); /*No NULTS*/
	  }
   FREE_MEMORY(selected, "c_select_grammar_rules", 1)



   /* Delete indexed access front end to grammar structure */
   delete_indexed_access(&access_blk);

   /* Get rid of curses window */
   (void) wclear(win);
   (void) wrefresh(win);
   delwin(win);

   }







private void c_select_identifiers_help()
/*
 * c_select_identifiers_help outputs a help screen for use
 * by c_select_identifiers.
 *
*/
   {
   WINDOW *win;

   /* Get a window to cover up the invocation window */
   win = newwin(0,0,0,0);
   output_centered(win,1,TRUE,"HELP: Identifier Selection");
   mvwaddstr(win,2,1,LINE);
   (void) wrefresh(win);

   mvwaddstr(win, 5,10,"The Following Commands Are Availible:");
   mvwaddstr(win, 7,10,"Return - Causes the next page of identifiers to be displayed.");
   mvwaddstr(win, 8,10,"         If on the last page, returns to previous menu.");
   mvwaddstr(win, 9,10,"Rule # - Entering an identifier number causes that identifier");
   mvwaddstr(win,10,10,"         to be displayed at the top of the current page.");
   mvwaddstr(win,11,10,"A - All: Selects every identifier.");
   mvwaddstr(win,12,10,"N - None: Un-selects every identifier.");
   mvwaddstr(win,13,10,"S # - Select: Selects the specified identifier. ");
   mvwaddstr(win,14,10,"R # - Remove: Un-selects the specified identifier.");
   mvwaddstr(win,15,10,"D - Done: Returns to the previous menu.");
   mvwaddstr(win,16,10,"H - Help: Displays this menu.");

   box(win,'|','-');
   (void) wrefresh(win);
   message_and_pause(win,18,"");

   (void) wclear(win);
   (void) wrefresh(win);
   delwin(win);

   }







public void c_select_identifiers(queue)
   QUEUE_PTR queue;
/*
 * c_select_identifiers outputs the identifiers in the current
 * symbol table to the screen, allowing the user to select identifiers
 * from the grammar.
 *
 * On Entry:
 *    A valid symbol table is current.
 *    queue represents a valid queue.
 *
 * On Exit:
 *	   Symbols for the selected identifiers have been entered
 *	into the queue.
*/
   {
   WINDOW *win;
   int cur_line,
       cur_col;
   int first_id;		   /* Remembers identifier at top of screen */
   char bottom[80];           /* Used to output message at bottom of screen */
   int cur_id = 0;
   struct id_elt {
	char select;
	SYMBOL sym;
	};
   struct id_elt *id_mem;	     /* Keeps track of selected identifiers */
   int id_num;
   int id_num_len;
   int i;
   QUEUE_PTR sym_queue;
   char quit = FALSE;				  /* Indicates user is done */
   int menubar_num;				 /* Number typed at menubar */
   char menubar_char;			      /* Character typed at menubar */


#ifdef lint                             /* Makes lint think cur_col is used */
   cur_col = 1;
   cur_line = cur_col++;
#endif

   /* Get a window to cover up the invocation window */
   win = newwin(0,0,0,0);
   output_centered(win,1,TRUE,"Identifier Selection");
   mvwaddstr(win,2,1,LINE);

   /* Get a queue of availible identifiers */
   sym_queue = get_symbol_queue(FALSE, TRUE, TRUE, TRUE, FALSE);

   /* Get an array to keep track of id_mem identifiers */
   id_num = count_queue(sym_queue);
   GET_MEMORY(id_mem, struct id_elt *, id_num, struct id_elt,
	      "c_select_identifiers", 1)
   for (i=0; i<id_num; i++)
	{
	id_mem[i].sym = (SYMBOL) dequeue(sym_queue);
	id_mem[i].select = FALSE;
	}
   delete_queue(sym_queue);


   cur_id = 0;
   while ( (cur_id < id_num) && (!quit) )
      {
      first_id = cur_id;
      cur_col = 3;
      cur_line = 3;
      (void) wmove(win,cur_line,cur_col);
      (void) wclrtobot(win);
      cur_line = 4;
      (void) wmove(win,cur_line,cur_col);
      while ( (cur_id < id_num) && (cur_line < CLIP_THRESHOLD) )
	{
	id_num_len=((int) log10((double) (cur_id+1)))+3;
	if ((cur_col + id_mem[cur_id].sym->l + id_num_len + 2) > COLS)
		{
		cur_line++;
		cur_col = 3;
		(void) wmove(win,cur_line,cur_col);
		}
	if (id_mem[cur_id].select) (void) wstandout(win);
	(void) wprintw(win,"(%d)",(cur_id+1));
	cur_col += c_prtsym(win,id_mem[cur_id].sym) + id_num_len + 5;
	if (id_mem[cur_id].select) (void) wstandend(win);
	(void) wmove(win,cur_line,cur_col);
        cur_id++;
	}
     (void) sprintf(bottom,
	 "[ %d/%d    Return,Rule #,(A)ll,(N)one,(S)el,(R)em,(D)one,(H)lp ] ",
		    cur_id,id_num);
      output_centered(win, LINES -2, TRUE, bottom);
      box(win,'|','-');
      (void) wrefresh(win);
      switch (get_menubar_response(win, TRUE, selection_char_opts, TRUE, TRUE,
					      TRUE, 1, id_num,
					      &menubar_char, &menubar_num))
	  {
	  case MENUBAR_CHAR_ONLY    :   {
					if (menubar_char == 'H')
					    {
					    c_select_identifiers_help();
					    cur_id = first_id;
					    break;
					    }
					if (menubar_char == 'D')
					    {
					    quit=TRUE;
					    break;
					    }
				        {	/* It's (A)ll or (N)one */
					char b = (menubar_char == 'A') ?
								TRUE : FALSE;
					int t = 0;

					for (t=0; t < id_num; t++)
					    id_mem[t].select = b;
					}
					cur_id = first_id;
					break;
					}
	  case MENUBAR_NUM_ONLY     :
					{
					cur_id = menubar_num -1;
					break;
					}
	  case MENUBAR_CHAR_AND_NUM :
					{	/* (S)elect or (R)emove */
					id_mem[menubar_num-1].select =
					    (menubar_char=='S')? TRUE : FALSE;
					cur_id = first_id;
					break;
					}
	  }
      }


   /* Put id_mem SYMBOLS in queue, and return id_mem array to O.S. */
   for (i=0; i < id_num; i++)
      if (id_mem[i].select) enqueue(queue,((void *) id_mem[i].sym));
   FREE_MEMORY(id_mem, "c_select_identifiers", 1)



   /* Get rid of curses window */
   (void) wclear(win);
   (void) wrefresh(win);
   delwin(win);

   }







public void c_output_rule_pair(win,rule1,rule2,diff1,diff2)
   WINDOW *win;
   LEFT_NODE_PTR rule1,
                 rule2;
   RIGHT_NODE_PTR diff1,
                  diff2;
/*
 * On Entry:
 *      win points at a valid curses window.
 *      rule1 and rule2 point at a pair of non-NULT rules in an
 *         existing EBNF grammar structure.
 *      diff1 and diff2 point at a pair of nodes in the two rules that
 *         should be output in highlight mode. If this is not desired,
 *         they should be set to NULL.
 *
 * On Exit:
 *      The two rules have been output to win starting at the current
 *         cursor.
*/
   {
   int cur_line;
   int cur_col;


#ifdef lint                             /* Makes lint think cur_col is used */
   cur_col = 1;
   cur_line = cur_col++;
#endif

   getyx(win,cur_line,cur_col);
   (void) wmove(win,cur_line,START_COL);
   (void) wclrtobot(win);
   (void) wmove(win,cur_line,START_COL);
   c_output_left_node(win,rule1);
   (void) c_output_right_side(win,rule1->right_side,FALSE,diff1);
   getyx(win,cur_line,cur_col);
   (void) wmove(win,(cur_line + 2),START_COL);
   c_output_left_node(win,rule2);
   (void) c_output_right_side(win,rule2->right_side,FALSE,diff2);
   }
