/*
 * $Id: stream.c,v 2.0 1992/09/23 08:45:04 toh-hei Exp $
 *
 * Copyright (c) 1992 Kimura Laboratory, Department of Information Science,
 * Tokyo Institute of Technology.  All Rights Reserved.
 *
 */

static char rcsid[] = "$Id: stream.c,v 2.0 1992/09/23 08:45:04 toh-hei Exp $";

#include <clu2c.h>
#include <type.h>
#include <glo.h>

#include <errno.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <strings.h>

#define MAXLINE 4096

extern char *sys_errlist[];

/* internal functions */

static void put_prompt(string prompt)
{
    printf("%s", prompt);	/* tentative version */
}

static char *tty_name(int filedes)
{
    char *src_name, *dst_name;
    
    src_name = (char *) ttyname(filedes);
    /*
     *	Because an object being allocated doesn't contain any pointer,
     *	malloc_atomic is used rather than malloc, expecting better efficiency.
     */
    dst_name = (char *) malloc_atomic((strlen(src_name) + 1) * sizeof(char));
    strcpy(dst_name, src_name);
    return dst_name;
}

static struct buf *buf_make_read(FILE *fp)
{
    struct buf *b;

    b = (struct buf *) malloc(sizeof(struct buf));
    b->tag = BUF_READ;
    b->val.read = (struct rbuf *) malloc(sizeof(struct rbuf));
    b->val.read->fp = fp;
    b->val.read->line = 1;
    return b;
}

static struct buf *buf_make_write(FILE *fp)
{
    struct buf *b;

    b = (struct buf *) malloc(sizeof(struct buf));
    b->tag = BUF_WRITE;
    b->val.write = (struct wbuf *) malloc(sizeof(struct wbuf));
    b->val.write->fp = fp;
    b->val.write->buffered = TRUE;
    return b;
}

static struct buf *buf_make_tty(FILE *fp)
{
    struct buf *b;

    b = (struct buf *) malloc(sizeof(struct buf));
    b->tag = BUF_TTY;
    b->val.tty = (struct tbuf *) malloc(sizeof(struct tbuf));
    b->val.tty->fp = fp;
    b->val.tty->ibuffered = TRUE;
    b->val.tty->obuffered = TRUE;
    b->val.tty->line = 1;
    b->val.tty->prompt = "";
    b->val.tty->want_prompt = TRUE;
    return b;
}

static struct buf *buf_make_istr(string chars)
{
    struct buf *b;

    b = (struct buf *) malloc(sizeof(struct buf));
    b->tag = BUF_ISTR;
    b->val.istr = (struct sbuf *) malloc(sizeof(struct sbuf));
    b->val.istr->chars = chars;
    b->val.istr->index = 1;
    b->val.istr->line = 1;
    return b;
}

static struct buf *buf_make_ostr()
{
    struct buf *b;

    b = (struct buf *) malloc(sizeof(struct buf));
    b->tag = BUF_OSTR;
    array_new();
    b->val.ostr = (array) retval_area[0];
    return b;
}

static struct buf *buf_make_closed()
{
    struct buf *b;

    b = (struct buf *) malloc(sizeof(struct buf));
    b->tag = BUF_CLOSED;
    b->val.closed = NIL;		/* dummy (no meaning) */
    return b;
}
    
static stream cons_rep(file_name fn, struct buf *b)
{
    stream res;

    res = (stream) malloc(sizeof(struct stream_rep));
    res->name = fn;
    sequence_new();
    res->scripts = (sequence) retval_area[0];
    res->scripting = FALSE;
    res->buf = b;
    return res;
}

/* end of internal functions */

/*
 * open = proc(f: file_name, access: string) returns(stream)
 *					     signals(not_possible(string))
 */

int stream_open(file_name file, string clu_mode)
{
    string fname, c_mode;
    FILE *fp;
    bool can_read, can_write;
    struct buf *b;
    
    file_name_unparse(file); /* signal handlling is not implemented. */
    fname = (string) retval_area[0];
    
    can_read = can_write = FALSE;
    if ( strcmp(clu_mode, "read") == 0 ) {
	c_mode = "r";
	can_read = TRUE;
    } else if ( strcmp(clu_mode, "write" ) == 0) {
	c_mode = "w";
	can_write = TRUE;
    } else if ( strcmp(clu_mode, "append" ) == 0) {
	c_mode = "a";
	can_write = TRUE;
    } else {
	SIGNAL1("not_possible", "bad access mode");
    }
    
    if ( (fp = fopen(fname, c_mode)) == NULL ) {
	SIGNAL1("not_possible", sys_errlist[errno]);
    }
    
    if ( isatty(fileno(fp)) ) {				/* terminal */
	b = buf_make_tty(fp);
    } else if ( can_read ) {				/* read stream */
	b = buf_make_read(fp);
    } else if ( can_write ) {				/* write stream */
	b = buf_make_write(fp);
    }
    RETURN1(cons_rep(file, b));
}

/*
 * primary_input = proc() returns(stream)
 */

int stream_primary_input()
{
    file_name fn;
    struct buf *b;

    if ( isatty(fileno(stdin)) ) {
	/* terminal */
	file_name_parse((string) tty_name(fileno(stdin)));
	fn = (file_name) retval_area[0];
	b = buf_make_tty(stdin);
    } else {
	/* read stream */
	fn = NULL;
	b = buf_make_read(stdin);
    }
    RETURN1(cons_rep(fn, b));
}

/*
 * primary_output = proc() returns(stream)
 */

int stream_primary_output()
{
    file_name fn;
    struct buf *b;

    if ( isatty(fileno(stdout)) ) {
	/* terminal */
	file_name_parse((string) tty_name(fileno(stdout)));
	fn = (file_name) retval_area[0];
	b = buf_make_tty(stdout);
    } else {
	/* write stream */
	fn = NULL;
	b = buf_make_write(stdout);
    }
    RETURN1(cons_rep(fn, b));
}

/*
 * error_output = proc() returns(stream)
 */

int stream_error_output()
{
    file_name fn;
    struct buf *b;

    if ( isatty(fileno(stderr)) ) {
	/* terminal */
	file_name_parse((string) tty_name(fileno(stderr)));
	fn = (file_name) retval_area[0];
	b = buf_make_tty(stderr);
    } else {
	/* write stream */
	fn = NULL;
	b = buf_make_write(stderr);
    }
    RETURN1(cons_rep(fn, b));
}

/*
 * can_read = proc(st: stream) returns(bool)
 */

int stream_can_read(stream s)
{
    switch ( s->buf->tag ) {
    case BUF_WRITE:
    case BUF_OSTR:
    case BUF_CLOSED:
	RETURN1(FALSE);
    default:
	RETURN1(TRUE);
    }
}

/*
 * can_write = proc(st: stream) returns(bool)
 */
  
int stream_can_write(stream s)
{
    switch ( s->buf->tag ) {
    case BUF_READ:
    case BUF_ISTR:
    case BUF_CLOSED:
	RETURN1(FALSE);
    default:
	RETURN1(TRUE);
    }
}

/*
 * getc = proc(st: stream) returns(char)
 *			   signals(end_of_file, not_possible(string))
 */

int stream_getc(stream s)
{
    int c;

    switch ( s->buf->tag ) {

    case BUF_READ: {
	struct rbuf *rb;
	FILE *fp;

	rb = s->buf->val.read;
	fp = rb->fp;
	if ( feof(fp) ) {
	    SIGNAL0("end_of_file");
	}
	clearerr(fp);
	if ( (c = getc(fp)) == EOF ) {
	    /* "end of file" or "error" */
	    if ( ferror(fp) ) {
		SIGNAL1("not_possible", sys_errlist[errno]);
	    } else {
		SIGNAL0("end_of_file");
	    }
	}
	if ( c == '\n' ) {
	    rb->line++;
	}
    	break;
    }

    case BUF_TTY: {
	struct tbuf *tb;
	FILE *fp;

	tb = s->buf->val.tty;
	fp = tb->fp;
	if ( feof(fp) ) {
	    SIGNAL0("end_of_file");
	}
	if ( tb->want_prompt ) {
	    string_equal(tb->prompt, "");
	    if ( retval_area[0] == FALSE ) {
		put_prompt(tb->prompt);
	    }
	}
	clearerr(fp);
	if ( (c = getc(fp)) == EOF ) {
	    /* "end of file" or "error" */
	    if ( ferror(fp) ) {
		SIGNAL1("not_possible", sys_errlist[errno]);
	    } else {
		SIGNAL0("end_of_file");
	    }
	}
	if ( c == '\n' ) {
	    tb->line++;
	}
	tb->want_prompt = (fp->_cnt == 0);
	break;
    }

    case BUF_ISTR: {
	struct sbuf *sb;

	sb = s->buf->val.istr;
	if ( string_fetch(sb->chars, sb->index) == SIG ) {
	    /* signame is assumed to be "bounds" */
	    SIGNAL0("end_of_file");
	} else {
	    c = (int) retval_area[0];
	    sb->index++;
	}
	if ( c == '\n' ) {
	    sb->line++;
	}
	break;
    }

    default:
	SIGNAL1("not_possible", "cannot read from this stream");
    }
    if ( s->scripting ) {
	/*$B!Z(Bscripting $B$N=hM}![(B*/
    }
    RETURN1(c);
}

/*
 * peekc = proc(st: stream) returns(char)
 *			    signals(end_of_file, not_possible(string))
 */

int stream_peekc(stream s)
{
    int c;

    switch ( s->buf->tag ) {

    case BUF_READ: {
	struct rbuf *rb;
	FILE *fp;

	rb = s->buf->val.read;
	fp = rb->fp;
	if ( feof(fp) ) {
	    SIGNAL0("end_of_file");
	}
	clearerr(fp);
	if ( (c = getc(fp)) == EOF ) {
	    /* "end of file" or "error" */
	    if ( ferror(fp) ) {
		SIGNAL1("not_possible", sys_errlist[errno]);
	    } else {
		SIGNAL0("end_of_file");
	    }
	}
	ungetc(c, fp);
	break;
    }

    case BUF_TTY: {
	struct tbuf *tb;
	FILE *fp;

	tb = s->buf->val.tty;
	fp = tb->fp;
	if ( feof(fp) ) {
	    SIGNAL0("end_of_file");
	}
	tb->want_prompt = (fp->_cnt == 0);
	clearerr(fp);
	if ( (c = getc(fp)) == EOF ) {
	    /* "end of file" or "error" */
	    if ( ferror(fp) ) {
		SIGNAL1("not_possible", sys_errlist[errno]);
	    } else {
		SIGNAL0("end_of_file");
	    }
	}
	ungetc(c, fp);
	break;
    }

    case BUF_ISTR: {
	struct sbuf *sb;

	sb = s->buf->val.istr;
	if ( string_fetch(sb->chars, sb->index) == SIG ) {
	    /* signame is assumed to be "bounds" */
	    SIGNAL0("end_of_file");
	} else {
	    c = (int) retval_area[0];
	}
	break;
    }

    default:
	SIGNAL1("not_possible", "cannot read from this stream");
    }
    RETURN1(c);
}

/*
 * empty = proc(st: stream) returns(bool) signals(not_possible(string))
 */

int stream_empty(stream s)
{
    bool is_empty;

    switch ( s->buf->tag ) {

    case BUF_READ: {
	FILE *fp;

	fp = s->buf->val.read->fp;
	if ( feof(fp) ) {
	    is_empty = TRUE;
	} else {
	    int c;
	    clearerr(fp);
	    if ( (c = getc(fp)) == EOF ) {
		/* "end of file" or "error" */
		if ( ferror(fp) ) {
		    SIGNAL1("not_possible", sys_errlist[errno]);
		} else {
		    is_empty = TRUE;
		}
	    } else {
		ungetc(c, fp);
		is_empty = FALSE;
	    }
	}
	break;
    }

    case BUF_TTY: {
	struct tbuf *tb;
	FILE *fp;

	tb = s->buf->val.tty;
	fp = tb->fp;
	if ( feof(fp) ) {
	    is_empty = TRUE;
	} else {
	    int c;
	    tb->want_prompt = (fp->_cnt == 0);
	    clearerr(fp);
	    if ( (c = getc(fp)) == EOF ) {
		/* "end of file" or "error" */
		if ( ferror(fp) ) {
		    SIGNAL1("not_possible", sys_errlist[errno]);
		} else {
		    is_empty = TRUE;
		}
	    } else {
		ungetc(c, fp);
		is_empty = FALSE;
	    }
	}
	break;
    }

    case BUF_ISTR: {
	struct sbuf *sb;

	sb = s->buf->val.istr;
	if ( string_fetch(sb->chars, sb->index) == SIG ) {
	    /* signame is assumed to be "bounds" */
	    is_empty = TRUE;
	} else {
	    is_empty = FALSE;
	}
	break;
    }

    default:
	SIGNAL1("not_possible", "cannot read from this stream");
    }
    RETURN1(is_empty);
}

/*
 * putc = proc(st: stream, c: char) signals(not_possible(string))
 */

int stream_putc(stream s, char c)
{
    switch ( s->buf->tag ) {

    case BUF_WRITE: {
	struct wbuf *wb;
	FILE *fp;

	wb = s->buf->val.write;
	fp = wb->fp;
	if ( wb->buffered ) {
	    /* $B!Z%P%C%U%!%j%s%0$"$j$N>l9g$N=hM}![(B */
	    if ( putc(c, fp) == EOF ) {
		SIGNAL1("not_possible", sys_errlist[errno]);
	    }
	} else {
	    /* $B!Z%P%C%U%!%j%s%0$J$7$N>l9g$N=hM}![(B */
	    if ( putc(c, fp) == EOF ) {
		SIGNAL1("not_possible", sys_errlist[errno]);
	    }
	}
	break;
    }

    case BUF_TTY: {
	struct tbuf *tb;
	FILE *fp;

	tb = s->buf->val.tty;
	fp = tb->fp;
	if ( tb->obuffered ) {
	    /* $B!Z%P%C%U%!%j%s%0$"$j$N>l9g$N=hM}![(B */
	    if ( putc(c, fp) == EOF ) {
		SIGNAL1("not_possible", sys_errlist[errno]);
	    }
	} else {
	    /* $B!Z%P%C%U%!%j%s%0$J$7$N>l9g$N=hM}![(B */
	    if ( putc(c, fp) == EOF ) {
		SIGNAL1("not_possible", sys_errlist[errno]);
	    }
	}
	break;
    }
	
    case BUF_OSTR:
	array_addh(s->buf->val.ostr, c);
	break;
	
    default:
	SIGNAL1("not_possible", "cannot write to this stream");
    }
    RETURN0;
}

/*
 * putc_image = proc(st: stream, c: char) signals(not_possible(string))
 */

/* stub version */

int stream_putc_image(stream s, char c)
{
    SIGNAL1("failure", "stream$putc_image: not implemented");
}

/*
 * getc_image = proc(st: stream) returns(char)
 *				 signals(end_of_file, not_possible(string))
 */

/* stub version */

int stream_getc_image(stream s)
{
    SIGNAL1("failure", "stream$getc_image: not implemented");
}

/*
 * get_lineno = proc(st: stream) returns(int)
 *				 signals(not_possible(string))
 */

int stream_get_lineno(stream s)
{
    int res;

    switch ( s->buf->tag ) {
    case BUF_READ:
	res = s->buf->val.read->line;
	break;
    case BUF_TTY:
	res = s->buf->val.tty->line;
	break;
    case BUF_ISTR:
	res = s->buf->val.istr->line;
	break;
    default:
	SIGNAL1("not_possible", "no line numbers");
    }
    RETURN1(res);
}

/*
 * set_lineno = proc(st: stream, ln: int) signals(not_possible(string))
 */

int stream_set_lineno(stream s, int i)
{
    RETURN0;
}

/*
 * reset = proc(st: stream) signals(not_possible(string))
 */

int stream_reset(stream s)
{
    switch ( s->buf->tag ) {

    case BUF_READ: {
	FILE *fp;

	fp = s->buf->val.read->fp;
	if ( rewind(fp) == -1 ) {
	    SIGNAL1("not_possible", sys_errlist[errno]);
	}
	clearerr(fp);
	s->buf->val.read->line = 1;
	break;
    }

    case BUF_WRITE: {
	FILE *fp;

	fp = s->buf->val.write->fp;
	if ( rewind(fp) == -1 ) {
	    SIGNAL1("not_possible", sys_errlist[errno]);
	}
	clearerr(fp);
	break;
    }

    case BUF_TTY: {
	FILE *fp;

	fp = s->buf->val.tty->fp;
	if ( rewind(fp) == -1 ) {
	    SIGNAL1("not_possible", sys_errlist[errno]);
	}
	clearerr(fp);
	s->buf->val.tty->line = 1;
	break;
    }

    case BUF_ISTR: {
	struct sbuf *sb;

	sb = s->buf->val.istr;
	sb->index = 1;
	sb->line = 1;
	break;
    }

    case BUF_OSTR:
	array_trim(s->buf->val.ostr, 1, 0);
	break;

    default:
	break;
    }
    RETURN0;
}

/*
 * flush = proc(st: stream) signals(not_possible(string))
 */

int stream_flush(stream s)
{
    switch ( s->buf->tag ) {

    case BUF_WRITE: {
	FILE *fp;

	fp = s->buf->val.write->fp;
	if ( fflush(fp) == EOF ) {
	    SIGNAL1("not_possible", sys_errlist[errno]);
	}
	break;
    }

    case BUF_TTY: {
	FILE *fp;

	fp = s->buf->val.tty->fp;
	if ( fflush(fp) == EOF ) {
	    SIGNAL1("not_possible", sys_errlist[errno]);
	}
	break;
    }

    case BUF_CLOSED:
	RETURN0;

    default:
	break;
    }
    if ( s->scripting ) {
	/* $B!Z(Bscripting $B$K4X$9$k=hM}![(B */
    }
    RETURN0;
}

/*
 * get_line_length = proc(st: stream) returns(int) signals(no_limit)
 */

/* tentative version */

int stream_get_line_length(stream s)
{
    int res;

    switch ( s->buf->tag ) {
    case BUF_TTY:
	res = 80;		/* tentative result */
	break;
    default:
	SIGNAL0("no_limit");
    }
    RETURN1(res);
}

/*
 * get_page_length = proc(st: stream) returns(int) signals(no_limit)
 */

/* tentative version */

int stream_get_page_length(stream s)
{
    int res;

    switch ( s->buf->tag ) {
    case BUF_TTY:
	res = 24;		/* tentative result */
	break;
    default:
	SIGNAL0("no_limit");
    }
    RETURN1(res);
}

/*
 * get_date = proc(st: stream) returns(date) signals(not_possible(string))
 */

int stream_get_date(stream s)
{
    struct stat status;
    struct tm *time;
    FILE *fp;
    
    switch ( s->buf->tag ) {
    case BUF_READ:
	fp = s->buf->val.read->fp;
	break;
    case BUF_WRITE:
	fp = s->buf->val.write->fp;
	break;
    default:
	SIGNAL1("not_possible", "cannot get date");
    }
    
    if ( fstat(fileno(fp), &status) != 0 ) {
	SIGNAL1("not_possible", sys_errlist[errno]);
    }
  
    time = localtime(&(status.st_mtime));

    if ( _cdate_create(time->tm_mday, time->tm_mon + 1, time->tm_year + 1900,
		       time->tm_hour, time->tm_min, time->tm_sec) == SIG )
    {
	SIGNAL1("failure", "stream$get_date: date$create failed");
    }

    /* retval_area[0] is return value of date$create */

    return RET;
}

/*
 * set_date = proc(st: stream, new_date: date) signals(not_possible(string))
 */

int stream_set_date(stream s, clus d)
{
    SIGNAL1("not_possible", "cannot set date");
}

/*
 * get_name = proc(st: stream) returns(file_name) signals(not_possible(string))
 */

int stream_get_name(stream s)
{
    switch ( s->buf->tag ) {
    case BUF_READ:
    case BUF_WRITE:
    case BUF_CLOSED:
	if ( s->name == NULL ) {
	    SIGNAL1("not_possible", "no name for this stream");
	}
	break;
    case BUF_ISTR:
    case BUF_OSTR:
	if ( s->name == NULL ) {
	    SIGNAL1("not_possible", "no name for string stream");
	}	
	break;
    }

    RETURN1(s->name);
}

/*
 * close = proc(st: stream) signals(not_possible(string))
 */

int stream_close(stream s)
{
    switch ( s->buf->tag ) {
    case BUF_READ:
	if ( fclose(s->buf->val.read->fp) == EOF ) {
	    SIGNAL1("not_possible", sys_errlist[errno]);
	}
	break;
    case BUF_WRITE:
	if ( fclose(s->buf->val.write->fp) == EOF ) {
	    SIGNAL1("not_possible", sys_errlist[errno]);
	}
	break;
    case BUF_TTY:
	if ( fclose(s->buf->val.tty->fp) == EOF ) {
	    SIGNAL1("not_possible", sys_errlist[errno]);
	}
	break;
    case BUF_ISTR:
    case BUF_OSTR:
	break;
    case BUF_CLOSED:
	RETURN0;
    }
    s->buf = buf_make_closed();
    sequence_new();
    s->scripts = (sequence) retval_area[0];
    s->scripting = FALSE;
    RETURN0;
}

/*
 * abort = proc(st: stream)
 */

int stream_abort(stream s)
{
    switch ( s->buf->tag ) {
    case BUF_READ:
	fclose(s->buf->val.read->fp);
	break;
    case BUF_WRITE:
	fclose(s->buf->val.write->fp);
	break;
    case BUF_TTY:
	fclose(s->buf->val.tty->fp);
	break;
    case BUF_ISTR:
    case BUF_OSTR:
	break;
    case BUF_CLOSED:
	RETURN0;
    }
    s->buf = buf_make_closed();
    sequence_new();
    s->scripts = (sequence) retval_area[0];
    s->scripting = FALSE;
    RETURN0;
}

/*
 * is_closed = proc(st: stream) returns(bool)
 */

int stream_is_closed(stream s)
{
    RETURN1(s->buf->tag == BUF_CLOSED);
}

/*
 * is_terminal = proc(st: stream) returns(bool)
 */

int stream_is_terminal(stream s)
{
    RETURN1(s->buf->tag == BUF_TTY);
}

/*
 * getl = proc(st: stream) returns(string)
 *			   signals(end_of_file, not_possible(string))
 */

int stream_getl(stream s)
{
    int i, c;
    char line[MAXLINE];
    string res;
    FILE *fp;
    int *lineno;
    
    switch ( s->buf->tag ) {

    case BUF_READ:
	fp = s->buf->val.read->fp;
	lineno = &(s->buf->val.read->line);
	break;

    case BUF_TTY: {
	struct tbuf *tb;
	tb = s->buf->val.tty;
	fp = tb->fp;
	lineno = &(s->buf->val.tty->line);	
	if ( tb->want_prompt ) {
	    string_equal(tb->prompt, "");
	    if ( retval_area[0] == FALSE ) {
		put_prompt(tb->prompt);
	    }
	}
	break;
    }

    case BUF_ISTR: {
	struct sbuf *sb;
	string chars;
	int first, last;
	bool exhausted = FALSE;
	
	stream_empty(s);
	if ( (bool) retval_area[0] ) {
	    SIGNAL0("end_of_file");
	}
	sb = s->buf->val.istr;
	chars = sb->chars;
	first = last = sb->index;
	while ( TRUE ) {
	    if ( string_fetch(chars, last) == SIG ) {
		/* exception name is assumed to be "bounds" */
		exhausted = TRUE;
		break;
	    }
	    if ( (char) retval_area[0] == '\n' ) {
		break;
	    }
	    last++;
	}
	if ( exhausted ) {
	    sb->index = last;
	} else {
	    sb->index = last + 1;
	}
	sb->line++;
	string_substr(chars, first, last - first);
	return RET;
    }

    default:
	SIGNAL1("not_possible", "cannot read from this stream");
    }
	
    /* Hereafter fp will be read */
    
    i = 0;
    clearerr(fp);
    while ( (c = getc(fp)) != '\n' ) {
	if ( c == EOF ) {
	    if ( ferror(fp) ) {
		SIGNAL1("not_possible", sys_errlist[errno]);
	    } else if ( i == 0 ) {	/* end of file */
		SIGNAL0("end_of_file");
	    } else {	/* end of file, but at least one character read */
		break;
	    }
	}
	if ( i >= MAXLINE - 1 ) {
	    SIGNAL1("not_possible", "too long line");
	}
	line[i++] = c;
    }
    (*lineno)++;
    line[i] = '\0';
    res = (string) malloc_atomic((i + 2) * sizeof(char));
    strcpy(res, line);
    RETURN1(res);
}

/*
 * putl = proc(st: stream, line: string) signals(not_possible(string))
 */

int stream_putl(stream s, string line)
{
    FILE *fp;
    int i, sz;

    string_size(line);
    sz = (int) retval_area[0];

    switch ( s->buf->tag ) {

    case BUF_WRITE:
	fp = s->buf->val.write->fp;
	break;

    case BUF_TTY:
	fp = s->buf->val.tty->fp;
	break;

    case BUF_OSTR: {
	array chars;

	chars = s->buf->val.ostr;
	for ( i = 0; i < sz; i++ ) {
	    array_addh(chars, line[i]);
	}
	array_addh(chars, '\n');
	RETURN0;
    }

    default:
	SIGNAL1("not_possible", "cannot write to this stream");
    }	

    /* Hereafter fp is written */

    for ( i = 0; i < sz; i++ ) {
	if ( putc(line[i], fp) == EOF ) {
	    SIGNAL1("not_possible", sys_errlist[errno]);
	}
    }
    if ( putc('\n', fp) == EOF ) {
	SIGNAL1("not_possible", sys_errlist[errno]);
    }
    RETURN0;
}

/*
 * gets = proc(st: stream, term: string) returns(string)
 *				signals(end_of_file, not_possible(string))
 */

int stream_gets(stream s, string term)
{
    char line[MAXLINE];
    string res;
    
    switch ( s->buf->tag ) {
	
    case BUF_READ: {
	struct rbuf *rb;
	FILE *fp;
	char c;
	int i;

	rb= s->buf->val.read;
	fp = rb->fp;
	i = 0;
	while ( TRUE ) {
	    if ( (c = getc(fp)) == EOF ) {
		if ( ferror(fp) ) {
		    SIGNAL1("not_possible", sys_errlist[errno]);
		} else {		/* end of file */
		    break;
		}
	    }
	    string_indexc(c, term);
	    if ( retval_area[0] > 0 ) {
		break;
	    }
	    if ( i >= MAXLINE - 1 ) {
		SIGNAL1("not_possible", "too long line");
	    }
	    line[i++] = c;
	    if ( c == '\n' ) {
		rb->line++;
	    }
	}
	if ( c != EOF ) {
	    ungetc(c, fp);
	}
	line[i] = '\0';
	res = (string) malloc_atomic(i + 2);
	strcpy(res, line);
	break;
    }
	
    case BUF_TTY: {
	struct tbuf *tb;
	FILE *fp;
	char c;
	int i;

	tb= s->buf->val.tty;
	fp = tb->fp;
	i = 0;
	while ( TRUE ) {
	    if ( (c = getc(fp)) == EOF ) {
		if ( ferror(fp) ) {
		    SIGNAL1("not_possible", sys_errlist[errno]);
		} else {		/* end of file */
		    break;
		}
	    }
	    string_indexc(c, term);
	    if ( retval_area[0] > 0 ) {
		break;
	    }
	    if ( i >= MAXLINE - 1 ) {
		SIGNAL1("not_possible", "too long line");
	    }
	    line[i++] = c;
	    if ( c == '\n' ) {
		tb->line++;
	    }
	}
	if ( c != EOF ) {
	    tb->want_prompt = (fp->_cnt == 0);
	    ungetc(c, fp);
	}
	line[i] = '\0';
	res = (string) malloc_atomic(i + 2);
	strcpy(res, line);
	break;
    }

    case BUF_ISTR: {
	struct sbuf *sb;
	string chars;
	int first, last;
	bool exhausted = FALSE;
	
	stream_empty(s);
	if ( (bool) retval_area[0] ) {
	    SIGNAL0("end_of_file");
	}
	sb = s->buf->val.istr;
	chars = sb->chars;
	first = last = sb->index;
	while ( TRUE ) {
	    char c;
	    if ( string_fetch(chars, last) == SIG ) {
		/* exception name is assumed to be "bounds" */
		exhausted = TRUE;
		break;
	    }
	    c = (char) retval_area[0];
	    string_indexc(c, term);
	    if ( retval_area[0] > 0 ) {
		break;
	    }
	    if ( c == '\n' ) {
		sb->line++;
	    }
	    last++;
	}
	if ( exhausted ) {
	    sb->index = last;
	} else {
	    sb->index = last + 1;
	}
	string_substr(chars, first, last - first);
	res = (string) retval_area[0];
	break;
    }

    default:
	SIGNAL1("not_possible", "cannot read from this stream");
    }

    if ( s->scripting ) {
	/*$B!Z(Bscripting $B$N=hM}![(B*/
    }
    RETURN1(res);
}

/*
 * gets_image = proc(st: stream, term: string) returns(string)
 */

/* stub version */

int stream_gets_image(stream s)
{
    SIGNAL1("failure", "stream$gets_image: not implemented");
}

/*
 * puts = proc(st: stream, s: string) signals(not_possible(string))
 */

int stream_puts(stream s, string line)
{
    FILE *fp;
    int i, sz;

    string_size(line);
    sz = (int) retval_area[0];

    switch ( s->buf->tag ) {

    case BUF_WRITE:
	fp = s->buf->val.write->fp;
	break;

    case BUF_TTY:
	fp = s->buf->val.tty->fp;
	break;

    case BUF_OSTR: {
	array chars;

	chars = s->buf->val.ostr;
	for ( i = 0; i < sz; i++ ) {
	    array_addh(chars, line[i]);
	}
	RETURN0;
    }

    default:
	SIGNAL1("not_possible", "cannot write to this stream");
    }	

    /* Hereafter fp is written */

    for ( i = 0; i < sz; i++ ) {
	if ( putc(line[i], fp) == EOF ) {
	    SIGNAL1("not_possible", sys_errlist[errno]);
	}
    }
    RETURN0;
}

/*
 * puts_image = proc(st: stream, s: string) signals(not_possible(string))
 */

/* stub version */

int stream_puts_image(stream s)
{
    SIGNAL1("failure", "stream$puts_image: not implemented");
}

/*
 * putzero = proc(st: stream, s: string, size: int)
 *			signals(not_possible(string), negative_field_width)

/* stub version */

int stream_putzero(stream s, string str, int i)
{
    SIGNAL1("failure", "stream$putzero: not implemented");
}

/*
 * putleft = proc(st: stream, s: string, size: int)
 *			signals(not_possible(string), negative_field_width)
 */


int stream_putleft(stream st, string s, int size)
{
    int n;

    if ( size < 0 ){
	SIGNAL0("negative_field_width");
    }
    if ( stream_puts(st, s) == SIG ) {
	/* exception name is assumed to be "not_possible" */
	return SIG;
    }
    string_size(s);
    n = size - (int) retval_area[0];
    if ( n > 0 ) {
	if ( stream_putspace(st, n) == SIG ) {
	    /* exception name is assumed to be "not_possible" */
	    return SIG;
	}
    }
    RETURN0;
}

/*
 * putright = proc(st: stream, s: string, size: int)
 *			signals(not_possible(string), negative_field_width)
 */

int stream_putright(stream st, string s, int size)
{
    int n;

    if ( size < 0 ){
	SIGNAL0("negative_field_width");
    }
    string_size(s);
    n = size - (int) retval_area[0];
    if ( n > 0 ) {
	if ( stream_putspace(st, n) == SIG ) {
	    /* exception name is assumed to be "not_possible" */
	    return SIG;
	}
    }
    if ( stream_puts(st, s) == SIG ) {
	/* exception name is assumed to be "not_possible" */
	return SIG;
    }
    RETURN0;
}

/*
 * putspace = proc(st: stream, len: int)
 *			signals(not_possible(string), negative_field_width)
 */

int stream_putspace(stream st, int len)
{
  int i;

  if ( len < 0 ){
      SIGNAL0("negative_field_width");
  }
  for ( i = 0; i < len; i++ ) {
      if ( stream_putc(st, ' ') == SIG ) {
	  /* exception name is assumed to be "not_possible" */
	  return SIG;
      }
  }
  RETURN0;
}

/*
 * set_output_buffered = proc(st: stream, flag: bool)
 */

int stream_set_output_buffered(stream st, bool flag)
{
    switch ( st->buf->tag ) {
    case BUF_WRITE:
	if ( flag == FALSE && st->buf->val.write->buffered ) {
	    /*$B!Z%P%C%U%!$K$?$^$C$F$$$kJ8;z$r$O$-$@$9![(B*/
	}
	st->buf->val.write->buffered = flag;
	/*$B!Z(Bflag $B$K1~$8$?%P%C%U%!%j%s%0=hM}$r9T$J$&![(B*/
	break;
    case BUF_TTY:
	if ( flag == FALSE && st->buf->val.tty->obuffered ) {
	    /*$B!Z%P%C%U%!$K$?$^$C$F$$$kJ8;z$r$O$-$@$9![(B*/
	}
	st->buf->val.tty->obuffered = flag;
	/*$B!Z(Bflag $B$K1~$8$?%P%C%U%!%j%s%0=hM}$r9T$J$&![(B*/
	break;
    case BUF_OSTR:
	SIGNAL1("not_possible", "output is always buffered");
    default:
	SIGNAL1("not_possible", "cannot write to this stream");
    }
    RETURN0;
}

/*
 * get_output_buffered = proc(st: stream) returns(bool)
 */

int stream_get_output_buffered(stream st)
{
    bool res;

    switch ( st->buf->tag ) {
    case BUF_WRITE:
	res = st->buf->val.write->buffered;
	break;
    case BUF_TTY:
	res = st->buf->val.tty->obuffered;
	break;
    case BUF_OSTR:
	res = TRUE;
	break;
    default:
	res = FALSE;
	break;
    }
    RETURN1(res);
}

/*
 * equal = proc(x, y: stream) returns(bool)
 */

int stream_equal(stream s1, stream s2)
{
    RETURN1((s1 == s2) ? TRUE : FALSE);
}

/*
 * similar = proc(x, y: stream) returns(bool)
 */

int stream_similar(stream s1, stream s2)
{
    RETURN1((s1 == s2) ? TRUE : FALSE);
}

/*
 * copy = proc(st: stream) returns(stream)
 */

int stream_copy(stream s)
{
    RETURN1(s);
}

/**************
 * string I/O *
 **************/

/*
 * create_input = proc(s: string) returns(stream)
 */

int stream_create_input(string s)
{
    RETURN1(cons_rep(NULL, buf_make_istr(s)));
}

/*
 * create_output = proc() returns(stream)
 */

int stream_create_output()
{
    RETURN1(cons_rep(NULL, buf_make_ostr()));
}

/*
 * get_contents = proc(st: stream) returns(string)
 *				   signals(not_possible(string))
 */

int stream_get_contents(stream st)
{
    switch ( st->buf->tag ) {
    case BUF_OSTR:
	string_ac2s(st->buf->val.ostr);
	return RET;
    default:
	SIGNAL1("not_possible", "not a string output stream");
    }
}


/****************
 * terminal I/O *
 ****************/

/*
 * getbuf = proc(st: stream, term: string) returns(string)
 *				signals(not_possible(string), end_of_file)
 */

/* stub version */

int stream_getbuf(stream s)
{
    SIGNAL1("failure", "stream$getbuf: not implemented");
}

/*
 * get_prompt = proc(st: stream) returns(string)
 */

int stream_get_prompt(stream st)
{
    switch ( st->buf->tag ) {
    case BUF_TTY:
	RETURN1(st->buf->val.tty->prompt);
    default:
	RETURN1("");
    }
}

/*
 * set_prompt = proc(st: stream, prompt: string)
 */

int stream_set_prompt(stream st, string prompt)
{
    switch ( st->buf->tag ) {
    case BUF_TTY:
	st->buf->val.tty->prompt = prompt;
    }
    RETURN0;
}

/*
 * get_input_buffered = proc(st: stream) returns(bool)
 */

int stream_get_input_buffered(stream st)
{
    bool res;

    switch ( st->buf->tag ) {
    case BUF_READ:
	res = TRUE;
	break;
    case BUF_TTY:
	res = st->buf->val.tty->ibuffered;
	break;
    default:
	res = FALSE;
	break;
    }
    RETURN1(res);
}

/*
 * set_input_buffered = proc(st: stream, flag: bool)
 */

int stream_set_input_buffered(stream st, bool flag)
{
    switch ( st->buf->tag ) {
    case BUF_READ:
	if ( flag == FALSE ) {
	    SIGNAL1("not_possible", "input is always buffered");
	}
	break;
    case BUF_TTY:
	st->buf->val.tty->ibuffered = flag;
	break;
    default:
	SIGNAL1("not_possible", "cannot read from this stream");
    }
    RETURN0;
}
 
/*************
 * scripting *
 *************/

/*
 * add_script = proc(st1, st2: stream) signals(script_failed)
 */

/* stub version */

int stream_add_script(stream st1, stream st2)
{
    SIGNAL1("failure", "stream$add_script: not implemented");
}

/*
 * rem_script = proc(st1, st2: stream)
 */

/* stub version */

int stream_rem_script(stream st1, stream st2)
{
    SIGNAL1("failure", "stream$rem_script: not implemented");
}

/*
 * unscript = proc(st: stream)
 */

/* stub version */

int stream_unscript(stream st)
{
    SIGNAL1("failure", "stream$unscript: not implemented");
}

/*********************************************************************
 * Following operations are not specified in "CLU Reference Manual", *
 * but implemented in MIT CLU.                                       *
 *********************************************************************/

/*
 * pending = proc(st: stream) returns(bool) signals(not_possible(string))
 */

/* tentative version */

int stream_pending(stream st)
{
    stream_empty(st);
    bool_not(retval_area[0]);
    RETURN0;
}

/*
 * display = proc(st: stream, s: string) returns(bool)
 *					 signals(not_possible(string))
 */

/* tentative version */

int stream_display(stream st, string s)
{
    bool res;

    switch ( st->buf->tag ) {
    case BUF_TTY:
	res = FALSE;
	break;
    case BUF_WRITE:
	res = FALSE;
	break;
    default:
	SIGNAL1("not_possible", "cannot write to this stream");
    }
    RETURN1(res);
}

/*
 * modify_display = proc(st: stream, term: string)
 *		    signals(not_possible(string))
 */

/* stub versin */

int stream_modify_display(stream st, string term)
{
    SIGNAL1("failure", "stream$modify_display: not implemented");
}

/*
 * get_rescan = proc(st: stream) returns(string)
 */

/* stub version */

int stream_get_rescan(stream st)
{
    SIGNAL1("failure", "stream$get_rescan: not implemented");
}

/*
 * set_rescan = proc(st: stream, s: string) signals(not_posssible(string))
 */

/* stub version */

int stream_set_rescan(stream st, string s)
{
    SIGNAL1("failure", "stream$set_rescan: not implemented");
}

/*
 * get_eof_flag = proc(st: stream) returns(bool)
 */

/* tentative version */

int stream_get_eof_flag(stream st)
{
    bool res;

    switch ( st->buf->tag ) {
    case BUF_READ:
    case BUF_TTY:
	res = TRUE;
	break;
    default:
	res = FALSE;
	break;
    }
    RETURN1(res);
}

/*
 * set_eof_flag = proc(st: stream, eofok: bool) signals(not_possible(string))
 */

/* tentative version */

int stream_set_eof_flag(stream st, bool eofok)
{
    switch ( st->buf->tag ) {
    case BUF_TTY:
	break;
    case BUF_READ:
    case BUF_ISTR:
    case BUF_CLOSED:
	if ( ! eofok ) {
	    SIGNAL1("not_possible", "cannot disable eof on this stream");
	}
	break;
    default:
	if ( eofok ) {
	    SIGNAL1("not_possible", "cannot enable eof on this stream" );
	}
	break;
    }
    RETURN0;
}

/*
 * print = proc(x: stream, ps: pstream)
 */

/* tentative version */

int stream_print(stream x, clus ps)
{
    if ( _cpstream_text(ps, "(stream)") == SIG ) {
	out_handler();
	return SIG;
    }
    RETURN0;
}

/*
 * _reset = proc()
 */

/* stub version */

int stream__reset()
{
    SIGNAL1("failure", "stream$_reset: not implemented");
}

/*
 * _open_streams = iter() yields(stream)
 */

/* stub version */

int stream__open_streams(bool init, elt **ivarp)
{
    SIGNAL1("failure", "stream$_open_streams: not implemented");
}

/*
 * _close_all = proc()
 */

/* stub version */

int stream__close_all()
{
    SIGNAL1("failure", "stream$_close_all: not implemented");
}

/*
 * end of file : stream.c
 *
 */
