/*
 * $Id: istream.c,v 2.0 1992/09/23 08:41:44 toh-hei Exp $
 *
 * Copyright (c) 1992 Kimura Laboratory, Department of Information Science,
 * Tokyo Institute of Technology.  All Rights Reserved.
 *
 */

/*
 *  OVERVIEW
 *	Istreams provide the means to read and write image files and to
 *	perform some other operations on file objects.  The operations
 *	allowed on any particular istream depend upon the access mode.
 *	In addition, certain operations may be null in some
 *	implementations.
 *	
 *	When an operation cannot be performed, because of an incorrect
 *	access mode, implementation limitations, or properties of an
 *	individual file or device, the operation will signal
 *	not_possible (unless the description of the operation explicitly
 *	says that the invocation will be ignored).
 *	
 *	Actual reading and writing of objects is performed by encode and
 *	decode operations of the types involved.  All of the built-in
 *	CLU types and type generators (except the routine type
 *	generators), and the file_name and date types, provide these
 *	operations.  Designers of abstract types are encouraged to
 *	provide them also.  The type specifications of the encode and
 *	decode operations for a type T are
 *	
 *		encode = proc (c: T, s: istream) 
 *			 signals (not_possible(string))
 *		decode = proc (s: istream) returns (T) 
 *			 signals (end_of_file not_possible(string))
 *	
 *	For parameterized types, encode will have a 'where' clause
 *	requiring encode operations for all components, and decode will
 *	have a 'where' clause requiring decode operations for all
 *	components.
 *	
 *	The encode operations are output operations.  They write an
 *	encoding of the given object onto the istream.  The decode
 *	operations are input operations.  They decode the information
 *	written by encode operations and return an object "similar" to
 *	the one encoded.  If the sequence of decode operations used to
 *	read a file does not match the sequence of encode operations
 *	used to write it, meaningless objects may be returned.  The
 *	system may in some cases be able to detect this condition, in
 *	which case the decode operation will signal not_possible("bad
 *	format").  The system is not guaranteed to detect all such
 *	errors.
 */

/*
 * REPRESENTATION INVARIANT
 *	r->mode == READ_MODE | r->mode == WRITE_MODE | r->mode == CLOSED.
 *	lower bound of r->history is one.
 */
    
#include <errno.h>
#include <strings.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <clu2c.h>
#include <type.h>
#include <glo.h>

#define READ_MODE	0
#define WRITE_MODE	1
#define CLOSED		2

#define rhist_hash(x) (abs((int) x) % RHIST_SIZE)

extern char *sys_errlist[];

/*
 *  open = proc (fn: file_name, mode: string) signals (not_possible(string)) 
 *	effects  The possible access modes are "read", "write", and
 *	    "append".  If mode is not one of these strings,
 *	    not_possible("bad access mode") is signaled. In those cases
 *	    where the system is able to detect that the specified
 *	    preexisting file is not an image file, not_possible("wrong
 *	    file type") is signaled.  If mode is "read", the named file
 *	    must exist. If the file exists, an image stream is returned
 *	    upon which decode operations can be performed.  If mode is
 *	    "write", a new file is created or an old file is rewritten.
 *	    An image stream is returned upon which encode operations can
 *	    be performed.  Write mode to storage files should guarantee
 *	    exclusive access to the file, if possible.  If mode is
 *	    "append" and if the named file does not exist, one is
 *	    created.  An image stream is returned, positioned at the end
 *	    of the file, upon which encode operations can be performed.
 *	    Append mode to storage files should guarantee exclusive
 *	    access to the file, if possible.
 */

int istream_open(file_name f, string access)
{
    char *filename, *type;
    int mode, i;
    FILE *fp;
    array history;
    istream s;

    file_name_unparse(f);
    filename = (char *) retval_area[0];

    if (strcmp((char *) access, "read") == 0) {
	type = "r";
	mode = READ_MODE;
    } else if (strcmp((char *) access, "write") == 0) {
	type = "w";
	mode = WRITE_MODE;
    } else if (strcmp((char *) access, "append") == 0) {
	type = "a";
	mode = WRITE_MODE;
    } else {
	signame = "not_possible";
	sigarg_area[0] = (elt) "bad access mode";
	return(SIG);
    }

    if ((fp = fopen(filename, type)) == NULL) {
	signame = "not_possible";
	sigarg_area[0] = (elt) sys_errlist[errno];
	return(SIG);
    }

    s = (istream) malloc(sizeof(struct istream_rep));
    s->fp = fp;
    s->mode = mode;
    s->fn = f;
    array_new();
    s->history = (array) retval_area[0];
    for (i = 0; i < RHIST_SIZE; i++) {
	s->rhistory[i] = NULL;
    }

    retval_area[0] = (elt) s;
    return(RET);
}

/*
 *  can_read = proc (s: istream) returns (bool)
 *	effects  Returns true if decode operations appear possible on s.
 */

int istream_can_read(istream s)
{
    switch (s->mode) {
    case READ_MODE:
	retval_area[0] = (elt) TRUE;
	break;
    default:
	retval_area[0] = (elt) FALSE;
	break;
    }
    return(RET);
}

/*
 *  can_write = proc (s: istream) returns (bool)
 *	effects  Returns true if encode operations appear possible on s.
 */

int istream_can_write(istream s)
{
    switch (s->mode) {
    case WRITE_MODE:
	retval_area[0] = (elt) TRUE;
	break;
    default:
	retval_area[0] = (elt) FALSE;
	break;
    }
    return(RET);
}

/*
 *  empty = proc (istream) returns (bool) 
 *	effects  Returns true if and only if there are no more objects
 *	    in the associated file.
 */

int istream_empty(istream s)
{
    FILE *fp;
    int c;

    switch (s->mode) {
    case READ_MODE:
	fp = s->fp;
	c = getc(fp);		/* touch the stream */
	ungetc(c, fp);		/* (ditto) */
	if (feof(fp)) { 
	    retval_area[0] = (elt) TRUE;
	} else {
	    retval_area[0] = (elt) FALSE;
	}
	return(RET);
    default:

	/*
	 * Here the not_possible exception is signaled, although not
	 * described in the manual to do so.  We think this is
	 * preferable.  The MIT implementation of istream$empty also
	 * signals this execption.
 	 */

	signame = "not_possible";
	sigarg_area[0] = (elt) "cannot read from this istream";
	return(SIG);
    }
}

/*
 *  reset = proc (s: istream) signals (not_possible(string)) 
 *	effects  Resets s so that the next input or output operation
 *	    will read or write the first item in the file.
 */

int istream_reset(istream s)
{
    int i;

    switch (s->mode) {
    case READ_MODE:
    case WRITE_MODE:
	if (rewind(s->fp) == -1) {
	    signame = "not_possible";
	    sigarg_area[0] = (elt) sys_errlist[errno];
	    return(SIG);
	}
	array_trim(s->history, 1, 0);
	for (i = 0; i < RHIST_SIZE; i++) {
	    s->rhistory[i] = NULL;
	}
	break;
    default:
	signame = "not_possible";
	sigarg_area[0] = (elt) "cannot reset closed istream";
	return(SIG);
    }
    return(RET);
}

/*
 *  flush = proc (s: istream) signals (not_possible(string))
 *	effects   Writes any buffered output to the associated file, if
 *	    possible.
 */

int istream_flush(istream s)
{
    switch (s->mode) {
    case WRITE_MODE:
	if (fflush(s->fp) == EOF) {
	    signame = "not_possible";
	    sigarg_area[0] = (elt) sys_errlist[errno];
	    return(SIG);
	}
	break;
    case READ_MODE:
	signame = "not_possible";
	sigarg_area[0] = (elt) "cannot flush input istream";
	return(SIG);
    case CLOSED:
	signame = "not_possible";
	sigarg_area[0] = (elt) "cannot flush closed istream";
	return(SIG);
    }
    return(RET);
}

/*
 *  get_date = proc (s: istream) returns (date) signals (not_possible(string)) 
 *	effects  Returns the date of the last modification of the
 *	    corresponding storage file.
 */

int istream_get_date(istream s)
{
    char *path;
    struct stat buf;
    long clock;
    struct tm *time;

    switch (s->mode) {
    case READ_MODE:
    case WRITE_MODE:
	break;
    default:
	signame = "not_possible";
	sigarg_area[0] = (elt) "cannot get date";
	return(SIG);
    }

    file_name_unparse(s->fn);
    path = (char *) retval_area[0];
    if (stat(path, &buf) < 0) {
	signame = "not_possible";
	sigarg_area[0] = (elt) sys_errlist[errno];
	return(SIG);
    }
    clock = (long) buf.st_mtime;
    time = localtime((long *) &clock);
    if (_cdate_create(time->tm_mday,
                      time->tm_mon + 1,
                      time->tm_year + 1900,
                      time->tm_hour,
                      time->tm_min,
                      time->tm_sec) == SIG) {
	signame = "not_possible";
	sigarg_area[0] = (elt) "cannot get date";
	return(SIG);
    }
    
    /*
     *	Here we have the result date object in retval_area[0], so we do
     *	not have to set return value into retval_area[0].
     */
    
    return(RET);
}

/*
 *  set_date = proc (s: istream, d: date) signals (not_possible(string))
 *	modifies  s. 
 *	effects  Sets the modification date of the corresponding storage
 *	    file. The modification date is set automatically when a file
 *	    is opened in "write" or "append" mode.
 */

int istream_set_date(istream s)
{
    signame = "not_possible";
    sigarg_area[0] = (elt) "cannot set date";
    return(SIG);
}

/*
 *  get_name = proc (s: istream) returns (file_name) 
 *	effects  Returns the name of the corresponding file.  It may be
 *	    different than the name used to open the file, in that
 *	    defaults have been resolved and link indirections have been
 *	    followed.
 */

int istream_get_name(istream s)
{
    retval_area[0] = (elt) s->fn;
    return(RET);
}

/*
 *  close = proc (s: istream) signals (not_possible(string))
 *	modifies  s. 
 *	effects  Attempts to terminate I/O and remove the association
 *	    between s and the file.  If successful, further use of
 *	    operations that signal not_possible will signal not_possible.
 *	    This operation will fail if buffered output cannot be
 *	    written.
 */

int istream_close(istream s)
{
    if (fclose(s->fp) == EOF) {
	signame = "not_possible";
	sigarg_area[0] = (elt) sys_errlist[errno];
	return(SIG);
    }
    s->mode = CLOSED;
    return(RET);
}

/*
 *  abort = proc (s: istream)
 *	modifies  s. 
 *	effects  Terminates I/O and removes the association between the
 *	    istream and the file.  If buffered output cannot be written,
 *	    it will be lost, and if a new file is being written, it may
 *	    or may not exist.
 */

int istream_abort(istream s)
{
    fclose(s->fp);
    s->mode = CLOSED;
    return(RET);
}

/*
 *  is_closed = proc (s: istream) returns (bool)
 *	effects  Returns true if and only if s is closed.
 */

int istream_is_closed(istream s)
{
    switch (s->mode) {
    case CLOSED:
	retval_area[0] = (elt) TRUE;
	break;
    default:
	retval_area[0] = (elt) FALSE;
	break;
    }
    return(RET);
}

/*
 *  equal = proc (s1, s2: istream) returns (bool)
 *	effects  Returns true if and only if both arguments are the same
 *	    istream.
 */

int istream_equal(istream s1, istream s2)
{
    retval_area[0] = (elt) ((s1 == s2) ? TRUE : FALSE);
    return(RET);
}

/*
 *  similar = proc (s1, s2: istream) returns (bool)
 *	effects  Returns true if and only if both arguments are the same
 *	    istream.
 */

int istream_similar(istream s1, istream s2)
{
    retval_area[0] = (elt) ((s1 == s2) ? TRUE : FALSE);
    return(RET);
}

/*
 *  copy = proc (s: istream) returns (istream)
 *	effects  Returns a stream that is equal to s.
 */

int istream_copy(istream s)
{
    retval_area[0] = (elt) s;
    return(RET);
}

/*
 *	internal operations
 */

/*
 *  geti = proc (s: istream) returns (int)
 *			     signals (end_of_file, not_possible(string))
 *	modifies  s.
 *	effects
 */

int istream_geti(istream s)
{
    FILE *fp = s->fp;
    int i;

    switch (s->mode) {
    case READ_MODE:
	break;
    default:
	signame = "not_possible";
	sigarg_area[0] = (elt) "cannot read from this istream";
	return(SIG);
    }

    i = getc(fp);
    ungetc(i, fp);
    if (feof(fp)) {
	signame = "end_of_file";
	return(SIG);
    }
    
    i = getw(fp);
    if (ferror(fp)) {
	signame = "not_possible";
	sigarg_area[0] = (elt) sys_errlist[errno];
	return(SIG);
    }
    retval_area[0] = (elt) i;
    return(RET);
}

/*
 *  puti = proc (s: istream, i: int) signals(not_possible(string))
 *	modifies  s.
 *	effects
 */

int istream_puti(istream s, int i)
{
    FILE *fp = s->fp;

    switch (s->mode) {
    case WRITE_MODE:
	break;
    default:
	signame = "not_possible";
	sigarg_area[0] = (elt) "cannot write to this istream";
	return(SIG);
    }

    putw(i, fp);
    if (ferror(fp)) {
	signame = "not_possible";
	sigarg_area[0] = (elt) sys_errlist[errno];
	return(SIG);
    }
    return(RET);
}

/*
 *  getc = proc (s: istream) returns (char)
 *			     signals (end_of_file, not_possible(string))
 *	modifies  s.
 *	effects
 */

int istream_getc(istream s)
{
    FILE *fp = s->fp;
    int i;

    switch (s->mode) {
    case READ_MODE:
	break;
    default:
	signame = "not_possible";
	sigarg_area[0] = (elt) "cannot read from this istream";
	return(SIG);
    }

    i = getc(fp);
    ungetc(i, fp);
    if (feof(fp)) {
	signame = "end_of_file";
	return(SIG);
    }

    if ((i = getc(fp)) == EOF) {
	signame = "not_possible";
	sigarg_area[0] = (elt) sys_errlist[errno];
	return(SIG);
    }
    retval_area[0] = (elt) i;
    return(RET);
}

/*
 *  putc = proc (s: istream, c: char) signals (not_possible(string))
 *	modifies  s.
 *	effects
 */

int istream_putc(istream s, char c)
{
    FILE *fp = s->fp;

    switch (s->mode) {
    case WRITE_MODE:
	break;
    default:
	signame = "not_possible";
	sigarg_area[0] = (elt) "cannot write to this istream";
	return(SIG);
    }

    if (putc(c, fp) == EOF) {
	signame = "not_possible";
	sigarg_area[0] = (elt) sys_errlist[errno];
	return(SIG);
    }
    return(RET);
}

int istream_check_history(istream s, elt x)
{
    struct bucket *buck, *buck0;
    int id, i;

    for (buck = s->rhistory[rhist_hash(x)]; buck != NULL; buck = buck->next) {
	if (buck->key == x) {
	    retval_area[0] = (elt) buck->val;
	    return(RET);
	}
    }

    /* not found */

    array_addh(s->history, x);
    array_high(s->history);
    id = (int) retval_area[0];
    i = rhist_hash(x);
    buck0 = s->rhistory[i];
    buck = (struct bucket *) malloc(sizeof(struct bucket));
    buck->key = x;
    buck->val = id;
    buck->next = buck0;
    s->rhistory[i] = buck;
    retval_area[0] = (elt) NOT_YET;
    return(RET);
}

int istream_add_history(istream s, elt x)
{
    array_addh(s->history, x);
    return(RET);
}

/*
 *  get_obj = proc(s: istream, id: int) returns(???) signals(bounds)
 *	effects  Returns object corresponding to id in history. Signals
 *	    bounds if there is no object corresponding to id.
 */

int istream_get_obj(istream s, int id)
{
    return(array_fetch(s->history, id));
}
