#ifndef	STRING_H
#define	STRING_H

/*$Header: String.h,v 2.204 89/10/07 23:21:59 keith Stab $*/

/* String.h -- declarations for character strings

	THIS SOFTWARE FITS THE DESCRIPTION IN THE U.S. COPYRIGHT ACT OF A
	"UNITED STATES GOVERNMENT WORK".  IT WAS WRITTEN AS A PART OF THE
	AUTHOR'S OFFICIAL DUTIES AS A GOVERNMENT EMPLOYEE.  THIS MEANS IT
	CANNOT BE COPYRIGHTED.  THIS SOFTWARE IS FREELY AVAILABLE TO THE
	PUBLIC FOR USE WITHOUT A COPYRIGHT NOTICE, AND THERE ARE NO
	RESTRICTIONS ON ITS USE, NOW OR SUBSEQUENTLY.

Authors:
	C. J. Eppich and K. E. Gorlen
	Bg. 12A, Rm. 2033
	Computer Systems Laboratory
	Division of Computer Research and Technology
	National Institutes of Health
	Bethesda, Maryland 20892
	Phone: (301) 496-1111
	uucp: uunet!nih-csl!keith
	Internet: keith@alw.nih.gov
	December, 1987

$Log:	String.h,v $
 * Revision 2.204  89/10/07  23:21:59  keith
 * Pre-release
 * 
 * Revision 2.203  89/08/08  15:24:43  keith
 * Pre-release
 * 
 * Revision 2.202.1.2  89/07/12  16:34:06  keith
 * Eliminate unnecessary SubString(String&, unsigned, unsigned)
 * constructor.
 * 
 * Revision 2.202.1.1  89/07/01  21:56:25  keith
 * Base revision for R2.00 MI version
 * 
 * Revision 2.202  89/06/23  10:43:00  keith
 * Base revision for AT&T C++ R2.0 release (Cycle 20)
 * 
 * Revision 2.201.1.5  89/06/23  10:41:13  keith
 * Implement:
 * 	SubString(const String&, unsigned, unsigned)
 * 	const SubString operator()(unsigned pos, unsigned lgt) const
 * 	const SubString operator()(const Range& r) const
 * 
 * Revision 2.201.1.4  89/06/23  07:39:31  keith
 * Change return type of operator[]() const and at() const to char
 * instead of const char&.
 * 
 * Revision 2.201.1.3  89/06/21  22:05:39  keith
 * Add private copy constructor to private classes.
 * 
 * Revision 2.201.1.2  89/06/01  23:17:42  keith
 * Remove base class argument from DECLARE_MEMBERS.
 * 
 * Revision 2.201.1.1  89/05/19  15:48:28  keith
 * Add base class arg to DECLARE_MEMBERS.
 * Place enums and typedefs in classes.
 * 
 * Revision 2.201  89/05/12  11:20:47  keith
 * Release for R2.0 Beta test.
 * 
 * Revision 2.200.1.4  89/05/12  11:02:37  keith
 * Revised Object I/O.
 * 
 * Revision 2.200.1.3  89/05/05  10:00:18  keith
 * Overload reference-returning member functions (e.g.
 * operator[](int) and at(int) for const instances.
 * 
 * Revision 2.200.1.2  89/04/25  10:37:15  keith
 * Remove SubString::SubString(const SubString) from String.h.
 * 
 * Revision 2.200.1.1  89/04/24  17:19:19  keith
 * Working revision for R2.0 Beta 6++
 * 
 * Revision 2.200  89/04/17  23:31:50  keith
 * Base revision for R2.0 Beta 6.
 * 
 * Revision 2.121  89/02/16  11:10:05  keith
 * Base revision for C++ R1.2.1 compatible version.
 * 
*/

#include "Object.h"
#include <string.h>
#include <malloc.h>

const unsigned DEFAULT_STRING_EXTRA = 15;
class String;
class Range;

class SubString : public NIHCL {
	char* sp;		// substring pointer 
        unsigned sl;		// substring length 
	String* st;		// String this is a SubString of
	SubString(const String&, unsigned, unsigned);
	SubString(const SubString& ss) {
		sp = ss.sp; sl = ss.sl; st = ss.st;
	}
	int compare(const char*) const;
	int compare(const String&) const;
	int compare(const SubString&) const;
	void replace(const char* st, unsigned ln);
	void checkSubStr() const;
	friend String;
public:
	const char* ptr() const	{ return sp; }
	unsigned position() const;
	unsigned length() const	{ return sl; }

	void operator=(const String&);
	void operator=(const SubString&);
	void operator=(const char*);

	bool operator<(const String&) const;
	bool operator>(const String&) const;
	bool operator<=(const String&) const;
	bool operator>=(const String&) const;
	bool operator==(const String&) const; 
	bool operator!=(const String&) const;

	bool operator<(const SubString& ss) const	{ return compare(ss)<0; } 
	bool operator>(const SubString& ss) const	{ return compare(ss)>0; }
	bool operator<=(const SubString& ss) const	{ return compare(ss)<=0; }
	bool operator>=(const SubString& ss) const	{ return compare(ss)>=0; }
	bool operator==(const SubString& ss) const	{ return compare(ss)==0; }
	bool operator!=(const SubString& ss) const	{ return compare(ss)!=0; }

	bool operator<(const char* cs) const	{ return compare(cs) < 0; }   
	bool operator>(const char* cs) const	{ return compare(cs) > 0; }   
	bool operator<=(const char* cs) const	{ return compare(cs) <= 0; }
	bool operator>=(const char* cs) const	{ return compare(cs) >= 0; }
	bool operator==(const char* cs) const	{ return compare(cs) == 0; }
	bool operator!=(const char* cs) const	{ return compare(cs) != 0; }

	friend bool operator<(const char* cs, const SubString& ss) {
	    return ss.compare(cs) > 0;
	}
	friend bool operator>(const char* cs, const SubString& ss) {	 
	    return ss.compare(cs) < 0;
	}
	friend bool operator<=(const char* cs, const SubString& ss) { 
	    return ss.compare(cs) >= 0; 
	}
	friend bool operator>=(const char* cs, const SubString& ss) { 
	    return ss.compare(cs) <= 0;
	}
	friend bool operator==(const char* cs, const SubString& ss) const {
	    return ss.compare(cs) == 0; 
	}
	friend bool operator!=(const char* cs, const SubString& ss) {
	    return ss.compare(cs) != 0; 
	}

	String operator&(const String&) const;
	String operator&(const SubString&) const;
	String operator&(const char*) const;
	friend String operator&(const char*, const SubString&);

	void dumpOn(ostream& strm =cerr) const;
	void printOn(ostream& strm =cout) const;
};

class String: public VIRTUAL Object {
	DECLARE_MEMBERS(String);
	char* p;		// character string
	unsigned len;		// length of string, excluding null character
	unsigned alloc;		// amount of storage allocated
	void indexRangeErr() const;
	friend SubString;
protected:		// storer() functions for object I/O
	virtual void storer(OIOofd&) const;
	virtual void storer(OIOout&) const;		
public:
	String(char& c, unsigned l=1, unsigned extra=DEFAULT_STRING_EXTRA);
	String();
	String(unsigned extra);
	String(const char*);
	String(const char*, unsigned extra);
	String(const String&);
	String(const String&, unsigned extra);
	String(const SubString&);
	String(const SubString&, unsigned extra);
	~String();
	operator const char*() const	{ return p; }
	SubString operator()(unsigned pos, unsigned lgt);
	const SubString operator()(unsigned pos, unsigned lgt) const;
	SubString operator()(const Range& r);
	const SubString operator()(const Range& r) const;
	char& operator[](unsigned i) {
		if (i >= len) indexRangeErr();
		return p[i];
    	}
	char operator[](unsigned i) const {
		if (i >= len) indexRangeErr();
		return p[i];
    	}

	void operator=(const String&);
	void operator=(const SubString&);
	void operator=(const char*);

	bool operator<(const String& s) const	{ return strcmp(p, s.p) < 0; }
	bool operator>(const String& s) const	{ return strcmp(p, s.p) > 0; }
	bool operator<=(const String& s) const	{ return strcmp(p, s.p) <= 0; }
	bool operator>=(const String& s) const	{ return strcmp(p, s.p) >= 0; }
	bool operator==(const String& s) const;
	bool operator!=(const String& s) const	{ return !(*this==s); }

	bool operator<(const SubString& ss) const 	{ return ss.compare(*this) > 0; }
	bool operator>(const SubString& ss) const	{ return ss.compare(*this) < 0; }
	bool operator<=(const SubString& ss) const	{ return ss.compare(*this) >= 0; }
	bool operator>=(const SubString& ss) const	{ return ss.compare(*this) <= 0; }
	bool operator==(const SubString& ss) const	{ return ss.compare(*this) == 0; }
	bool operator!=(const SubString& ss) const	{ return ss.compare(*this) != 0; }

	bool operator<(const char* cs) const	{ return strcmp(p,cs) < 0; }   
	bool operator>(const char* cs) const	{ return strcmp(p,cs) > 0; }   
	bool operator<=(const char* cs) const	{ return strcmp(p,cs) <= 0; }
	bool operator>=(const char* cs) const	{ return strcmp(p,cs) >= 0; }
	bool operator==(const char* cs) const	{ return strcmp(p,cs) == 0; }
	bool operator!=(const char* cs) const	{ return strcmp(p,cs) != 0; }

	friend bool operator<(const char* cs, const String& s) {
	    return strcmp(cs, s.p) < 0;
	}
	friend bool operator>(const char* cs, const String& s) {	 
	    return strcmp(cs, s.p) > 0;
	}
	friend bool operator<=(const char* cs, const String& s) { 
	    return strcmp(cs, s.p) <= 0; 
	}
	friend bool operator>=(const char* cs, const String& s) { 
	    return strcmp(cs, s.p) >= 0;
	}
	friend bool operator==(const char* cs, const String& s) {
	    return strcmp(cs, s.p) == 0; 
	}
	friend bool operator!=(const char* cs, const String& s) {
	    return strcmp(cs, s.p) != 0; 
	}

	String operator&(const String& s) const;
	String operator&(const SubString& ss) const;
	String operator&(const char* cs) const;
	friend String operator&(const char*, const SubString&);
	friend String operator&(const char* cs, const String& s);

	String& operator&=(const String&);
	String& operator&=(const SubString&);
	String& operator&=(const char* cs);

	char& at(unsigned i)			{ return (*this)[i]; }
	char at(unsigned i) const		{ return (*this)[i]; }
	unsigned length() const	{ return len; } 
	unsigned reSize(unsigned new_capacity);

	virtual void toAscii();
	virtual void toLower();
	virtual void toUpper();
	virtual unsigned capacity() const	{ return alloc - 1; }
	virtual int compare(const Object& ob) const;
	virtual void deepenShallowCopy();
	virtual unsigned hash() const;
	virtual bool isEqual(const Object&) const;
	virtual void printOn(ostream& strm =cout) const;
	virtual void scanFrom(istream& strm);
	virtual unsigned size() const;
	virtual const Class* species() const;
};

inline ostream& operator<<(ostream& strm, const SubString& ss)
{
	ss.printOn(strm);
	return(strm);
}

inline SubString::SubString(const String& s, unsigned pos, unsigned lgt)
{ 
	sp = &((String&)s).p[pos];
	sl = lgt;
	st = &(String&)s;
	checkSubStr();
}
	
inline unsigned SubString::position() const	{ return sp - st->p; }

inline SubString String::operator()(unsigned pos, unsigned lgt)
{	
	return SubString(*this, pos, lgt);
}

inline const SubString String::operator()(unsigned pos, unsigned lgt) const
{	
	return SubString(*this, pos, lgt);
}

inline bool SubString::operator<(const String& s) const { return compare(s) < 0; }
inline bool SubString::operator>(const String& s) const { return compare(s) > 0; }
inline bool SubString::operator<=(const String& s) const { return compare(s) <= 0; }
inline bool SubString::operator>=(const String& s) const { return compare(s) >= 0; }
inline bool SubString::operator==(const String& s) const { return compare(s) == 0; }
inline bool SubString::operator!=(const String& s) const { return compare(s) != 0; }

#endif

