/* Exception.c -- NIHCL exception handling

	THIS SOFTWARE FITS THE DESCRIPTION IN THE U.S. COPYRIGHT ACT OF A
	"UNITED STATES GOVERNMENT WORK".  IT WAS WRITTEN AS A PART OF THE
	AUTHOR'S OFFICIAL DUTIES AS A GOVERNMENT EMPLOYEE.  THIS MEANS IT
	CANNOT BE COPYRIGHTED.  THIS SOFTWARE IS FREELY AVAILABLE TO THE
	PUBLIC FOR USE WITHOUT A COPYRIGHT NOTICE, AND THERE ARE NO
	RESTRICTIONS ON ITS USE, NOW OR SUBSEQUENTLY.

Author:
	K. E. Gorlen
	Bg. 12A, Rm. 2033
	Computer Systems Laboratory
	Division of Computer Research and Technology
	National Institutes of Health
	Bethesda, Maryland 20892
	Phone: (301) 496-1111
	uucp: uunet!nih-csl!keith
	Internet: keith@alw.nih.gov
	February, 1986

Function:

NIHCL exception handling.
	
$Log:	Exception.c,v $
 * Revision 2.204  89/10/07  23:19:17  keith
 * Pre-release
 * 
 * Revision 2.203  89/08/08  15:21:47  keith
 * Pre-release
 * 
 * Revision 2.202.1.5  89/08/01  20:11:07  keith
 * Put returns removed for compatibility with OSI C++
 * back in constructors.
 * 
 * Revision 2.202.1.4  89/07/07  14:02:34  keith
 * Clean up cast-away const where possible
 * 
 * Revision 2.202.1.3  89/07/03  17:45:32  keith
 * Define static member variables.
 * 
 * Revision 2.202.1.2  89/07/01  22:59:42  keith
 * Now OK to use Object::nil to initialize Catch::obj
 * since Object::nil initialized by NIHCL::NIHCL().
 * 
 * Revision 2.202.1.1  89/07/01  21:53:44  keith
 * Base revision for R2.00 MI version
 * 
 * Revision 2.202  89/06/22  20:53:45  keith
 * Base revision for AT&T C++ R2.0 release (Cycle 20)
 * 
 * Revision 2.201.1.4  89/06/21  14:48:38  sandy
 * modified setError() to make variable length argument list portable
 * 
 * Revision 2.201.1.3  89/06/20  15:35:18  keith
 * Add explicit base class names to constructor initializer
 * lists.
 * 
 * Revision 2.201.1.2  89/06/07  11:37:38  keith
 * Fix topCatch so it doesn't use nil pointer,
 * which may not be initialized when
 * topCatch constructed.
 * 
 * Revision 2.201.1.1  89/05/19  15:41:15  keith
 * Add base class arg to DECLARE_MEMBERS.
 * Place enums and typedefs in classes.
 * 
 * Revision 2.201  89/05/12  11:16:47  keith
 * Release for R2.0 Beta test.
 * 
 * Revision 2.200.1.2  89/05/12  10:57:32  keith
 * Revised Object I/O.
 * 
 * Revision 2.200.1.1  89/04/24  17:14:13  keith
 * Working revision for R2.0 Beta 6++
 * 
 * Revision 2.200  89/04/17  23:28:22  keith
 * Base revision for R2.0 Beta 6.
 * 
 * Revision 2.121  89/02/16  11:04:17  keith
 * Base revision for C++ R1.2.1 compatible version.
 * 
*/

#include "ExceptAct.h"
#include <osfcn.h>
#include <libc.h>
#include <errlib.h>
#include <stdarg.h>

#include "nihclconfig.h"
#include "nihclerrs.h"
#include "nihclerrsx.h"

static char rcsid[] = "$Header: Exception.c,v 2.204 89/10/07 23:19:17 keith Stab $";

static ExceptionEnv topExceptionEnv;

static Catch topCatch;
Catch* Catch::stackTop = &topCatch;
ExceptionEnv* ExceptionEnv::stackTop = 0;
ExceptionEnv* ExceptionEnv::lastResort = 0;

void ExceptionEnv::raise(int exception)
{
	if (exception == 0) {
		cerr << "Tried to RAISE exception code 0\n";
		abort();
	}
	if (prev == 0) {	// i.e.,  this == lastResort
		cerr << "NIHCL: fatal: Unhandled exception code " << exception << '\n';
		exit(1);
	}
	pop();
#if STACK_GROWS_DOWN
	for (register Catch* p = Catch::stackTop->next; p < (Catch*)this && p != Catch::stackTop; p = p->next)
#else
	for (register Catch* p = Catch::stackTop->next; p > (Catch*)this && p != Catch::stackTop; p = p->next)
#endif
		p->obj->destroyer();
	Catch::stackTop->next = p;
	p->prev = Catch::stackTop;
	longjmp(env,exception);
}

Catch::Catch()
{
	next = prev = this;
	obj = Object::nil;
}

Catch::Catch(Object* ob)
{
	obj = ob;
	void* top[2];
#if STACK_GROWS_DOWN
	if ((void*)this < (void*)ExceptionEnv::stackTop && (void*)this > (void*)top) {	// this on stack
#else
	if ((void*)this > (void*)ExceptionEnv::stackTop && (void*)this < (void*)top) {	// this on stack
#endif
		next = stackTop->next;
		prev = stackTop;
		stackTop->next = this;
		next->prev = this;
	}
	else next = 0;		// Catch object was allocated by new
}

Catch::~Catch()
{
	if (next != 0) {
		next->prev = prev;
		prev->next = next;
	}
	obj->destroyer();
}

static ExceptionActionTbl mainExceptionActionTbl;
ExceptionActionTbl* ExceptionActionTbl::active = &mainExceptionActionTbl;

void NIHCL::setError(int err, int sev, ...)
{
	ExceptionActionTbl::active->act((unsigned)err,sev);
#ifdef TRACE
printf("seterror> Error No: %d  Severity: %d\n",err,sev);
#endif
	errno = err;
        char buf[132],buf0[132];

	errfac_errorText(err,buf/*OUT*/);
     
	char* p = errfac_argumentCodes(err);
#ifdef TRACE
printf("seterror> argument codes %s\n",p);
#endif
	va_list ap;
	va_start(ap,sev);
	while ( *p != '\0' ) { // set up next arg
	  errfac_prepNextArg(buf/*IN*/,buf0/*OUT*/);
	  switch ( *(p++) ) {
	     case 'I': 
		int i = va_arg(ap,int);
	     	sprintf(buf,buf0,i);
    	      	break;
	     case 'S':
		char* s = va_arg(ap,char*);
	     	sprintf(buf,buf0,s);
	        break;
	     case 'D':
	        double d = va_arg(ap,double);
	     	sprintf(buf,buf0,d);
	        break;
	     default : // ignored
	        break;
	     }// end switch
          }// end while
	va_end(ap);

	int severity = (sev < 0)? sev:(errfac_severity(err));
	errfac_errorMessage(severity,buf);
	if ( errfac_isExitSeverity(severity) ) {
	   cerr << "NIHCL: fatal: Tried to continue after error "
		<< err <<'\n';
	   abort();
	   }
}

ExceptionActionTbl::ExceptionActionTbl()
{
	client_exception_trap = NULL;
	for (register int i=NIHCL__LAST_ERROR-NIHCL__FIRSTERROR; i>=0; i--)
		action[i] = ExceptionAction::ABORT;
}

void ExceptionActionTbl::act(unsigned error, int sev)
{
	if (client_exception_trap != NULL) {
		(*client_exception_trap)(error,sev);
	}
	if (error >= NIHCL__FIRSTERROR && error <= NIHCL__LAST_ERROR) {
		switch (action[error-NIHCL__FIRSTERROR]) {
			case ExceptionAction::RAISE: RAISE(error);
			case ExceptionAction::ABORT: return;
		}
	}
}

ExceptionAction::ExceptionAction(unsigned error)
{
	if ( (error < NIHCL__FIRSTERROR)||(error > NIHCL__LAST_ERROR) ) {
		setError(NIHCL_BADERRNUM,DEFAULT,"ExceptionAction",error);
		return;
	}
	error_code = error;
	old_action = ExceptionActionTbl::active->action[error-NIHCL__FIRSTERROR];
}

ExceptionAction::~ExceptionAction()
{
	ExceptionActionTbl::active->action[error_code-NIHCL__FIRSTERROR] = old_action;
}

AbortException::AbortException(unsigned error) : ExceptionAction(error)
{
	ExceptionActionTbl::active->action[error-NIHCL__FIRSTERROR] = ExceptionAction::ABORT;
}

RaiseException::RaiseException(unsigned error) : ExceptionAction(error)
{
	ExceptionActionTbl::active->action[error-NIHCL__FIRSTERROR] = ExceptionAction::RAISE;
}

ExceptionTrap::ExceptionTrap(exceptionTrapTy xtrap)
{
	old_trap = ExceptionActionTbl::active->client_exception_trap;
	ExceptionActionTbl::active->client_exception_trap = xtrap;
}

ExceptionTrap::~ExceptionTrap()
{
	ExceptionActionTbl::active->client_exception_trap = old_trap;
}
