#ifndef System_First
#define System_First

#include "Port.h"
#include "Object.h"

//---- Directory ---------------------------------------------------------------

class Directory : public Root {
public:
    Directory(char*)
	{ }
    virtual ~Directory();
    virtual char *operator()();
};

//---- abstract class for a connection with a pseudo tty ---------------------

struct PttyChars { // struct to return the settings of the various tty characters
    char erase;
    char kill;
    char rprnt;
    char susp;
    char intr;
    char quit;
    char eof;
};

class PttyConnection : public Root {
protected:
    PttyConnection();
public:
    virtual ~PttyConnection();
    virtual FILE *GetFile();
    virtual int GetPid();
    virtual int GetFileNo();
    virtual bool SubmitToSlave(char *buf, int n);
    virtual int Read(char *buf, int size);
    virtual int GetMode();
    virtual void KillChild();
    virtual void CleanupPtty();
    virtual bool Echo();
    virtual bool RawOrCBreak();
    virtual void BecomeConsole();
    virtual void SetSize(int rows, int cols);
    virtual void GetPttyChars(PttyChars *);
};

//---- class SysEvtHandler ---------------------------------------
// abstract class for handling events from system resources

enum SysEventCodes {
    eSysEvtRead,
    eSysEvtWrite,
    eSysEvtZombie,
    eSysEvtSignal,
    eSysEvtAsyncSignal,
    eSysEvtTimeout
};

class SysEvtHandler : public Object {
    bool remove;
    int resourceId;
public:
    MetaDef(SysEvtHandler);
    SysEvtHandler(int resource);
    virtual ~SysEvtHandler();
    int GetResourceId();
    void SetResourceId(int);
    virtual void Notify(SysEventCodes code, int val);
    virtual bool HasInterest();
    virtual bool ShouldRemove();
    void Remove();
    bool IsEqual(Object *op);
};

//---- System ------------------------------------------------------------------

class System : public Root {
protected:
    bool done;
    char *errorstr, *osid;

    class ObjList *fileInputHandler;
    class ObjList *fileOutputHandler;
    class ObjList *zombieHandler;    
    class ObjList *signalHandler;    
    class ObjList *asyncSignalHandler;    
    class ObjList *timeoutHandler;
    class ObjList *cleanupList;

public:  
    static bool anyremoved;

public:
    System(char *name= "GENERIC");
    virtual ~System();

    virtual bool Init();

    char *GetErrorStr()
	{ return errorstr; }
    char *GetSystemId()
	{ return osid; }
    virtual void Control();
    void ExitControl()
	{ done= TRUE; }
    virtual void InnerLoop();

    virtual class ClipBoard *MakeClipboard();
    virtual class Directory *MakeDirectory(char *name);
    virtual class FileType *GetFileType(char *path);
	
    //---- expand the metacharacters in Pat as in the shell
    virtual bool ExpandPathName(char *patbuf, int buflen);    
    virtual bool AccessPathName(char *path, int mode= 0);
    virtual bool ChangeDirectory(char *path);
    virtual char *WorkingDirectory();
    
    virtual void exit(int code, bool mode= TRUE);
    virtual void GetTtyChars(char &backspace, char &rubout);
    virtual void wait(unsigned int);
    virtual void abort(int code= 0);

    //---- pseudo tty connection --------------------------------
    virtual class PttyConnection *MakePttyConnection(char *prog, char **args);

    //---- environment manipulation -----------------------------
    virtual void Setenv(char *name, char *value);
	// set environment variable name to value   
    virtual void Unsetenv(char *name);
	// remove environment variable
    virtual char *getenv(char *);

    //---- stack trace ------------------------------------------
    virtual void stacktrace(int level, bool abortafter);

    //---- handling of system events ----------------------------
    SysEvtHandler *AddFileInputHandler(SysEvtHandler *);
    SysEvtHandler *AddFileOutputHandler(SysEvtHandler *);
    SysEvtHandler *AddZombieHandler(SysEvtHandler *);
    SysEvtHandler *AddSignalHandler(SysEvtHandler*, bool sync= TRUE);
    SysEvtHandler *AddTimeoutHandler(SysEvtHandler *);
    void AddCleanupObject(Object*);
    void Remove();
};

extern System   *gSystem;
extern char     gBackspace;
extern char     gRubout;
extern char     *LibName;

inline void Wait(unsigned int duration)
    { gSystem->wait(duration); }

inline void Exit(int code, bool mode= TRUE)
    { gSystem->exit(code, mode); }

inline void Abort()
    { gSystem->abort(); }

inline void StackTrace(int level= 0, bool abortafter= TRUE)
    { gSystem->stacktrace(level, abortafter); }

inline char *Getenv(char *name)
    { return gSystem->getenv(name); }

#endif System_First
