//$StyledText,StyledTextIter,Style,StyleTable,TextRunArray$

#include "StyledText.h"
#include "RunArray.h"
#include "Mark.h"
#include "OrdCollection.h"
#include "Port.h"
#include "Error.h"
#include "ObjectTable.h"

//------ StyleTable ------------------------------------------------------

class StyleTable: public OrdCollection {
public:
    MetaDef(StyleTable);
    StyleTable():(cCollectionInitCap)
	{}
    Style *MakeStyle(GrFont ft, GrFace fc, int Size);
    Style *MakeStyle(Style *);
};

static StyleTable Styles;

MetaImpl0(StyleTable); 

Style *StyleTable::MakeStyle(GrFont ft, GrFace fc, int size)
{
    Iter next(this);
    register Style *sp;
    while (sp= (Style*)next()) {
	register Font *f= sp->fp;
	if (f->Fid() == ft && f->Size() == size && f->Face() == fc) 
	    return sp;
    }
    sp= new Style(0, new Font(ft, size, fc));
    Add(sp);
    return sp;
}

Style *StyleTable::MakeStyle(Style *asp)
{
    Iter next(this);
    Style *sp;
    while (sp= (Style*)next()) 
	if (sp->IsEqual(asp))
	    return sp;
    Add(asp);
    return asp;
}

//------ Style -----------------------------------------------------------

MetaImpl(Style, I_FT(fp)); 

Style::Style(int, FontPtr f)
{
    fp= f;    
}

Style::Style(FontPtr f)
{ 
    if (this == 0)
	this= Styles.MakeStyle(f->Fid(), f->Face(), f->Size());
    else {
	Error("Style::Style", "no static instances");
	this= 0;    // can't reach
    }
}

Style::Style(GrFont ft, GrFace fc, int size)
{ 
    if (this == 0)
	this= Styles.MakeStyle(ft, fc, size);
    else {
	Error("Style::Style", "no static instances");
	this= 0;    // can't reach
    }
}

Style::~Style()
{ 
    this= 0;
}

ostream &Style::PrintOn(ostream &s) 
{
    return s << fp SP;
}

istream &Style::ReadFrom(istream &s) 
{ 
    return s >> fp;
}

ostream &Style::DisplayOn(ostream &s) 
{
    return s << fp->AsString() NL;
}

bool Style::IsEqual(Object* op)
{
    if (!op->IsKindOf(Style))
	return FALSE;

    register Font *f= ((Style*)op)->fp;
    return (f->Fid() == fp->Fid() && f->Size() == fp->Size()
						&& f->Face() == fp->Face());
}

void Style::GetStyle(GrFont *fid, GrFace *face, int *size)
{
    *fid= fp->Fid();
    *face= fp->Face();
    *size= fp->Size();
}

ObjPtr Style::DeepClone()
{
    return this;
}

//---- class TextRunArray --------------------------------------------------

MetaImpl(TextRunArray, I_O(st)); 

TextRunArray::TextRunArray(StyledText *sp, int elements) : (elements)
{
    st= sp;
}

ObjPtr TextRunArray::RunAt(int i, int *start, int *end, int *size ,int *lenat)
{
    // requests for styles beyound the end of the text receive the
    // same style information as at the end of the text
    int at= i;
    i= min(i, Size()-1);
    if (i < 0) { // take styleHere into account
	*start= *end= at;
	*size= *lenat= 0;
	return st->GetCurrentStyle();
    }
    return RunArray::RunAt(i, start, end, size, lenat);
}

ostream &TextRunArray::PrintOn(ostream &s) 
{
    RunArray::PrintOn(s);
    return s << st SP;
}

istream &TextRunArray::ReadFrom(istream &s) 
{ 
    RunArray::ReadFrom(s);
    return s >> st;
}

Object *TextRunArray::ReadItem(istream& s)
{
    Style *sp;
    s >> sp;
    sp= Styles.MakeStyle(sp);
    return sp;    
}

//----- class StyledTextIter ---------------------------------------------

StyledTextIter::StyledTextIter(Text *s,int from = 0,int to = cMaxInt) : (s,from,to)
{
    if (!s->IsKindOf(StyledText))
	Error("StyledTextIter::StyledTextIter", "StyledText expected (%s received)",
							    s->ClassName());
    nextFontChange= ce;
    NextFontChange();
    if (sp == 0) {
	sp= new Style(gSysFont);
	Error("StyledTextIter::StyledTextIter", "style is nil");
    }
}

void StyledTextIter::Reset(Text *s,int from,int to)
{ 
    if (!s->IsKindOf(StyledText))
	Error("StyledTextIter::Reset", "StyledText expected (%s received)",
								s->ClassName());
    GapTextIter::Reset(s,from,to);
    nextFontChange= ce;
    NextFontChange();
    if (sp == 0) {
	sp= new Style(gSysFont);
	Error("StyledTextIter::Reset", "style is nil");
    }
}
int StyledTextIter::operator()()  
{
    return GapTextIter::operator()();
}

int StyledTextIter::Line(LineDesc *ld)
{
    register StyledText *pt = (StyledText*)ct;
    unget = ce;
    NextFontChange();
    if (ld) 
	ld->FromFont(sp->GetStyle());

    if (ce == upto) { // special case if last line is empty
	int last = pt->CharAt(ce-1);
	ce++;
	if (last == '\n' || last == '\r')
	    return upto;
    }
    if (ce > upto)
	return cEOT;

    while (ce < upto) {
	int ch = pt->CharAt(ce);
	if (NextFontChange() && ld)
	    ld->Max(sp->GetStyle());
	if (DoEscape(ce,ch)) 
	    pt->CalcEscape(ce,ld);
	ce++;
	if (ch == '\n' || ch == '\r')
	    break;
    }
    return ce;
}

int StyledTextIter::operator()(int *w, LineDesc *ld)  
{
    register StyledText *pt = (StyledText*)ct;

    *w = 0;
    if (ce == upto )
	return cEOT;

    NextFontChange();
    int ch = CharAt(ce);
    if (DoEscape(ce,ch))
	*w = pt->CalcEscape(ce,ld);
    else {      
	*w = sp->GetStyle()->Width(ch);
	if (ld)
	    ld->FromFont(sp->GetStyle());
    }
    ce++;
    return ch;
}

int StyledTextIter::Token(int *w,LineDesc *ld)  
{
    register StyledText *pt = (StyledText*)ct;

    unget= ce;
    *w = 0;
    NextFontChange();
    if (ld) 
	ld->FromFont(sp->GetStyle());
    if (ce >= upto)
	return cEOT;
    register int ch= CharAt(ce);
    if (Isspace(ch)) {
	*w = sp->GetStyle()->Width(ch);
	ce++;
	return (ch);
    } else if (DoEscape(ce,ch)) {
	*w = pt->CalcEscape(ce,ld);
	ce++;
	return (ch);
    }
    while (ce < upto && !Isspace(CharAt(ce)) && !DoEscape(ce,CharAt(ce))) {
	if (NextFontChange() && ld)
	    ld->Max(sp->GetStyle());
	ch = CharAt(ce++);
	*w += sp->GetStyle()->Width(ch);
    }
    return (ch);
}

bool StyledTextIter::NextFontChange ()
{ 
    if (ce == nextFontChange) {
	sp = (Style*)  ((StyledText*)ct)->styles->RunAt(ce,&start,&end,&size,&lenat);
	nextFontChange = ce + lenat;
	return TRUE;
    }
    else
	return FALSE;
}

//----- class StyledText --------------------------------------------------

MetaImpl(StyledText, (I_O(styles), I_O(sp), I_O(styleHere), I_B(applyStyleHere),
    I_C(escape), I_I(nextFontChange), I_I(start), I_I(end), I_I(size), I_I(lenat)));    

static TextChanges changeRec;

StyledText::StyledText()
{
    styles= new TextRunArray(this);
    Init(gSysFont, TRUE);
}

StyledText::StyledText(int size, FontPtr fd) : (size, fd)
{
    styles= new TextRunArray(this);
    Init(fd, TRUE);
}

StyledText::StyledText(byte *buf, int len, bool ic, FontPtr fd)  :(buf,len,ic,fd)
{
    styles= new TextRunArray(this);

    if (Size()) {
	styles->Insert(new Style(fd), 0, 0, Size());
	Init(gSysFont, FALSE);
    } else 
	Init(fd, TRUE);
}

StyledText::StyledText(TextRunArray *st, byte *buf, int len, bool ic)
								:(buf,len,ic)
{
    if (st->Size() != Size())
	Error("StyledText", "StyleArray does not correspond to the text");
    styles = st;
    Init(gSysFont, FALSE);
}

StyledText::StyledText(FontPtr fd, char* fmt, ...) : ()
{
    Style *st= new Style(fd);
    char *buf;

    va_list ap;
    va_start(ap, fmt);   
    buf= strvprintf(fmt, ap);
    va_end(ap);
    BuildStylesFromString(st, buf);
    Init(fd, FALSE);
    SafeDelete(buf);
}

StyledText::~StyledText()
{
    SafeDelete(styles);
    SafeDelete(styleHere);
}

void StyledText::InitNew()
{
    GapText::InitNew();
    styles= new TextRunArray(this);
    Init(gSysFont, TRUE);
}

void StyledText::Init(FontPtr fd, bool nullStyle)
{
    styleHere= new Style(fd);
    if (nullStyle)
	applyStyleHere= nullStyle;
    escape= '0';
}

void StyledText::ReplaceWithStr(byte *str,int len)
{
    GapText::ReplaceWithStr(str, len);
    SafeDelete(styles);
    Init(GetFont(), FALSE);   
}

void StyledText::Cut(int from,int to)
{
    int start, end, size, lenat;
    // if we delete a complete run |xxxx|XXXXXXX
    //                             ^         ^ we will set styleHere

    Style *sp= (Style*)styles->RunAt(from,&start,&end,&size,&lenat);
    applyStyleHere = FALSE;
    if (start == from && size <= to - from) {
	applyStyleHere = TRUE;
	styleHere = sp;
    }
    GapText::Cut(from,to);
    styles->Cut(from,to,FALSE);
}

void StyledText::Paste(Text* t,int from,int to)
{
    GapText::Paste(t,from,to);
    if (t->IsKindOf(StyledText)) { 
	StyledText *st = (StyledText *)t;
	if (styles)
	    styles->Paste(st->styles,from,to,FALSE);
	else
	    Error("Paste", "styles == 0");
    }  else {
	int start, end, size, lenat;
	if (from != to) {
	    Style *sp= (Style*)styles->RunAt(from,&start,&end,&size,&lenat);
	    if (start == from && size <= to - from) {
		applyStyleHere= TRUE;
		styleHere= sp;
	    }
	    styles->Cut(from,to,FALSE);
	}
	if (applyStyleHere && t->Size()) { // insert new run with styleHere
	    styles->Insert (new Style(styleHere->GetStyle()),from,from,t->Size());
	    applyStyleHere= FALSE;
	}
	else 
	    styles->ChangeRunSize(from, t->Size());
    }       
}

void StyledText::Copy(Text* save,int from, int to)
{
    if (!CheckRange(Size(),from,to) || save == 0)
	return;
    GapText::Copy(save,from,to);

    if (!save->IsKindOf(StyledText)) 
	return;

    StyledText *st = (StyledText *)save;
    if (applyStyleHere && Size() == 0) {
	st->applyStyleHere = TRUE;
	st->styleHere = styleHere;
    } else 
	styles->Copy(st->styles,from,to);
}

TextPtr StyledText::Save(int from, int to)
{
    if (!CheckRange(Size(),from,to))
	return 0;

    StyledText* t= new StyledText(to-from);
    Copy(t, from, to);
    return t;
}

void StyledText::SetFStringVL(char *fmt, va_list ap)
{
    char *buf= strvprintf(fmt, ap);
    FontPtr fd= GetFont();
    Empty();
    Style *st= GetStyle(0);
    BuildStylesFromString(st, buf);
    SafeDelete(buf);
    Init(fd, FALSE);
}

void StyledText::Insert(byte c, int from,int to)
{
    GapText::Insert(c,from,to);
    if (applyStyleHere)  { // insert new run with styleHere 
	styles->Insert(new Style(styleHere->GetStyle()), from, from, 1);
	applyStyleHere= FALSE;
    } else 
	styles->ChangeRunSize(from,1);
}

Text *StyledText::GetScratchText(int size)
{
    return new GapText(size);
}

void StyledText::DrawText(int from, int to, Rectangle clip)
{
    if (!CheckRange(Size(), from ,to))
	return;
    int ch;
    nextFontChange= from;
    NextFontChange(from);
    GrSetFont(sp->GetStyle());
    Point start= GrGetTextPos();
    for (int i= from; i < to; i++) {
	if (NextFontChange(i)) 
	    GrSetFont(sp->GetStyle());
	if ((ch= CharAt(i)) == '\t') 
	    GrTextAdvance(Tabulate(GrGetTextPos().x-start.x)); 
	else if (DoEscape(i,ch)) {
	    DrawEscape(i,clip);
	    NextFontChange(nextFontChange = i); // reload current style
	    GrSetFont(sp->GetStyle());
	} else
	    GrDrawChar(ch);
    }
}

void StyledText::DrawTextJust (int from, int to, int w, Point start, Rectangle r)
{
    register byte ch;
    int ntab, longBlanks, addSpace, seenTabs;

    if (!CheckRange(Size(), from, to))
	return;

    ntab= longBlanks= seenTabs = 0;

    CalcIws(w, from, &to, &addSpace, &longBlanks, &ntab);    
    GrTextMoveto(start);
    NextFontChange(nextFontChange= from);
    GrSetFont(sp->GetStyle());
    for (int i = from, nBlanks = 0; i < to; i++) {
	if (NextFontChange(i)) 
	    GrSetFont(sp->GetStyle());
	switch (ch = CharAt(i)) {
	case '\t':
	    GrTextAdvance(Tabulate(GrGetTextPos().x-start.x));
	    seenTabs++;
	    break;
	case ' ':
	    GrTextAdvance(sp->GetStyle()->Width(' '));
	    if (seenTabs == ntab) {
		GrTextAdvance(addSpace);
		if (nBlanks < longBlanks)
		    GrTextAdvance(1);
		nBlanks++;
	    }
	    break;
	default:
	    if (DoEscape(i,ch)) {
		DrawEscape(i,r);
		NextFontChange(nextFontChange = i); // reload current style
		GrSetFont(sp->GetStyle());
	    }
	    else
		GrDrawChar (CharAt(i));
	}
    }
}

void StyledText::JustifiedMap(int from, int to, int w, int stopAt, int posX, 
							   int *charPos, int *x)
{
    register byte ch;
    int ntab, longBlanks, addSpace, seenTabs, cx, wx;

    if (!CheckRange(Size(), from, to))
	return;

    wx= cx= ntab= longBlanks= addSpace= seenTabs= 0;

    CalcIws(w, from, &to, &addSpace, &longBlanks, &ntab);    
    nextFontChange = from;

    for (int i = from, nBlanks = 0; i < to && i < stopAt; i++) {
	NextFontChange(i); 
	switch (ch = CharAt(i)) {
	case '\t':
	    wx = Tabulate(cx);
	    seenTabs++;
	    break;
	case ' ':
	    wx= sp->GetStyle()->Width(' ');
	    if (seenTabs == ntab) {
		wx += addSpace;
		if (nBlanks < longBlanks)
		    wx++; 
		nBlanks++;
	    }
	    break;
	default:
	    if (DoEscape(i,ch)) 
		wx = CalcEscape(i);
	    else 
		wx = sp->GetStyle()->Width(ch); 
	    break;
	}
	if (cx + (wx/2) > posX)
	    break;
	cx += wx;
    }
    if (x)
	*x = cx;
    if (charPos)
	*charPos= i;
}

int StyledText::TextWidth(int from, int to)
{
    if (!CheckRange(Size(), from ,to))
	return 0;

    int w= 0, ch;

    nextFontChange= from;
    NextFontChange(from);

    for (int i= from; i < to; i++) {
	NextFontChange(i);
	ch= CharAt(i);
	if (ch == '\t') 
	    w+= Tabulate(w); 
	else {
	    if (DoEscape(i,ch))
		w+= CalcEscape(i);
	    else 
		w+= sp->GetStyle()->Width (ch);
	}
    }
    return w;
}

TextIter *StyledText::GetIterator(int from=0, int to = cMaxInt)
{ 
    return new StyledTextIter(this, from, to); 
}

ostream &StyledText::PrintOn(ostream &s) 
{
    GapText::PrintOn(s);
    return s << styles SP << (int)escape SP;
}

istream &StyledText::ReadFrom(istream &s) 
{
    int esc;

    GapText::ReadFrom(s);
    s >> styles >> esc;
    escape= esc;
    if (Size())
	applyStyleHere= FALSE;
    else
	applyStyleHere= TRUE;
    return s;
}

void StyledText::SetStyle(StChangeStyle mode, int from, int to, StyleSpec st)
{
    int start, end, size;

    if (from == to) {
	Style *sp= applyStyleHere ? styleHere : GetStyle(max(0, from-1));
	applyStyleHere= TRUE;
	styleHere= ApplySpec(sp, st, mode);
    } else {
	Style **spp;
	RunArray newStyles(2); 
	styles->Copy(&newStyles, from, to);
	RunArrayIter next(&newStyles);
	while (spp= (Style**)next.RunPtr(&start, &end, &size)) 
	    *spp= ApplySpec(*spp, st, mode);
	styles->Paste(&newStyles, from, to, FALSE);
	GetMarkList()->RangeChanged(from, to - from);
	changeRec.Set(eTextChangedRange, from, to);
	ChangedWhat(&changeRec);     
    }
}

Style *StyledText::ApplySpec(Style *ostyle, StyleSpec st, StChangeStyle mode)
{
    GrFont fid;
    GrFace face;
    int    fsize;

    ostyle->GetStyle (&fid, &face, &fsize);
    if ((mode & eStFont) == eStFont)
	fid= st.font;
    if ((mode & eStFace) == eStFace)
	if (st.face == eFacePlain)
	    face= st.face;
	else {
	    if (st.xor)
		face^= st.face;
	    else
		face|= st.face;
	}
    if ((mode & eStSize) == eStSize)
	fsize= st.size;
    if ((mode & eStAddSize) == eStAddSize)
	fsize+= st.size;
    return Styles.MakeStyle(fid, face, fsize);
}

void StyledText::BuildStylesFromString(Style *st, char *buf)
{
    GrFace face= eFacePlain;
    GrFont fid; 
    int run= 0, size;
    char *p;

    styles = new TextRunArray(this);
    for (p= buf; *p; *p++) {
	if (*p == '@') {
	    if (*(p+1) && index("OUSBIP",*(p+1))) {
	       if (run) {
		   st->GetStyle(&fid, &face, &size);
		   int sz= styles->Size();
		   styles->Insert(st->DeepClone(), sz, sz, run);
		   run = 0;
		}
		p++;
		switch (*p) {
		case 'P':
		    face= eFacePlain;
		    break;
		case 'O':
		    face^= eFaceOutline;
		    break;
		case 'S':
		    face^= eFaceShadow;
		    break;
		case 'U':
		    face^= eFaceUnderline;
		    break;
		case 'B':
		    face^= eFaceBold;
		    break;
		case 'I':
		    face^= eFaceItalic;
		    break;
		}
		st= new Style(fid, face, size);
	     } else { 
		 p++;
		 if (!*p)
		     break;
		 GapText::Insert(*p, Size(), Size());
		 run++;
	     }
	} else {
	    GapText::Insert(*p, Size(), Size());
	    run++;
	}
    }
    if (run) {
	int sz= styles->Size();
	styles->Insert(st->DeepClone(), sz, sz, run);
    }
}

Style *StyledText::GetStyle(int at)
{
    int start, end, size, lenat;
    return (Style*)styles->RunAt(at,&start,&end,&size,&lenat);
}

void StyledText::SetFont(FontPtr fp)
{
    SetStyle(eStAll, 0, Size(), StyleSpec(fp));
}

FontPtr StyledText::GetFont(int at)
{
    Style *st = GetStyle(at);
    return st->GetStyle();
}

void StyledText::ReplaceStyles(RunArray *st, int from, int to)
{
    if (to-from != st->Size())
	Error("ReplaceStyle", "RunArray does not match text");
    styles->Paste(st,from,to,FALSE);
    GetMarkList()->RangeChanged(from,to - from);
    changeRec.Set(eTextChangedRange, from, to);
    ChangedWhat(&changeRec);     
}

void StyledText::CopyStyles(RunArray *st,int from, int to)
{
    styles->Copy(st,from,to);
}

void StyledText::ResetCurrentStyle()
{
    if (Size())
	applyStyleHere = FALSE;
}

const Style *StyledText::GetCurrentStyle()
{
    if (applyStyleHere)
	return styleHere;
    return 0;    
}

TextRunArray *StyledText::GetStyles()
{
    return styles; 
}

TextRunArray *StyledText::SetStyles(TextRunArray *st)
{
    TextRunArray *old = styles;
    styles = st;
    //changeRec.Set(eTextChangedRange, 0, Size());
    //ChangedWhat(&changeRec);     
    return old;
}

void StyledText::SetEscapeChar(byte ch)
{
    escape = ch;
}

byte StyledText::GetEscapeChar()
{
    return escape;
}

int StyledText::CalcEscape (int, LineDesc*)
{
    return 0;
}

void StyledText::DrawEscape (int,Rectangle)
{
}

bool StyledText::IsEscape (int)
{
    return FALSE;
}

ONENTRY(StyledText)
{
    ObjectTableAddRoot(&Styles);
}

