//$TextShapeSketcher,TextShapeStretcher,TextShape$
#include "TextShape.h"
#include "BoxShape.h"
#include "DrawView.h"
#include "StyledText.h"
#include "VObjectText.h"
#include "VObjectTextView.h"

const int cMinParagraph = 30;       // minimum size of a p

//---- TextShape ---------------------------------------------------------------

static short TextImage[]= {
#   include  "images/TextShape.im"
};

MetaImpl(TextShape, (I_O(to), I_O(attachedTo), I_B(captured)));

TextShape::TextShape()
{
    attachedTo= 0;
    to= 0;
}

TextShape::~TextShape()
{
    if (attachedTo)
	attachedTo->RemoveDependent(this);
    SafeDelete(to);
}

short *TextShape::GetImage()
{
    return TextImage;
}

bool TextShape::IsGarbage()
{   
    if (((DrawView*)GetView())->GetActiveText() == this)
	return FALSE;
    return Shape::IsGarbage() || to->Empty();
}

GrCursor TextShape::SketchCursor()
{
    return eCrsIBeam;
}

void TextShape::Init(Point p1, Point p2)
{
    Shape::Init(p1, p2);
    pattern= ePatNone;
    captured= bbox.Width() <= cMinParagraph;
    int width= captured ? cFit : bbox.extent.x;
    to= new VObjectTextView(GetView(), Rectangle(bbox.origin, Point(width,cFit)),
	    new VObjectText(16, new Font(eFontHelvetica, 14)),
	    eLeft, eOne, TRUE, eVObjDefault, gPoint0);
    bbox.extent.y= to->Height();
    to->AddDependent(this); // register myself as dependent of a text obj
}

void TextShape::Draw(Rectangle r, Point p)
{
    GrFillRect(bbox+p);
    to->SetOrigin(bbox.origin+p);
    GrSetTextPattern(penpattern);
    to->Draw(r);
}

void TextShape::SetFont(RunArray *newfont)
{
    StyledText *st= (StyledText*) to->GetText();
    st->ReplaceStyles(newfont, 0, st->Size());
    to->Reformat();
    Changed();
}

void TextShape::ApplyFont(StChangeStyle st, StyleSpec sp)
{
    StyledText *text= (StyledText*) to->GetText();
    text->SetStyle(st, 0, text->Size(), sp);    
    to->Reformat();
    Changed();
}

RunArray *TextShape::GetFont()
{
    RunArray *ra= new RunArray;
    StyledText *st= (StyledText*) to->GetText();
    st->CopyStyles(ra, 0, st->Size());
    return ra;
}

void TextShape::SetProperty(ShapeProperties what, int p)
{
    switch (what) {
    case eShapeAdjust:
	to->SetJust(eTextJust(p));
	break;
    case eShapeSpacing:
	to->SetSpacing(eSpacing(p));
	break;
    default:
	Shape::SetProperty(what, p);
	return;
    }
    Changed();
}

int TextShape::GetProperty(ShapeProperties what)
{
    switch (what) {
    case eShapeAdjust:
	return int(to->GetJust());
    case eShapeSpacing:
	return int(to->GetSpacing());
    default:
	return Shape::GetProperty(what);
    }
}

Rectangle TextShape::InvalRect()
{
    return Shape::InvalRect().Expand(2);
}

void TextShape::InvalDiff(Rectangle r2)
{
    if (GetView()) {
	Rectangle r[4];
	int n= Difference(r, r2, bbox);
	
	for (int i= 0; i < n; i++)
	    GetView()->InvalidateRect(r[i]);
    }
}

void TextShape::Highlight(HighlightState h)
{
    if (GetView()) {
	// if there is no active Text highlight the object as a shape
	if (((DrawView*)GetView())->GetActiveText() != this) {
	    if (captured)
		GrSetPattern(ePatGrey50);
	    Shape::Highlight(h);
	    if (captured)
		GrSetPattern(ePatBlack);
	} else if (to)
	    to->DoHighlightSelection(h);
    }
}

void TextShape::SetSpan(Rectangle r)
{
    if (captured)
	return;    
    bbox= NormRect(r.origin, r.extent);
    to->SetExtent(Point(bbox.extent.x, cFit));
    Changed();
}

void TextShape::MakeDependentOn(Shape *s)
{
    s->AddDependent(this);
    attachedTo= s;    
}

void TextShape::DoUpdate(Object *op, void *what)
{
    if (to == 0)
	return;
    if (op == to) { // text obj changed 
	switch ((int) what) {
	case eExtent:
	    if (pattern != ePatNone)
		InvalDiff(Rectangle(bbox.origin, to->GetExtent()));
	    bbox.extent= to->GetExtent();
	    break;
	    
	case eOrigin:
	    if (pattern != ePatNone)
		InvalDiff(Rectangle(bbox.origin, to->GetOrigin()));
	    bbox.origin= to->GetOrigin();
	    break;
	}
    } else {   
	Rectangle r= Guard(op,Shape)->GetTextRect();
	SetSpan(Rectangle(r.NW(), r.SE()));
    }
}

void TextShape::SetView(View *vp)
{
    Shape::SetView(vp);
    if (to)
	to->SetView(vp);
}

ostream& TextShape::PrintOn(ostream& s)
{
    Shape::PrintOn(s);
    return s << attachedTo SP << to SP << captured SP;
}

istream& TextShape::ReadFrom(istream& s)
{
    Shape::ReadFrom(s);
    return s >> attachedTo >> to >> Bool(captured);
}

ShapeSketcher *TextShape::NewSketcher(class DrawView *dv, SketchModes m)
{
    return new TextShapeSketcher(dv, this, m); 
}

ShapeStretcher *TextShape::NewStretcher(class DrawView *dv, int handle)
{
    if (captured)
	return (ShapeStretcher*) gNoChanges;
    return new TextShapeStretcher(dv, this, handle);
}

//---- TextShapeSketcher -------------------------------------------------------

TextShapeSketcher::TextShapeSketcher(DrawView *dv, Shape *pro, SketchModes m)
								  : (dv, pro, m) 
{
    Font *f= new Font(eFontTimes, 14);
    lineHeight= TextViewLineHeight(new Font(eFontTimes, 14)); 
    baseHeight= f->Ascender(); 
}

Command *TextShapeSketcher::TrackMouse(TrackPhase tp, Point ap, Point pp, Point np)
{
    ShapeSketcher::TrackMouse(tp, ap, pp, np);
    if (tp == eTrackRelease) {
	newshape= (Shape*)proto->Clone();
	newshape->SetView(view);
	newshape->Init(ap, np);
	view->Insert(newshape);
	view->SetActiveText((TextShape*)newshape);
	return gNoChanges;
    }
    return this;
}

void TextShapeSketcher::TrackConstrain(Point ap, Point pp, Point *np)
{
    if (ap == pp) {     // constrain left/top corner
	ShapeSketcher::TrackConstrain(ap, pp, np);
	np->y-= baseHeight;
    } else              // constrain right/bottom corner
	np->y= ap.y + lineHeight;
}

//---- TextShapeStretcher ------------------------------------------------------

TextShapeStretcher::TextShapeStretcher(DrawView *dv, Shape *p, int h) : (dv, p, h) 
{
}

void TextShapeStretcher::TrackConstrain(Point, Point, Point *np)
{
    np->x= max(sp->bbox.origin.x + cMinParagraph, np->x);
}
