//$ShapeSelector,GroupCommand,UngroupCommand,SCutCopyCommand$
//$SPasteCommand,DupCommand,FrontBackCommand,ConnectCommand,PropertyCommand$
//$FontCommand,CursorMoveCommand,ShapeDragger$

#include "Commands.h"
#include "Error.h"
#include "Connection.h"
#include "DrawView.h"
#include "Group.h"
#include "ObjList.h"
#include "CmdNo.h"
#include "StyledText.h"
#include "ObjArray.h"
#include "TextShape.h"

int _Commands__ctor;    // hack for dyn linking

//---- Group Command ------------------------------------------------------------

GroupCommand::GroupCommand(DrawView *dv) : (dv, cGROUP, "group") 
{
    group= new Group(view->GetDeepCopyOfSelection());
}     

void GroupCommand::SaveDoIt()
{
    view->SetDeleted(saveselection, TRUE);
    view->Insert(group);
}

void GroupCommand::RestoreUndoIt()
{
    view->SetDeleted(saveselection, FALSE);
    group->SetDeleted(TRUE);
}

void GroupCommand::RestoreRedoIt()
{
    view->SetDeleted(saveselection, TRUE);
    group->SetDeleted(FALSE);
}

//---- Ungroup Command ------------------------------------------------------------

UngroupCommand::UngroupCommand(DrawView *dv, Group *gp) : (dv, cUNGROUP, "ungroup") 
{
    group= gp;
    newshapes= group->Ungroup();
}

void UngroupCommand::~UngroupCommand()
{
    SafeDelete(newshapes);
}

void UngroupCommand::SaveDoIt()
{
    group->SetDeleted(TRUE);
    view->InsertShapes(newshapes);
}

void UngroupCommand::RestoreUndoIt()
{
    group->SetDeleted(FALSE);
    view->SetDeleted(newshapes, TRUE);
}

void UngroupCommand::RestoreRedoIt()
{
    group->SetDeleted(TRUE);
    view->SetDeleted(newshapes, FALSE);
}

//---- SCutCopy Command --------------------------------------------------------

SCutCopyCommand::SCutCopyCommand(DrawView *dv, int cmd, char *name) : (dv, cmd, 0) 
{
    if (name == 0) {
	if (cmd == cCUT)
	    name= "cut";
	else if (cmd == cCOPY)
	    name= "copy";
    }
    SetName(name);
    if (cmd == cCOPY)
	ResetFlag(eCmdCausesChange);
}

void SCutCopyCommand::SaveDoIt()
{
    if (GetId() != cCOPY)
	view->SetDeleted(saveselection, TRUE);
}

void SCutCopyCommand::RestoreUndoIt()
{
    if (GetId() != cCOPY)
	view->SetDeleted(saveselection, FALSE);
}

//---- SPaste Command -----------------------------------------------------------

SPasteCommand::SPasteCommand(DrawView *dv, ObjList *ns, Point p) : (dv, cPASTE, "paste") 
{
    Rectangle bbox;
    Iter next(ns);
    register Shape *s;
    
    newshapes= ns;
    
    while (s= (Shape*)next()) {
	s->SetView(view);
	bbox.Merge(s->bbox);
    }
    p-= bbox.origin;
    newshapes->ForEach(Shape,Moveby)(p);
}

SPasteCommand::SPasteCommand(DrawView *dv, Shape *ns, Point p) : (dv, cPASTE, "paste") 
{
    newshapes= new ObjList;
    newshapes->Add(ns);
    newshapes->ForEach(Shape,SetView)(view);
    ns->Moveby(p - ns->bbox.origin);
}

SPasteCommand::~SPasteCommand()
{
    SafeDelete(newshapes);
}

void SPasteCommand::SaveDoIt()
{
    view->InsertShapes(newshapes);
}

void SPasteCommand::RestoreUndoIt()
{
    view->SetDeleted(newshapes, TRUE);
}

void SPasteCommand::RestoreRedoIt()
{
    view->SetDeleted(newshapes, FALSE);
}

//---- Duplicate Command --------------------------------------------------------

DupCommand::DupCommand(DrawView *dv) : (dv, cDUP, "dup") 
{
    dups= view->GetDeepCopyOfSelection();
    dups->ForEach(Shape,SetView)(view);
    ddelta= 16;
}
	
DupCommand::~DupCommand()
{
    SafeDelete(dups);
}

void DupCommand::Done(Command *nextcmd)
{
    if (nextcmd->GetId() == cDUP)
	((DupCommand*)nextcmd)->ddelta= ddelta;
    if (nextcmd->GetId() == cDRAG)
	((CursorMoveCommand*)nextcmd)->dragDelta= ddelta;
}

void DupCommand::SaveDoIt()
{
    dups->ForEach(Shape,Moveby)(ddelta);
    view->InsertShapes(dups);
}

void DupCommand::RestoreUndoIt()
{
    view->SetDeleted(dups, TRUE);
}

void DupCommand::RestoreRedoIt()
{
    view->SetDeleted(dups, FALSE);
}

//---- FrontBack Command --------------------------------------------------------

FrontBackCommand::FrontBackCommand(DrawView *dv, int cmd, char *s) : (dv, cmd, s)
{
    newshapes= view->GetDeepCopyOfSelection();
    newshapes->ForEach(Shape,SetView)(view);
}

FrontBackCommand::~FrontBackCommand()
{
    SafeDelete(newshapes);
}

void FrontBackCommand::SaveDoIt()
{
    view->InsertShapes(newshapes, GetId() == cTOFRONT);
    view->SetDeleted(saveselection, TRUE);
}

void FrontBackCommand::RestoreUndoIt()
{
    view->SetDeleted(saveselection, FALSE);
    view->SetDeleted(newshapes, TRUE);
}

void FrontBackCommand::RestoreRedoIt()
{
    view->SetDeleted(saveselection, TRUE);
    view->SetDeleted(newshapes, FALSE);
}

//---- ConnectCommand -----------------------------------------------------------

ConnectCommand::ConnectCommand(DrawView *dv) : (dv, cCONNECT, "connect") 
{
    Iterator *next= view->GetSelectionIter();
    Object *p1, *p2;
    
    connections= new ObjList;
    p1= (*next)();
    while (p1 && (p2= (*next)()) ) {
	connections->Insert(new Connection(view, (Shape*) p1, (Shape*) p2));
	p1= p2;
    }
    SafeDelete(next);
}

ConnectCommand::~ConnectCommand()
{
    SafeDelete(connections);
}

void ConnectCommand::SaveDoIt()
{
    view->InsertShapes(connections);
}

void ConnectCommand::RestoreUndoIt()
{
    view->SetDeleted(connections, TRUE);
}

void ConnectCommand::RestoreRedoIt()
{
    view->SetDeleted(connections, FALSE);
}

//---- CursorMoveCommand Methods ------------------------------------------------

CursorMoveCommand::CursorMoveCommand(DrawView *dv, Point d, bool s) : (dv, cDRAG, "move") 
{
    undoDelta= gPoint0;
    dragDelta= gPoint16;
    delta= d;
    scroll= s;
}

void CursorMoveCommand::Done(Command *nextcmd)
{
    if (nextcmd->GetId() == cDRAG) {
	((CursorMoveCommand*)nextcmd)->undoDelta= undoDelta;
	((CursorMoveCommand*)nextcmd)->dragDelta= dragDelta;
    }
    if (nextcmd->GetId() == cDUP)
	((DupCommand*)nextcmd)->ddelta= dragDelta;
}

void CursorMoveCommand::SaveDoIt()
{
    if (scroll)
	view->ShowSelection();
    view->Invalidate(saveselection);
    saveselection->ForEach(Shape,Moveby)(delta);
    view->Invalidate(saveselection);
    undoDelta+= delta;
    dragDelta-= delta;
}
	
void CursorMoveCommand::RestoreUndoIt()
{
    view->Invalidate(saveselection);
    saveselection->ForEach(Shape,Moveby)(-undoDelta);
    view->Invalidate(saveselection);
}

void CursorMoveCommand::RestoreRedoIt()
{
    view->Invalidate(saveselection);
    saveselection->ForEach(Shape,Moveby)(undoDelta);
    view->Invalidate(saveselection);
}

//---- Dragger Methods ----------------------------------------------------------

ShapeDragger::ShapeDragger(DrawView *dv, Shape *s) : (dv, gPoint0, FALSE) 
{
    sp= s;
    onlyone= view->Selected() == 1;
    bbox= view->BoundingBox();
    span= sp->GetSpan();
}
       
void ShapeDragger::TrackFeedback(Point, Point, bool)
{
    if (moved) {
	sp->Outline(span.origin+delta, span.extent+delta);
	if (! onlyone) {
	    GrSetPenPattern(ePatGrey50);
	    GrStrokeRect(Rectangle(bbox.origin+delta, bbox.extent));
	    GrSetPenPattern(ePatBlack);
	}
    }
}

void ShapeDragger::TrackConstrain(Point ap, Point pp, Point *np)
{
    Rectangle r= bbox;
    r.origin+= pp-ap;
    *np+= r.AmountToTranslateWithin(view->GetExtent());
    DrawCommand::TrackConstrain(ap, pp, np);
}

Command *ShapeDragger::TrackMouse(TrackPhase tp, Point ap, Point pp, Point np)
{
    DrawCommand::TrackMouse(tp, ap, pp, np);
    view->ShowInfo(tp, "x: %d y: %d", bbox.origin.x+delta.x, bbox.origin.y+ delta.y);
    switch (tp) {
    case eTrackMove:
	if (moved)
	    GrSetCursor(eCrsMoveHand);
	break;
    case eTrackRelease:
	if (delta == gPoint0)
	    return gNoChanges;
	undoDelta= delta;
	break;
    }
    return this;
}

//---- Shape Selector Methods ---------------------------------------------------

void ShapeSelector::TrackFeedback(Point anchorPoint, Point nextpoint, bool)
{
    GrSetPenPattern(ePatGrey50);
    GrStrokeRect(NormRect(anchorPoint, nextpoint));
}

Command *ShapeSelector::TrackMouse(TrackPhase tp, Point ap, Point pp, Point np)
{
    DrawCommand::TrackMouse(tp, ap, pp, np);
    switch (tp) {
    case eTrackPress:
	GrSetCursor(eCrsHand); 
	break;
    case eTrackRelease:
	view->SelectInRect(NormRect(ap, np));
	return gNoChanges;
    }
    return this;
}

//---- Property Command --------------------------------------------------------

PropertyCommand::PropertyCommand(DrawView *dv, ShapeProperties w, int prop,
				char *cmd) : (dv, w+1, cmd)
{
    oldprop= new int[cnt];
    newprop= prop;
    what= w;
}

PropertyCommand::~PropertyCommand()
{
    SafeDelete(oldprop);
}

void PropertyCommand::SetProperty(Shape *p, int)
{
    p->SetProperty(what, newprop);
}

void PropertyCommand::SaveProperty(Shape *p, int i)
{
    oldprop[i]= p->GetProperty(what);
}

void PropertyCommand::RestoreProperty(Shape *p, int i)
{
    p->SetProperty(what, oldprop[i]);
}

//---- Font Command -------------------------------------------------------------

FontCommand::FontCommand(DrawView *dv, int m, int v, char *n) : (dv, m, n)
{
    oldfont= new ObjArray(cnt);
    newval= v;
    mode= m;
}

FontCommand::~FontCommand()
{
    Object *op;
    for (int i= 0; i < cnt; i++) {
	op= oldfont->At(i);
	SafeDelete(op);
    }
    SafeDelete(oldfont);
}

void FontCommand::SetProperty(Shape *p, int)
{
    StChangeStyle st;
    StyleSpec sp;
    
    switch (mode) {
    case cTEXTFONT:
	st= eStFont;
	sp= StyleSpec(newval,0,0);
	break;
    case cTEXTSIZE:
	st= eStSize;
	sp= StyleSpec(0,0,newval);
	break;
    case cTEXTFACE:
	st= eStFace;
	sp= StyleSpec(0,newval,0);
	break;
    }
    p->ApplyFont(st, sp);
}

void FontCommand::SaveProperty(Shape *p, int i)
{
    (*oldfont)[i]= p->GetFont();
}

void FontCommand::RestoreProperty(Shape *p, int i)
{
    p->SetFont((RunArray*)oldfont->At(i));
}
