/*
 * list.c  - borrowed from Murray Goldberg
 * 
 * the following routines implement the list data structure 
 *
 * Copyright (C) 1991, 1992 the University of British Columbia
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 */

#include "snacc_config.h"
#include "mem.h"
#include "list.h"

/* remove the entire list and all its elements */
void
GListFree PARAMS((list),
GLIST* list )
{
    GLISTNODE *node, *next;

    node = list->first;
    while( node )
	{
	next = node->next;
	Free( node );
	node = next;
	}

    Free( list );
}  /* GListFree */



/*
 * this routine removes the current node from the list. After removal the
 * current pointer will point to the next node in line, or NULL if the
 * removed item was at the tail of the list.
 */
void
GListRemove PARAMS( (list),
GLIST* list )
{
    GLISTNODE* node;

    if( list->curr )
	{
	if( list->curr->next )
	    list->curr->next->prev = list->curr->prev;
	else
	    list->last = list->curr->prev;

	if( list->curr->prev )
	    list->curr->prev->next = list->curr->next;
	else
	    list->first = list->curr->next;

	node       = list->curr;

	list->curr = list->curr->next;
	list->count--;

	Free( node );
	}
}

/*
 * this creates a new node after the current node and returns the
 * address of the memory allocated for data. The current pointer is changed
 * to point to the newly added node in the list. If the current pointer is
 * initially off the list then this operation fails.
 */
void*
GListAdd PARAMS((list),
GLIST* list )
{
    GLISTNODE* newNode;
    void*      dataAddr;

    if( list->curr )
        {
	newNode  = (GLISTNODE *) Malloc( sizeof(GLISTNODE) + list->dataSize );
	dataAddr = (void *) &(newNode->data);

	newNode->next = list->curr->next;
	newNode->prev = list->curr;
	if( list->curr->next )
	    list->curr->next->prev = newNode;
	else
	    list->last = newNode;
	list->curr->next = newNode;

	list->curr = newNode;
	list->count++;
	}

    else
	dataAddr = NULL;

    return( dataAddr );
}

/*
 * this creates a new node before the current node and returns the
 * address of the memory allocated for data. The current pointer is changed 
 * to point to the newly added node in the list. If the current pointer is
 * initially off the list then this operation fails.
 */
void*
GListInsert PARAMS( (list),
GLIST* list )
{
    GLISTNODE* newNode;
    void*      dataAddr;

    if( list->curr )
        {
	newNode  = (GLISTNODE *) Malloc( sizeof(GLISTNODE) + list->dataSize );
	dataAddr = (void *) &(newNode->data);

	newNode->next = list->curr;
	newNode->prev = list->curr->prev;
	if( list->curr->prev )
	    list->curr->prev->next = newNode;
	else
	    list->first  = newNode;
	list->curr->prev = newNode;

	list->curr = newNode;
	list->count++;
	}

    else
	dataAddr = NULL;

    return( dataAddr );
}



GLIST*
GListNew PARAMS( (dataSize),
int dataSize )
{
    GLIST* list;

    list = (GLIST *) Malloc( sizeof(GLIST) );
    list->first = list->last = list->curr = NULL;
    list->count = 0;
    list->dataSize = dataSize;

    return( list );
}

/*
 * backs up the current pointer by one and returns the data address of the new
 * current node. If the current pointer is off the list, the new current node
 * will be the last node of the list (unless the list is empty).
 */
void*
GListPrev PARAMS( (list),
GLIST* list )
{
    void* retVal;

    if( list->curr == NULL )
	list->curr = list->last;
    else
	list->curr = list->curr->prev;

    if( list->curr == NULL )
	retVal = NULL;
    else
	retVal = (void *) &(list->curr->data);

    return( retVal );
}

/*
 * advances the current pointer by one and returns the data address of the new
 * current node. If the current pointer is off the list, the new current node
 * will be the first node of the list (unless the list is empty).
 */
void*
GListNext PARAMS( (list),
GLIST* list )
{
    void* retVal;

    if( list->curr == NULL )
	list->curr = list->first;
    else
	list->curr = list->curr->next;

    if( list->curr == NULL )
	retVal = NULL;
    else
	retVal = (void *) &(list->curr->data);

    return( retVal );
}

/*
 * returns the data address of the last node (if there is one) and sets the
 * current pointer to this node.
 */
void*
GListLast PARAMS((list),
GLIST* list )
{
    void* retVal;

    list->curr = list->last;

    if( list->curr == NULL )
	retVal = NULL;
    else
	retVal = (void *) &(list->curr->data);

    return( retVal );
}

/*
 * returns the data address of the first node (if there is one) and sets the
 * current pointer to this node.
 */
void*
GListFirst PARAMS( (list),
GLIST* list )
{
    void* retVal;

    list->curr = list->first;

    if( list->curr == NULL )
	retVal = NULL;
    else
	retVal = (void *) &(list->curr->data);

    return( retVal );
}

/*
 * this creates a new node at the beginning of the list and returns the
 * address of the memory allocated for data. The current pointer is changed 
 * to point to the newly added node in the list.
 */
void*
GListPrepend PARAMS( (list),
GLIST* list )
{
    GLISTNODE* newNode;
    void*      dataAddr;

    newNode  = (GLISTNODE *) Malloc( sizeof(GLISTNODE) + list->dataSize );
    dataAddr = (void *) &(newNode->data);

    newNode->prev = NULL;

    if( list->first == NULL )
	{
	newNode->next = NULL;
	list->first   = list->last = newNode;
	}
    else
	{
	newNode->next     = list->first;
	list->first->prev = newNode;
	list->first       = newNode;
	}

    list->curr = newNode;
    list->count++;

    return( dataAddr );
}

/*
 * this creates a new node at the end of the list and returns the
 * address of the memory allocated for data. The current pointer is changed
 * to point to the newly added node in the list.
 */
void*
GListAppend PARAMS( (list),
GLIST* list )
{
    GLISTNODE* newNode;
    void*      dataAddr;

    newNode  = (GLISTNODE *) Malloc( sizeof(GLISTNODE) + list->dataSize );
    dataAddr = (void *) &(newNode->data);

    newNode->next = NULL;

    if( list->last == NULL )
        {
        newNode->prev = NULL;
        list->first   = list->last = newNode;
        }
    else
        {
        newNode->prev     = list->last;
        list->last->next  = newNode;
        list->last        = newNode;
        }

    list->curr = newNode;
    list->count++;

    return( dataAddr );
}

void*
GListCurr PARAMS( (list),
GLIST* list )
{
    void* retVal;

    if( list->curr )
	retVal = (void *) &(list->curr->data);
    else
	retVal = NULL;

    return( retVal );
}

int
GListCount PARAMS( (list),
GLIST* list )
{
    return( list->count );
}

/*
 * joins 2 lists into one
 * does not check for compatible data sizes
 * always modifies the first list structure (l1)
 * so you can free the second list head (l2)
 */
GLIST*
GListConcat PARAMS( (l1,l2),
GLIST* l1 _AND_
GLIST* l2)
{
    if (l2->count == 0)
        return(l1);

    if (l1->count == 0)
    {
        l1->count = l2->count;
        l1->last = l2->last;
        l1->first = l2->first;
        l1->curr = l1->first;
    }
    else
    {
        l1->count += l2->count;
        l1->last->next = l2->first;
        l2->first->prev = l1->last;
        l1->last = l2->last;
    }

    return (l1);
}





/*
 * Returns the index (starting a 0 for the first elmt)
 * of the given elmt in the given list
 * returns -1 if the elmt is not in the list
 * Assumes that the list node contains a single pointer
 */
long int
GetGListElmtIndex PARAMS((elmt, list),
void* elmt _AND_
GLIST* list)
{
    void* tmp;
    void* tmpElmt;
    long int index;

    index = 0;
    tmp = (void*) CURR_LIST_NODE(list);
    FOR_EACH_LIST_ELMT(tmpElmt, list)
    {
        if (tmpElmt == elmt)
        {
            SET_CURR_LIST_NODE(list, tmp);
            return(index);
        }
        else
            index++;
    }

    SET_CURR_LIST_NODE(list, tmp);
    return(-1);
    
}  /* GetGListElmtIndex */
