/*
 * str_util.c  - bunch of ASN.1/C string utilities
 * 
 *
 * Mike Sample
 * 91/08/12
 * Copyright (C) 1991, 1992 Michael Sample
 *            and the University of British Columbia
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 */

#include <stdio.h>
#include <ctype.h>
#include <unistd.h>  /* for pathconf(..) */
#include <sys/param.h>
#include "snacc_config.h"
#include "basetypes.h"
#include "ber.h"
#include "list.h"
#include "asn1module.h"
#include "mem.h"
#include "define.h"
#include "rules.h"
#include "type_info.h"
#include "c_kwd.h"
#include "cpp_kwd.h"
#include "str_util.h"


#define DIGIT_TO_ASCII(d)  (((d) % 10) + '0')


/*
 * allocates new and returns a copy of the given
 * string with '-'s (dashes) replaced by  '_'s (underscores)
 */
char*
Asn1TypeName2CTypeName PARAMS((aName),
char* aName)
{
    char* retVal;
    if (aName == NULL)
        return (NULL);

    retVal = Malloc(strlen(aName) + 1);
    strcpy(retVal, aName);
    Dash2Underscore(retVal, strlen(retVal));

    return(retVal);
}  /* Asn1TypeName2CTypeName */


/*
 * allocates new str and returns a copy of the given
 * string with '-'s (dashes) replaced by  '_'s (underscores)
 */
char*
Asn1FieldName2CFieldName PARAMS((aName),
char* aName)
{
    char* retVal;
    if (aName == NULL)
        return (NULL);

    retVal = Malloc(strlen(aName) + 1);
    strcpy(retVal, aName);
    Dash2Underscore(retVal, strlen(retVal));

    return(retVal);
}  /* Asn1FieldName2CFieldName */


/*
 * allocates new str and returns a copy of the given
 * string with '-'s (dashes) replaced by  '_'s (underscores)
 */
char*
Asn1ValueName2CValueName PARAMS((aName),
char* aName)
{
    char* retVal;
    if (aName == NULL)
        return (NULL);

    retVal = Malloc(strlen(aName) + 1);
    strcpy(retVal, aName);
    Dash2Underscore(retVal, strlen(retVal));

    return(retVal);
}  /* Asn1FieldName2CFieldName */


/*
 * allocates and returns a string with all of 
 * the caps from the given string
 */
char*
GetCaps PARAMS((str),
char* str)
{
    int i, j;
    char* retVal;
    
    if (str == NULL)
        return (NULL);

    retVal = Malloc(strlen(str) + 1);

    for (j = 0, i = 0; i < strlen(str); i++)
    {
        if (isupper(str[i]))
            retVal[j++] = str[i];
    }

    retVal[j] = '\0';  /* null terminate */

    return(retVal);

}  /* GetCaps */


/*
 * allocates and returns a string with all of
 * the caps and digits from the given string
 */
char*
GetCapsAndDigits PARAMS((str),
char* str)
{
    int i, j;
    char* retVal;

    if (str == NULL)
        return (NULL);

    retVal = Malloc(strlen(str) + 1);

    for (j = 0, i = 0; i < strlen(str); i++)
    {
        if ( (isupper(str[i])) ||  (isdigit(str[i])))
            retVal[j++] = str[i];
    }

    retVal[j] = '\0';  /* null terminate */

    return(retVal);

}  /* GetCapsAndDigits */


/*
 * replaces lowercase chars in given str
 * with upper case version
 * NOTE: modifies given str
 */
void
Str2UCase PARAMS((str, len),
char* str _AND_
int len)
{
    int i;
    for (i=0; i < len; i++)
    {
        if (islower(str[i]))
            str[i] = toupper(str[i]);
    }
} /* Str2UCase */


/*
 * replaces uppercase chars in given str
 * with lower case version
 * NOTE: modifies given str
 */
void
Str2LCase PARAMS((str, len),
char* str _AND_
int len)
{
    int i;
    for (i=0; i < len; i++)
    {
        if (isupper(str[i]))
            str[i] = tolower(str[i]);
    }
} /* Str2LCase */


/*
 * replace dash chars in given str
 * with underscores
 * NOTE: modifies given str
 */
void
Dash2Underscore PARAMS((str, len),
char* str _AND_
int len)
{
    int i;
    for (i=0; i < len; i++)
    {
        if (str[i] == '-')
            str[i] = '_';
    }
} /* Dash2Underscore */


/*
 * tacks on the ascii version of the given digit
 * at the end of the given str. 
 * NOTE: make sure the str you give has enough space
 * for the digits
 */
void
AppendDigit PARAMS((str, digit),
char* str _AND_
int digit)
{ 
    int high = 1000000000;
    int currDigit;
    int value;
    char digitStr[20]; /* arbitrary length > max */
    
    if (digit < 0)
        digit *= -1;
    
    currDigit = 0;
    while ( high > 0)
    {
        value = digit / high;
        if (value != 0)
            digitStr[currDigit++]= DIGIT_TO_ASCII(value);

        digit = digit % high;
        high  = high/10;
    }

    if (currDigit == 0)
        strcat(str, "0");
    else
    {
        digitStr[currDigit] = '\0';  /* null terminate */
        strcat(str, digitStr);
    }
} /* AppendDigit */




/* 
 * given a defined object list containing null termintated strs,
 * a str to be made unique wrt to the list by adding digits to the
 * end, the max number of digits to add and the digit to start
 * at, str is modified to be unique.  It is not added to the 
 * defined object list.  The given str must have enough spare,
 * allocated chars after it's null terminator to hold maxDigits
 * more characters.
 * Only appends digits if the string is not unique or is a C keyword.
 *
 * Eg  MakeCStrUnique( { "Foo", "Bar" }, "Foo\0   ", 3 , 1)
 *         modifies the the Str "Foo" to "Foo1"
 */
void
MakeCStrUnique PARAMS((nameList, str, maxDigits, startingDigit),
DefinedObj* nameList _AND_
char* str _AND_
int maxDigits _AND_
int startingDigit)
{
    int digit, len, maxDigitVal;

    if (ObjIsDefined(nameList, str, StrObjCmp) || IsCKeyWord(str))
    {
        for (maxDigitVal = 1; maxDigits > 0 ; maxDigits--)
            maxDigitVal *= 10;
        
        len = strlen(str);
        digit = startingDigit;
        do
        {
            str[len] = '\0';
            AppendDigit(str, digit++);
        } while (ObjIsDefined(nameList, str, StrObjCmp) &&  (digit < maxDigitVal));
    }
}  /* MakeCStrUnique */


/*
 * same as MakeCStrUnique except checks against C++ keywords
 */
void
MakeCppStrUnique PARAMS((nameList, str, maxDigits, startingDigit),
DefinedObj* nameList _AND_
char* str _AND_
int maxDigits _AND_
int startingDigit)
{
    int digit, len, maxDigitVal;

    if (ObjIsDefined(nameList, str, StrObjCmp) || IsCppKeyWord(str))
    {
        for (maxDigitVal = 1; maxDigits > 0 ; maxDigits--)
            maxDigitVal *= 10;
        
        len = strlen(str);
        digit = startingDigit;
        do
        {
            str[len] = '\0';
            AppendDigit(str, digit++);
        } while (ObjIsDefined(nameList, str, StrObjCmp) &&  (digit < maxDigitVal));
    }
}  /* MakeCppStrUnique */


/*
 * allocates and returns a base file name generated from
 * the module's name.  May shorten the name if the
 * expected length exceed the systems max path component length
 * (eg to support SYS V 14 char filename len limit) 
 * Base file name is used a the base name for the generated
 * C source files.
 */
char*
MakeBaseFileName PARAMS((moduleName),
char* moduleName)
{
    int fNameLen;
    int cpyLen;
    char* retVal;
    int maxPathComponentLen;
    char pathName[1024];
#   define MAX_SUFFIX_LEN 2 /* .c, .h, .C */
    extern int maxFileNameLenG; /* declared in snacc.c */

    /*
     * if the user has not given the max file name len
     * via the -mf option, 
     * find the max filename len (ala POSIX method)
     * if possible.  Otherwise hardwire it to 14
     * to support underpowered OSes
     */
    if (maxFileNameLenG > 2)
        maxPathComponentLen = maxFileNameLenG;
    else
#ifdef _PC_NAME_MAX
        maxPathComponentLen = pathconf(getcwd(pathName, 1024), _PC_NAME_MAX);
#else
        maxPathComponentLen = 14;
#endif

    retVal = (char*) Malloc(strlen(moduleName) +1);
    fNameLen = strlen(moduleName) + MAX_SUFFIX_LEN;
    if ((fNameLen > maxPathComponentLen) && (maxPathComponentLen != -1))
    {
        cpyLen = maxPathComponentLen - MAX_SUFFIX_LEN;

        /* don't allow trailing dash */
        if (moduleName[cpyLen-1] == '-')
            cpyLen--;

        strncpy(retVal, moduleName, cpyLen);
        retVal[cpyLen] = '\0';
    }
    else
        strcpy(retVal, moduleName);

    return(retVal);

} /* MakeBaseFileName */




/*
 * given a module name and a suffix, the
 * suffix is appended to the module name
 * and the whole string is put into lower case
 * and underscores are inserted in likely places
 * (ie MTSAbstractSvc.h -> mts_abstract_svc.h)
 */
char*
MakeFileName PARAMS((moduleName, suffix),
char* moduleName _AND_
char* suffix)
{
    int i, cpyIndex, len;
    char* hdrCpy;
    int fNameLen;
    char* fName;
#define MAX_UNDERSCORE 10    

    fName = Malloc(strlen(moduleName) + strlen(suffix) + 1);
    strcpy(fName, moduleName);
    strcat(fName, suffix);


    fNameLen = strlen(fName);

    /*
     * convert dashes to underscores, add spaces
     */
    Dash2Underscore(fName, fNameLen);


    /*
     * remove the next two lines if you uncomment the 
     * following underscore inserter
     */
    Str2LCase(fName, fNameLen - strlen(suffix));    
    return(fName);
    
    /*
     *  NO LONGER DONE - LET THE USER MODIFY THE ASN.1 IF DESIRED
     *  add underscore between Lcase/Ucase of UCase/UcaseLcasce
     *  eg MTSAbstractSvc -> MTS_Abstract_Svc
     *  (if enough space)
    len = strlen(fName) + MAX_UNDERSCORE + 1;
    hdrCpy = (char*) Malloc(len);

    hdrCpy[0] = fName[0];
    for(i = 1, cpyIndex = 1; (cpyIndex < len) && (i < fNameLen); i++)
    {
        if ( ((islower(fName[i-1])) && (isupper(fName[i]))) ||
             ((isupper(fName[i-1])) && (isupper(fName[i])) &&
                            ( (i < (fNameLen-1)) && (islower(fName[i+1])))))
        {
            hdrCpy[cpyIndex++] = '_';
            hdrCpy[cpyIndex++] = fName[i];
        }
        else
            hdrCpy[cpyIndex++] = fName[i];
    }
    hdrCpy[cpyIndex++] = '\0';

    Str2LCase(hdrCpy, cpyIndex - strlen(suffix));

    Free(fName);
    return(hdrCpy);
    */


}  /* MakeFileName */


char*
MakeCHdrFileName PARAMS((moduleName),
char* moduleName)
{
    return(MakeFileName(moduleName, ".h"));
}

char*
MakeCSrcFileName PARAMS((moduleName),
char* moduleName)
{
    return(MakeFileName(moduleName, ".c"));
}

char*
MakeCppHdrFileName PARAMS((moduleName),
char* moduleName)
{
    return(MakeFileName(moduleName, ".h"));
}

char*
MakeCppSrcFileName PARAMS((moduleName),
char* moduleName)
{
    return(MakeFileName(moduleName, ".C"));
}





