/*
 * gen_c_code.c - generate C hdr and src files
 *
 * Assumes you have called FillCTypeInfo
 *
 * MS 92
 * Copyright (C) 1991, 1992 Michael Sample
 *            and the University of British Columbia
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/file.h>
#include <ctype.h>
#include "snacc_config.h"
#include "basetypes.h"
#include "ber.h"
#include "list.h"
#include "asn1module.h"
#include "mem.h"
#include "print.h"
#include "rules.h"
#include "type_info.h"
#include "util.h"
#include "gen_c_type.h"
#include "gen_c_enc.h"
#include "gen_c_dec.h"
#include "gen_c_vals.h"
#include "gen_c_free.h"
#include "gen_c_print.h"
#include "gen_c_any.h"
#include "gen_c_code.h"

/* unexported prototypes */
static void PrintCSrcComment PROTO((FILE* src, Module* m));
static void PrintCSrcIncludes PROTO((FILE* src, Module* m, ModuleList* mods));
static void PrintCHdrComment PROTO((FILE* hdr, Module* m));

/*
 * Fills the hdr file with the C type and encode/decode prototypes
 * Fills the src file with the encoded/decode routine definitions
 */
void
PrintCCode PARAMS((src, hdr, mods, m, r, longJmpVal, printTypes, printValues, printEncoders, printDecoders, printPrinters, printFree),
FILE* src _AND_
FILE* hdr _AND_
ModuleList* mods _AND_
Module* m _AND_
CRules* r _AND_
long int longJmpVal _AND_
int printTypes _AND_
int printValues _AND_
int printEncoders _AND_
int printDecoders _AND_
int printPrinters _AND_
int printFree)
{
    TypeDef* td;
    ValueDef* vd;

    PrintCSrcComment(src, m);
    PrintCSrcIncludes(src, m, mods);

    PrintCHdrComment(hdr, m);
    PrintConditionalIncludeOpen(hdr, m->cHdrFileName);

    fprintf(hdr,"\n\n");
    fprintf(src,"\n\n");


    if (printValues)
    {
        /* put value defs at beginning of .c file */
        FOR_EACH_LIST_ELMT(vd, m->valueDefs)
        {
            PrintCValueDef(src, r, vd);
        }
    }

    PrintCAnyCode(src, hdr, r, mods, m);

    FOR_EACH_LIST_ELMT(td, m->typeDefs)
    {
        if (printTypes)
            PrintCTypeDef(hdr, r, m, td);

        /* for PDU type or types ref'd with ANY/ANY DEF BY */
        if (printEncoders &&
            ((td->anyRefs != NULL) || td->cTypeDefInfo->isPdu))
            PrintCBerEncoder(src, hdr, r, m, td);

        /* for PDU type or types ref'd with ANY/ANY DEF BY */
        if (printDecoders &&
            ((td->anyRefs != NULL) || td->cTypeDefInfo->isPdu))
            PrintCBerDecoder(src, hdr, r, m, td, &longJmpVal);

        if (printEncoders)
            PrintCBerContentEncoder(src, hdr, r, m, td);

        if (printDecoders)
            PrintCBerContentDecoder(src, hdr, r, m, td, &longJmpVal);


        if (printPrinters)
            PrintCPrinter(src, hdr, r, mods, m, td);

        if (printFree)
            PrintCFree(src, hdr, r, mods, m, td);

        /* only print new lines for normal types */
        switch (td->type->basicType->choiceId)
        {
            case(BASICTYPE_SEQUENCEOF):  /* list types */
            case(BASICTYPE_SETOF):
            case(BASICTYPE_CHOICE):
            case(BASICTYPE_SET):
            case(BASICTYPE_SEQUENCE):
                fprintf(src, "\n\n\n");
                /* fall through */

            case(BASICTYPE_IMPORTTYPEREF):  /* type references */
            case(BASICTYPE_LOCALTYPEREF):
            case(BASICTYPE_BOOLEAN):  /* library type */
            case(BASICTYPE_REAL):  /* library type */
            case(BASICTYPE_OCTETSTRING):  /* library type */
            case(BASICTYPE_NULL):  /* library type */
            case(BASICTYPE_OID):  /* library type */
            case(BASICTYPE_INTEGER):  /* library type */
            case(BASICTYPE_BITSTRING):  /* library type */
            case(BASICTYPE_ENUMERATED):  /* library type */
            case(BASICTYPE_ANYDEFINEDBY):  /* ANY types */
            case(BASICTYPE_ANY):
                fprintf(hdr, "\n\n\n");
                break;
        } 

    }

    if (printValues)
    {
        /* put value externs at end of .h file */
        FOR_EACH_LIST_ELMT(vd, m->valueDefs)
        {
            PrintCValueExtern(hdr, r, vd);
        }
    }

    PrintConditionalIncludeClose(hdr, m->cHdrFileName);

} /* PrintCCode */


static void
PrintCSrcComment PARAMS((src, m),
FILE* src _AND_
Module* m)
{
    long int t;

    t = time(0);
    fprintf(src, "/*\n");
    fprintf(src, " *    %s\n *\n", m->cSrcFileName);
    fprintf(src, " *    \"%s\" ASN.1 module encode/decode/print/free C src.\n *\n", m->modId->name);
    fprintf(src, " *    This file was generated by snacc on %s *\n", ctime(&t));
    fprintf(src, " *    UBC snacc written by Mike Sample\n *\n");
    fprintf(src, " *    NOTE: This is a machine generated file - editing not recommended\n");
    fprintf(src, " */\n\n\n");

} /* PrintSrcComment */



static void
PrintCSrcIncludes PARAMS((src, m, mods),
FILE* src _AND_
Module* m _AND_
ModuleList* mods)
{
    void* tmp;
    Module* impMod;

    /*
     * include snacc runtime library related hdrs
     */
    fprintf(src, "\n#include \"asn_incl.h\"\n");

    /*
     * print out include files in same order of the module
     * list. every module in the list includes the others and it's
     * own .h
     */
    tmp = (void*)CURR_LIST_NODE(mods);
    FOR_EACH_LIST_ELMT(impMod, mods)
    {
        fprintf(src, "#include \"%s\"\n", impMod->cHdrFileName);
    }
    SET_CURR_LIST_NODE(mods, tmp);

}  /* PrintCSrcIncludes */


static void
PrintCHdrComment PARAMS((f, m),
FILE* f _AND_
Module* m)
{
    long int t;

    t = time(0);
    fprintf(f, "/*\n");
    fprintf(f, " *    %s\n *\n", m->cHdrFileName);
    fprintf(f, " *    \"%s\" ASN.1 module C type definitions and prototypes\n *\n", m->modId->name);
    fprintf(f, " *    This .h file was by snacc on %s *\n", ctime(&t));
    fprintf(f, " *    UBC snacc written compiler by Mike Sample\n *\n");
    fprintf(f, " *    NOTE: This is a machine generated file - editing not recommended\n");
    fprintf(f, " */\n\n\n");
} /* PrintCHdrComment */



void
PrintConditionalIncludeOpen PARAMS((f, fileName),
FILE* f _AND_
char* fileName)
{
    char hdrFileDefSym[256];
    int i;

    strcpy(hdrFileDefSym, fileName);
    for (i = 0; i < strlen(hdrFileDefSym); i++)
        if (hdrFileDefSym[i] == '.')
            hdrFileDefSym[i] = '_';

    fprintf(f, "#ifndef _%s_\n", hdrFileDefSym );
    fprintf(f, "#define _%s_\n\n\n", hdrFileDefSym);
} /* PrintConditionalIncludeOpen */


void
PrintConditionalIncludeClose PARAMS((f, fileName),
FILE* f _AND_
char* fileName)
{
    fprintf(f, "\n#endif /* conditional include of %s */\n", fileName);

} /* PrintConditionalIncludeClose */
