/*
 * type_info.c  - fills in c++ type information
 * 
 * MS 91/92
 * Copyright (C) 1991, 1992 Michael Sample 
 *           and the University of British Columbia
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 */


#include <stdio.h>
#include <ctype.h>
#include "snacc_config.h"
#include "basetypes.h"
#include "ber.h"
#include "list.h"
#include "define.h"
#include "asn1module.h"
#include "mem.h"
#include "snacc_util.h"
#include "str_util.h"
#include "cpp_rules.h"
#include "cpp_kwd.h"
#include "cpp_types.h"

extern Module* usefulTypeModG;
static DefinedObj* definedNamesG;

/* unexported prototypes */

void FillCppTypeDefInfo PROTO((CppRules* r,  Module* m,  TypeDef* td));

static void FillCppFieldNames PROTO((CppRules* r, NamedTypeList* firstSibling));

static void FillCppTypeRefInfo PROTO(( CppRules* r,  Module* m,  TypeDef* head,
                                      Type* parent, Type* t));

static void FillCppStructElmts PROTO((CppRules* r,  Module* m,  TypeDef* head,
                                      Type* parent, NamedTypeList* t));

static void FillCppChoiceElmts PROTO(( CppRules* r, Module* m, TypeDef* head,
                                      Type* parent, NamedTypeList* first));

static int IsCppPtr PROTO((CppRules* r, TypeDef* td, Type* parent, Type* t));

void FillCppTDIDefaults PROTO((CppRules* r, CppTDI* ctdi, TypeDef* td));


/*
 *  allocates and fills all the cppTypeInfos
 *  in the type trees for every module in the list
 */
void
FillCppTypeInfo PARAMS((r, modList), 
CppRules* r _AND_
ModuleList* modList)
{
    TypeDef* td;
    Module* m;

    /*
     * go through each module's type defs and fill
     * in the C type and enc/dec routines etc
     */
    definedNamesG = NULL;

    /* do useful types first */
    if (usefulTypeModG != NULL)
    {
        FOR_EACH_LIST_ELMT(td, usefulTypeModG->typeDefs)
            FillCppTypeDefInfo( r, usefulTypeModG, td);
    }

    FOR_EACH_LIST_ELMT(m, modList)
    {
        FOR_EACH_LIST_ELMT(td, m->typeDefs)
            FillCppTypeDefInfo( r, m, td);
    }

    /* 
     * now that type def info is filled in
     * set up set/seq/list/choice elements that ref
     * those definitions
     */

    /* do useful types first */
    if (usefulTypeModG != NULL)
    {
        FOR_EACH_LIST_ELMT(td, usefulTypeModG->typeDefs)
            FillCppTypeRefInfo( r, usefulTypeModG, td, NULL, td->type);
    }

    FOR_EACH_LIST_ELMT(m, modList)
    {
        FOR_EACH_LIST_ELMT(td, m->typeDefs)
            FillCppTypeRefInfo( r, m, td, NULL, td->type);
    }

    /*
     * modules compiled together (ie one call to snacc with
     * multiple args) likely to be C compiled together so
     * need a unique routines/types/defines/enum values
     * since assuming they share same name space.
     *  All Typedefs, union, struct & enum Tags, and defined values 
     * (enum consts), #define names
     *  are assumed to share the same name space
     */

    /* done with checking for name conflicts */
    FreeDefinedObjs(&definedNamesG);

}  /* FillCppTypeInfo */


/*
 *  allocates and fills structure holding C type definition information
 *  fo the given ASN.1 type definition.  Does not fill CTRI for contained
 *  types etc.
 */
void    
FillCppTypeDefInfo PARAMS((r, m, td),
CppRules* r _AND_
Module* m _AND_
TypeDef* td)
{
    int digit;
    int len;
    char* tmpName;
    CppTDI* cpptdi;
    
    /*
     * if CppTDI is present this type def has already been 'filled'
     */
    if (td->cppTypeDefInfo != NULL)
        return;

    
    cpptdi = MT(CppTDI);
    td->cppTypeDefInfo = cpptdi;

    /* get default type def attributes from table for type on rhs of ::= */

    FillCppTDIDefaults(r, cpptdi, td);


    /* 
     * if defined by a ref to another type definition fill in that type
     * def's CppTDI so can inherit (actully completly replace default
     * attributes) from it
     */
    if ((td->type->basicType->choiceId == BASICTYPE_LOCALTYPEREF) ||
        (td->type->basicType->choiceId == BASICTYPE_IMPORTTYPEREF))
    {
        /*
         * Fill in CppTDI for defining type if nec.
         * this works for importTypeRef as well since both a.localTypeRef
         * and a.importTypeRef are of type TypeRef
         */
        FillCppTypeDefInfo(r, td->type->basicType->a.localTypeRef->module,
                         td->type->basicType->a.localTypeRef->link);

        tmpName = cpptdi->className; /* save className */
        /* copy all type def info and restore name related stuff - hack*/
        *cpptdi = *td->type->basicType->a.localTypeRef->link->cppTypeDefInfo;
        cpptdi->className = tmpName; /* restore className */
    }


    /* 
     * check for any "--snacc" attributes that overide the current
     * cpptdi fields 
     * UNDEFINED FOR C++
    ParseTypeDefAttribs(cpptdi, td->attrList);
    */

}  /* FillCppTypeDefInfo */


static void
FillCppTypeRefInfo PARAMS((r, m, head, parent, t),
CppRules* r _AND_
Module* m _AND_
TypeDef* head _AND_
Type* parent _AND_
Type* t)
{
    CppTRI* cpptri;
    CppTDI* tmpCpptdi;
    ValueDef* namedElmt;
    CNamedElmt* cne;
    CNamedElmt** cneHndl;
    char*       elmtName;
    char*       listName;
    char*       choiceName;
    char*       unionName;
    Type* tmpT;
    int         len, digit;
    enum BasicTypeChoiceId basicTypeId;

    /*
     * you must check for cycles yourself before calling this
     */
    if ( t->cppTypeRefInfo == NULL)
    {
        cpptri = MT(CppTRI);
        t->cppTypeRefInfo = cpptri;
    }
    else
        cpptri =  t->cppTypeRefInfo;

    basicTypeId = t->basicType->choiceId;

    tmpCpptdi = &r->typeConvTbl[basicTypeId];

    /* get base type def info from the conversion table in the rules */
    cpptri->isEnc = tmpCpptdi->isEnc;
    cpptri->className = tmpCpptdi->className;
    cpptri->optTestRoutineName = tmpCpptdi->optTestRoutineName;

    
    /*
     * convert named elmts to C++ names.
     * check for name conflict with other defined Types/Names/Values
     */
    if (( (basicTypeId == BASICTYPE_INTEGER) || 
         (basicTypeId == BASICTYPE_ENUMERATED) || 
         (basicTypeId == BASICTYPE_BITSTRING)) &&
        !(LIST_EMPTY(t->basicType->a.integer)))
    {
        cpptri->namedElmts = GListNew(sizeof(void*));
        FOR_EACH_LIST_ELMT(namedElmt, t->basicType->a.integer)
        {
            cneHndl = (CNamedElmt**)GListAppend(cpptri->namedElmts);
            cne = *cneHndl = MT(CNamedElmt);
            elmtName = Asn1ValueName2CValueName(namedElmt->definedName);
            len = strlen(elmtName);
            cne->name = Malloc( len + 1 + r->maxDigitsToAppend);
            strcpy(cne->name, elmtName);
            Free(elmtName); /* not very efficient */

            if (namedElmt->value->basicValue->choiceId == BASICVALUE_INTEGER)
                cne->value = namedElmt->value->basicValue->a.integer;
            else
            {
                fprintf(stderr,"Warning: unlinked defined value. Using -9999999\n");
                cne->value = -9999999;
            }

            if (r->capitalizeNamedElmts)
                Str2UCase(cne->name, len);

            /*
             * append digits if enum value name is a keyword
             */
            MakeCppStrUnique(definedNamesG, cne->name, r->maxDigitsToAppend, 1);
           /*  not nec since each class hides the enum scope
             DefineObj(&definedNamesG, cne->name); */
        }
    }

    /* fill in rest of type info depending on the type */
    switch(basicTypeId)
    {
        case(BASICTYPE_BOOLEAN):  /* library types */
        case(BASICTYPE_INTEGER):
        case(BASICTYPE_BITSTRING):
        case(BASICTYPE_OCTETSTRING):
        case(BASICTYPE_NULL):
        case(BASICTYPE_OID):
        case(BASICTYPE_REAL):
        case(BASICTYPE_ENUMERATED):
            /* don't need to do anything else */
            break;


        case(BASICTYPE_SEQUENCEOF):  /* list types */
        case(BASICTYPE_SETOF):
            /* fill in component type */
            FillCppTypeRefInfo( r, m, head, t, t->basicType->a.setOf);
            break;

        case(BASICTYPE_IMPORTTYPEREF):  /* type references */
        case(BASICTYPE_LOCALTYPEREF):
            /*
             * grab class name from link (link is the def of the
             * the ref'd type )
             */
            if (t->basicType->a.localTypeRef->link != NULL)
            {
                /* inherit attributes from referenced type */
                tmpCpptdi=  t->basicType->a.localTypeRef->link->cppTypeDefInfo;
                cpptri->className = tmpCpptdi->className;
                cpptri->isEnc = tmpCpptdi->isEnc;
                cpptri->optTestRoutineName = tmpCpptdi->optTestRoutineName;
            }

            break;

        case(BASICTYPE_ANYDEFINEDBY):  /* ANY types */
            break;  /* these are handled now */

        case(BASICTYPE_ANY):
            PrintErrLoc( m->asn1SrcFileName, t->lineNo);
            fprintf(stderr,"Warning - generated code for the \"ANY\" type in type \"%s\" will need modification by YOU.", head->definedName);
            fprintf(stderr,"  The source files will have a \"/* ANY - Fix Me! */\" comment before related code.\n\n");

            break;

        case(BASICTYPE_CHOICE):
           /*
            * must fill field names BEFORE filling choice elmts
            * (allows better naming for choice ids)
            */
            FillCppFieldNames( r, t->basicType->a.choice);
            FillCppChoiceElmts( r, m, head, t, t->basicType->a.choice);
            break;

        case(BASICTYPE_SET):
        case(BASICTYPE_SEQUENCE):
            FillCppStructElmts(r, m, head, t, t->basicType->a.set);
            FillCppFieldNames( r, t->basicType->a.set);
            break;

        case(BASICTYPE_COMPONENTSOF):
        case(BASICTYPE_SELECTION):
            fprintf(stderr,"Compiler error - COMPONENTS OF or SELECTION type slipped through normalizing phase.\n");
            break;

        case(BASICTYPE_UNKNOWN):
        case(BASICTYPE_MACRODEF):
        case(BASICTYPE_MACROTYPE):
            /* do nothing */
            break;

    }

    /* 
     * figure out whether this is a ptr based on the enclosing
     * type (if any) and optionality/default
     */
    cpptri->isPtr = IsCppPtr(r, head, parent, t);

    /* let user overide any defaults with the --snacc attributes */
    /* undefined for C++ ParseTypeRefAttribs(ctri, t->attrList); */


}  /* FillCppTypeRefInfo */



static void
FillCppStructElmts PARAMS((r, m, head, parent, elmts),
CppRules* r _AND_
Module* m _AND_
TypeDef* head _AND_
Type* parent _AND_
NamedTypeList* elmts)
{
    NamedType* et;

    FOR_EACH_LIST_ELMT(et, elmts)
    {
        FillCppTypeRefInfo( r, m, head, parent, et->type);
    }

}  /* FillCppStructElmts */



/*
 *  Figures out non-conflicting enum names for the 
 *  choice id's
 */
static void
FillCppChoiceElmts PARAMS((r, m, head, parent, elmts),
CppRules* r _AND_
Module* m _AND_
TypeDef* head _AND_
Type* parent _AND_
NamedTypeList* elmts)
{
    NamedType* et;
    int idCount = 0;
    CppTRI* cpptri;
    int   len;

    /*
     * fill in type info for elmt types first
     */
    FOR_EACH_LIST_ELMT(et, elmts)
        FillCppTypeRefInfo( r, m, head, parent, et->type);

    /*
     * set choiceId Symbol & value
     * eg
     *  Car ::= CHOICE {          class Car {
     *     chev ChevCar,   ->         enum ChoiceIdEnum {
     *     ford FordCar                  chevCid, 
     *     toyota ToyotaCar              fordCid,
     *     }                             toyotaCid } choiceId;
     *                                union CarChoiceUnion {
     *                                      ChevCar* chev;
     *                                      FordCar* ford;
     *                                      ToyotaCar* toyota; };
     *                                  ...
     *                               }
     * NOTE that the union is anonymous
     */
    FOR_EACH_LIST_ELMT(et, elmts)
    {
        cpptri =  et->type->cppTypeRefInfo;
        
        if (cpptri == NULL)
            continue; /* wierd type */

        cpptri->choiceIdValue = idCount++;

        len = strlen(cpptri->fieldName);
        cpptri->choiceIdSymbol = Malloc(len + 4);
        strcpy(cpptri->choiceIdSymbol, cpptri->fieldName);
        strcat(cpptri->choiceIdSymbol, "Cid");
        
        if (r->capitalizeNamedElmts)
            Str2UCase(cpptri->choiceIdSymbol, len);

    }
            
}  /* FillCppChoiceElmts */


/* 
 * takes a list of "sibling" (eg same level in a structure)
 * ElmtTypes and fills sets up the c field names in 
 * the CppTRI struct 
 */
static void
FillCppFieldNames PARAMS((r, elmts),
CppRules* r _AND_
NamedTypeList* elmts)
{
    NamedType*  et;
    CppTRI* cpptri;
    DefinedObj* fieldNames;
    int        len, num, digit, i, tmpLen;
    char*      tmpName;
    char*      asn1FieldName;
    char*      cFieldName;

    /*
     * Initialize fieldname data
     * allocate (if nec) and fill in CTRI fieldname if poss 
     * from asn1 field name.  leave blank otherwise
     */
    fieldNames = NewObjList();
    FOR_EACH_LIST_ELMT(et, elmts)
    {
        cpptri =  et->type->cppTypeRefInfo;
        if (cpptri == NULL)
        {
            cpptri = MT(CppTRI);
            et->type->cppTypeRefInfo = cpptri;
        }
        if (et->fieldName != NULL)
        {
            asn1FieldName = et->fieldName;
            cpptri->fieldName = Asn1FieldName2CFieldName(asn1FieldName);

            MakeCppStrUnique(fieldNames, cpptri->fieldName, r->maxDigitsToAppend, 1);
            DefineObj(&fieldNames, cpptri->fieldName);
        }
    }

    
    FOR_EACH_LIST_ELMT(et, elmts)
    {
        cpptri =  et->type->cppTypeRefInfo;

        /*
         * generate field names for those without them
         */
        if (cpptri->fieldName == NULL)
        {
            if ( (et->type->basicType->choiceId == BASICTYPE_LOCALTYPEREF) ||
                 (et->type->basicType->choiceId == BASICTYPE_IMPORTTYPEREF) )
            {
                /*
                 * take ref'd type name as field name
                 * convert first let to lower case
                 */
                tmpName = et->type->basicType->a.localTypeRef->link->cppTypeDefInfo->className;
                tmpName =  Asn1TypeName2CTypeName(tmpName);
                cFieldName = Malloc(strlen(tmpName) + r->maxDigitsToAppend +1);
                strcpy(cFieldName, tmpName);
                Free(tmpName);
                if (isupper(cFieldName[0]))
                    cFieldName[0] = tolower(cFieldName[0]);
            }
            else
            {
                /*
                 * get default field name for this type
                 */
                tmpName = r->typeConvTbl[et->type->basicType->choiceId].defaultFieldName;
                cFieldName = Malloc(strlen(tmpName) + r->maxDigitsToAppend +1);
                strcpy(cFieldName, tmpName);

                if (isupper(cFieldName[0]))
                    cFieldName[0] = tolower(cFieldName[0]);
            }   
            

            len = strlen(cFieldName);

            /* 
             * try to use just the type name (with lower case first char).
             * if that is already used in this type or a C++ keyword,
             * append ascii digits to field name until unique
             * in this type
             */
            MakeCppStrUnique(fieldNames, cFieldName, r->maxDigitsToAppend, 1);
            DefineObj(&fieldNames, cFieldName);
            cpptri->fieldName = cFieldName;
        }
    }      
    FreeDefinedObjs(&fieldNames);
}  /* FillCppFieldNames */



/*
 * returns true if this c type for this type should be
 * be ref'd as a ptr
 */
static int
IsCppPtr PARAMS((r, td, parent, t),
CppRules* r _AND_
TypeDef* td _AND_
Type* parent _AND_
Type* t)
{
    CppTDI* cpptdi;
    int retVal = FALSE;

    /* 
     * inherit ptr attriubutes from ref'd type if any
     * otherwise grab lib c type def from the CppRules
     */
    if ((t->basicType->choiceId == BASICTYPE_LOCALTYPEREF) ||
        (t->basicType->choiceId == BASICTYPE_IMPORTTYPEREF))
    {
        cpptdi = t->basicType->a.localTypeRef->link->cppTypeDefInfo;
    }
    else
        cpptdi = &r->typeConvTbl[GetBuiltinType(t)];

    /* no parent means t is the root of a typedef */
    if ((parent == NULL) && (cpptdi->isPtrForTypeDef))
        retVal = TRUE;

    else if (( parent != NULL) &&
             (( parent->basicType->choiceId == BASICTYPE_SET) || 
             ( parent->basicType->choiceId == BASICTYPE_SEQUENCE)) &&
             ( cpptdi->isPtrInSetAndSeq))
        retVal = TRUE;

    else if (( parent != NULL) &&
             (( parent->basicType->choiceId == BASICTYPE_SETOF) || 
             ( parent->basicType->choiceId == BASICTYPE_SEQUENCEOF)) &&
             ( cpptdi->isPtrInList))
        retVal = TRUE;

    else if (( parent != NULL) &&
             ( parent->basicType->choiceId == BASICTYPE_CHOICE) &&
             ( cpptdi->isPtrInChoice))
        retVal = TRUE;

    else if (((t->optional) || (t->defaultVal != NULL)) && (cpptdi->isPtrForOpt))
        retVal = TRUE;

    return (retVal);
}  /* IsCppPtr */



/* fill given cpptdi with defaults from table for given typedef */
void
FillCppTDIDefaults PARAMS((r, cpptdi, td),
CppRules* r _AND_
CppTDI* cpptdi _AND_
TypeDef* td)
{
    CppTDI* tblCpptdi;
    int typeIndex;
    char* tmpName;

    typeIndex = GetBuiltinType(td->type);

    if (typeIndex < 0)
        return;

    tblCpptdi = &r->typeConvTbl[typeIndex];
    
    memcpy(cpptdi, tblCpptdi, sizeof(CppTDI));

    /* make sure class name is unique wrt to previously defined classes */
    tmpName = Asn1TypeName2CTypeName(td->definedName);
    cpptdi->className = Malloc(strlen(tmpName) + r->maxDigitsToAppend +1);    
    strcpy( cpptdi->className, tmpName);
    Free(tmpName);
    
    MakeCppStrUnique(definedNamesG, cpptdi->className, r->maxDigitsToAppend, 1);
    DefineObj(&definedNamesG, cpptdi->className);

} /* FillCppTDIDefaults */

