/*
 *  asn_oid.C   - OBJECT IDENTIFIER
 * 
 *  Mike Sample
 *  92/07/02
 * Copyright (C) 1992 Michael Sample and the University of British Columbia
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 */

#include "asn_config.h"
#include "asn_len.h"
#include "asn_tag.h"
#include "asn_type.h"
#include "asn_oid.h"


// Initializes an AsnOid with a string and it's length.
// The string should hold the encoded OID.
// The string is copied
void AsnOid::Set(const char* encOid, unsigned long int len)
{
    if (encOid != oid)
    {
        octetLen = len;
        oid = new char[octetLen];
        memcpy(oid, encOid, octetLen);
    }
}

// Inits an AsnOid from another OID.
// The oid string is copied.
void AsnOid::Set(const AsnOid& o)
{
    if (&o != this)
    {
        octetLen = o.octetLen;
        oid = new char[octetLen];
        memcpy(oid, o.oid, octetLen);
    }
}


// Given some arc numbers, an AsnOid is built.
// Set(1, 2, 3, 4, 5, -1, -1, -1 , -1 ,-1, -1) results in
// oid { 1 2 3 4 5 }.  The first negative arc number represnts
// the end of the arc numbers - at least 2 are required.
// The prototype in the AsnOid class provides default -1 parameters
// so you only need to provide the number of arc number in the oid
// as params. (eg Set(1,2,3,4,5))
void AsnOid::Set(unsigned long int a1, unsigned long int a2, 
               long int a3, long int a4, long int a5,
               long int a6, long int a7, long int a8,
               long int a9, long int a10, long int a11)
{
    static long int arcNumArr[11];
    static char buf[11*5];  /* make big enough for max oid with 11 arcs*/
    static char* tmpBuf;
    unsigned long int totalLen;
    unsigned long int elmtLen;
    long int tmpArcNum;
    long int headArcNum;

    tmpBuf = buf;

    arcNumArr[0] = a1;
    arcNumArr[1] = a2;
    arcNumArr[2] = a3;
    arcNumArr[3] = a4;
    arcNumArr[4] = a5;
    arcNumArr[5] = a6;
    arcNumArr[6] = a7;
    arcNumArr[7] = a8;
    arcNumArr[8] = a9;
    arcNumArr[9] = a10;
    arcNumArr[10] = a11;

    // munge together first oid arc numbers
    headArcNum = tmpArcNum = (arcNumArr[0] * 40) + arcNumArr[1];

    // figure encoded length for this arc number
    for(elmtLen = 1; (tmpArcNum >>= 7) != 0; elmtLen++);

    // write bytes except the last/least significant of the head arc number
    // more bit is on
    totalLen = elmtLen;
    for(int i = 1 ; i < elmtLen; i++)
    {
        *(tmpBuf++) = 0x80 | (headArcNum >> ((elmtLen-i)*7));
    }

    // write least significant (more bit is off)
    *(tmpBuf++) = 0x7f & headArcNum;

    // repeat for the rest of the arc numbers
    for (i = 2; (i < 11) && (arcNumArr[i] > 0); i++)
    {
        tmpArcNum = arcNumArr[i];
        for(elmtLen = 1; (tmpArcNum >>= 7) != 0; elmtLen++);
        totalLen += elmtLen;
        tmpArcNum = arcNumArr[i];
        for(int j = 1 ; j < elmtLen; j++)
        {
            *(tmpBuf++) = 0x80 | (tmpArcNum >> ((elmtLen-j)*7));
        }
        *(tmpBuf++) = 0x7f & tmpArcNum;
    }

    oid = Asn1Alloc(totalLen);
    memcpy(oid, buf, totalLen);
    octetLen = totalLen;

}  /* AsnOid::Set */



// Like Set except frees old oid value first
void AsnOid::ReSet(const char* encOid, unsigned long int len)
{
    if (encOid != oid)
    {
        delete oid;
        Set(encOid, len);
    }
}

// Like Set except frees old oid value first
void AsnOid::ReSet(const AsnOid& o)
{
    if (&o != this)
    {
        delete(oid);
        Set(o);
    }
}

void AsnOid::ReSet(unsigned long int a1, unsigned long int a2, 
               long int a3, long int a4, long int a5,
               long int a6, long int a7, long int a8,
               long int a9, long int a10, long int a11)
{
    delete (oid);
    Set(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, a11);
}

// Prints an AsnOid in ASN.1 Value Notation.
// Decodes the oid to get the individual arc numbers
void AsnOid::Print(ostream& os)
{
    unsigned short int firstArcNum;
    unsigned long int arcNum;
    int i;

    // print oid in 
    os << "{";

    // un-munge first two arc numbers
    for (arcNum = 0, i=0; (i < octetLen) && (oid[i] & 0x80);i++)
        arcNum = (arcNum << 7) + (oid[i] & 0x7f);

    arcNum = (arcNum << 7) + (oid[i] & 0x7f);
    i++;
    firstArcNum = arcNum/40;
    if (firstArcNum > 2)
        firstArcNum = 2;

    os << firstArcNum  << " " << arcNum - (firstArcNum * 40);

    for (; i < octetLen ; )
    {
        for (arcNum = 0; (i < octetLen) && (oid[i] & 0x80);i++)
            arcNum = (arcNum << 7) + (oid[i] & 0x7f);

        arcNum = (arcNum << 7) + (oid[i] & 0x7f);
        i++;
        os << " " << arcNum ;
    }

    os << "}";

}  /* AsnOid::Print */



// Decodes the content of a BER OBJECT IDENTIFIER value and puts
// the results in this AsnOid object.
void AsnOid::BDecContent( BUF_TYPE b, AsnTag tagId, AsnLen elmtLen,
                              AsnLen& bytesDecoded, ENV_TYPE env)
{
    /* treat like primitive octet string */
    octetLen = elmtLen;
    oid =  Asn1Alloc(elmtLen);
    b.Copy(oid, elmtLen);
    
    if (b.ReadError())
    {
        Asn1Error("BDecOctetString: ERROR - decoded past end of data\n");
        longjmp(env, -17);
    }
    bytesDecoded += elmtLen;
} /* AsnOid::BDecContent */



// returns the number of arc numbers in the OID value
unsigned long int AsnOid::NumArcs()
{
    int i;
    int numArcs;

    for (numArcs=0, i=0; i < octetLen ; )
    {
        // skip octets in this arc num with the 'more' bit set
        for (; (i < octetLen) && (oid[i] & 0x80);i++);

        // skip last octet in this arc num (no more bit)
        i++;

        numArcs++;
    }

    // add one to return value because the first two arcs are
    // crunched together into a single one.
    return(numArcs +1); 

}  /* AsnOid::NumArcs */

