/* 
 * asn_len.C - ASN.1 Length manipluation routines
 *
 * MS 92/06/18
 * Copyright (C) 1992 Michael Sample and the University of British Columbia
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 */

#include "asn_config.h"
#include "asn_len.h"


/*
 * Encodes the given length to the given buffer.
 * returns the number of octets written to the buffer.
 */
AsnLen
BEncDefLen ( BUF_TYPE  b, AsnLen len)
{
    /*
     * unrolled for efficiency
     * (check each possibitlity of the 4 byte integer)
     */
    if (len < 128)
    {
        b.PutByteRvs( len);
        return (1);
    }
    else if (len < 256)
    {
        b.PutByteRvs( len);
        b.PutByteRvs( 0x81);
        return(2);
    }
    else if (len < 65536) 
    {
        b.PutByteRvs( len);
        b.PutByteRvs( len >> 8);
        b.PutByteRvs( 0x82);
        return (3);
    }
    else if (len < 16777126)
    {
        b.PutByteRvs( len);
        b.PutByteRvs( len >> 8);
        b.PutByteRvs( len >> 16);
        b.PutByteRvs( 0x83);
        return (4);
    }
    else
    {
        b.PutByteRvs( len);
        b.PutByteRvs( len >> 8);
        b.PutByteRvs( len >> 16);
        b.PutByteRvs( len >> 24);
        b.PutByteRvs( 0x84);
        return (5);
    }
} /*  EncodeDefLen */

/*
 * Decode a BER length from the given buffer. Increments bytesDecoded
 * by the number of octets of the encoded length.  Flags an
 * error if the length is too large or a read error occurs
 */
AsnLen
BDecLen( BUF_TYPE b, AsnLen& bytesDecoded, ENV_TYPE env)
{
    AsnLen  len;
    unsigned char  byte;
    int  lenBytes;

    byte = b.GetByte();

    if (b.ReadError())
    {
        Asn1Error("BDecLen: decoded past end of data\n");
        longjmp(env, -9);
    }

    bytesDecoded++;
    if (byte < 128)   /* short length */
        return(byte);

    else if (byte == (unsigned char) 0x080)  /* indef len indicator */
        return(INDEFINITE_LEN);
  
    else  /* long len form */
    {
        /*
         * strip high bit to get # bytes left in len
         */
        lenBytes = byte & (unsigned char) 0x7f;

        if (lenBytes > sizeof(long int))
        {
            Asn1Error("BDecLen: ERROR - length overflow\n");
            longjmp(env, -10);
        }

        bytesDecoded += lenBytes;
               
        for (len = 0; lenBytes > 0; lenBytes--)
            len = (len << 8) | (unsigned long int) b.GetByte();


        if (b.ReadError())
        {
            Asn1Error("BDecLen: decoded past end of data\n");
            longjmp(env, -11);
        }
        
        return(len);
    }
    /* not reached */
}


/*
 * Encodes an End of Contents (EOC) to the given buffer.
 * Returns the encoded length.
 */
AsnLen
BEncEoc(BUF_TYPE b)
{

    b.PutByteRvs(0);
    b.PutByteRvs(0);
    return(2);
}  /* BEncEoc */

/*
 * Decodes an EOC from the given buffer.  flags an error if the
 * octets are non-zero or if read error occured.
 */
void
BDecEoc(BUF_TYPE b, AsnLen& bytesDecoded, ENV_TYPE env)
{
        
    if ( (b.GetByte() != 0) || (b.GetByte() != 0) || b.ReadError())
    {
        Asn1Error("BDecEoc: ERROR - non zero byte in EOC or end of data reached\n");
        longjmp(env, -12);
    }
    bytesDecoded += 2;
}  /* BDecEoc */

